"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("../../../core");
const constructs_1 = require("constructs");
const enums_1 = require("./enums");
const util_1 = require("./util");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends constructs_1.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new constructs_1.DependencyGroup();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancingv2_BaseTargetGroupProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, TargetGroupBase);
            }
            throw error;
        }
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.any({ produce: () => (0, util_1.renderAttributes)(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.string({ produce: () => this.targetType }),
            targets: cdk.Lazy.any({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.string({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.any({ produce: () => this.healthCheck?.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck?.interval?.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.string({ produce: () => this.healthCheck?.path }),
            healthCheckPort: cdk.Lazy.string({ produce: () => this.healthCheck?.port }),
            healthCheckProtocol: cdk.Lazy.string({ produce: () => this.healthCheck?.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck?.timeout?.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck?.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck?.unhealthyThresholdCount }),
            matcher: cdk.Lazy.any({
                produce: () => this.healthCheck?.healthyHttpCodes !== undefined || this.healthCheck?.healthyGrpcCodes !== undefined ? {
                    grpcCode: this.healthCheck.healthyGrpcCodes,
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
        this.node.addValidation({ validate: () => this.validateTargetGroup() });
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancingv2_HealthCheck(healthCheck);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureHealthCheck);
            }
            throw error;
        }
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_elasticloadbalancingv2_LoadBalancerTargetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLoadBalancerTarget);
            }
            throw error;
        }
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validateTargetGroup() {
        const ret = new Array();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-elasticloadbalancingv2-targetgroup.html#cfn-elasticloadbalancingv2-targetgroup-name
        const targetGroupName = this.resource.name;
        if (!cdk.Token.isUnresolved(targetGroupName) && targetGroupName !== undefined) {
            if (targetGroupName.length > 32) {
                ret.push(`Target group name: "${targetGroupName}" can have a maximum of 32 characters.`);
            }
            if (targetGroupName.startsWith('-') || targetGroupName.endsWith('-')) {
                ret.push(`Target group name: "${targetGroupName}" must not begin or end with a hyphen.`);
            }
            if (!/^[0-9a-z-]+$/i.test(targetGroupName)) {
                ret.push(`Target group name: "${targetGroupName}" must contain only alphanumeric characters or hyphens.`);
            }
        }
        return ret;
    }
}
_a = JSII_RTTI_SYMBOL_1;
TargetGroupBase[_a] = { fqn: "aws-cdk-lib.aws_elasticloadbalancingv2.TargetGroupBase", version: "2.74.0" };
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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