"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const util_1 = require("./util");
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @example
 *   /// fixture=withRepoAndSqsQueue
 *   // publish to an SQS queue every time code is committed
 *   // to a CodeCommit repository
 *   repository.onCommit('onCommit', { target: new targets.SqsQueue(queue) });
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sqs_IQueue(queue);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_targets_SqsQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SqsQueue);
            }
            throw error;
        }
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     */
    bind(rule, _id) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_IRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const restrictToSameAccount = core_1.FeatureFlags.of(rule).isEnabled(cxapi.EVENTS_TARGET_QUEUE_SAME_ACCOUNT);
        let conditions = {};
        if (!this.queue.encryptionMasterKey) {
            conditions = {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            };
        }
        else if (restrictToSameAccount) {
            // Add only the account id as a condition, to avoid circular dependency. See issue #11158.
            conditions = {
                StringEquals: { 'aws:SourceAccount': rule.env.account },
            };
        }
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', { conditions }));
        if (this.props.deadLetterQueue) {
            (0, util_1.addToDeadLetterQueueResourcePolicy)(rule, this.props.deadLetterQueue);
        }
        return {
            ...(0, util_1.bindBaseTargetConfig)(this.props),
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
SqsQueue[_a] = { fqn: "aws-cdk-lib.aws_events_targets.SqsQueue", version: "2.74.0" };
exports.SqsQueue = SqsQueue;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3FzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3FzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUNBLHFDQUFxQztBQUVyQyxxQ0FBMEM7QUFDMUMsc0NBQXNDO0FBQ3RDLGlDQUFtRztBQTBCbkc7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBYSxRQUFRO0lBRW5CLFlBQTRCLEtBQWlCLEVBQW1CLFFBQXVCLEVBQUU7UUFBN0QsVUFBSyxHQUFMLEtBQUssQ0FBWTtRQUFtQixVQUFLLEdBQUwsS0FBSyxDQUFvQjs7Ozs7OzsrQ0FGOUUsUUFBUTs7OztRQUdqQixJQUFJLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRTtZQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDM0U7S0FDRjtJQUVEOzs7OztPQUtHO0lBQ0ksSUFBSSxDQUFDLElBQWtCLEVBQUUsR0FBWTs7Ozs7Ozs7OztRQUMxQyxNQUFNLHFCQUFxQixHQUFHLG1CQUFZLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUV0RyxJQUFJLFVBQVUsR0FBUSxFQUFFLENBQUM7UUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUU7WUFDbkMsVUFBVSxHQUFHO2dCQUNYLFNBQVMsRUFBRSxFQUFFLGVBQWUsRUFBRSxJQUFJLENBQUMsT0FBTyxFQUFFO2FBQzdDLENBQUM7U0FDSDthQUFNLElBQUkscUJBQXFCLEVBQUU7WUFDaEMsMEZBQTBGO1lBQzFGLFVBQVUsR0FBRztnQkFDWCxZQUFZLEVBQUUsRUFBRSxtQkFBbUIsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRTthQUN4RCxDQUFDO1NBQ0g7UUFFRCw2QkFBNkI7UUFDN0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUUvRixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQzlCLElBQUEseUNBQWtDLEVBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDdEU7UUFFRCxPQUFPO1lBQ0wsR0FBRyxJQUFBLDJCQUFvQixFQUFDLElBQUksQ0FBQyxLQUFLLENBQUM7WUFDbkMsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUTtZQUN4QixLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPO1lBQ3pCLGNBQWMsRUFBRSxJQUFJLENBQUMsS0FBSztZQUMxQixhQUFhLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDckcsQ0FBQztLQUNIOzs7O0FBM0NVLDRCQUFRIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJy4uLy4uL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJy4uLy4uL2F3cy1zcXMnO1xuaW1wb3J0IHsgRmVhdHVyZUZsYWdzIH0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tICcuLi8uLi9jeC1hcGknO1xuaW1wb3J0IHsgYWRkVG9EZWFkTGV0dGVyUXVldWVSZXNvdXJjZVBvbGljeSwgVGFyZ2V0QmFzZVByb3BzLCBiaW5kQmFzZVRhcmdldENvbmZpZyB9IGZyb20gJy4vdXRpbCc7XG5cbi8qKlxuICogQ3VzdG9taXplIHRoZSBTUVMgUXVldWUgRXZlbnQgVGFyZ2V0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3FzUXVldWVQcm9wcyBleHRlbmRzIFRhcmdldEJhc2VQcm9wcyB7XG5cbiAgLyoqXG4gICAqIE1lc3NhZ2UgR3JvdXAgSUQgZm9yIG1lc3NhZ2VzIHNlbnQgdG8gdGhpcyBxdWV1ZVxuICAgKlxuICAgKiBSZXF1aXJlZCBmb3IgRklGTyBxdWV1ZXMsIGxlYXZlIGVtcHR5IGZvciByZWd1bGFyIHF1ZXVlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBtZXNzYWdlIGdyb3VwIElEIChyZWd1bGFyIHF1ZXVlKVxuICAgKi9cbiAgcmVhZG9ubHkgbWVzc2FnZUdyb3VwSWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBtZXNzYWdlIHRvIHNlbmQgdG8gdGhlIHF1ZXVlLlxuICAgKlxuICAgKiBNdXN0IGJlIGEgdmFsaWQgSlNPTiB0ZXh0IHBhc3NlZCB0byB0aGUgdGFyZ2V0IHF1ZXVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0aGUgZW50aXJlIEV2ZW50QnJpZGdlIGV2ZW50XG4gICAqL1xuICByZWFkb25seSBtZXNzYWdlPzogZXZlbnRzLlJ1bGVUYXJnZXRJbnB1dDtcbn1cblxuLyoqXG4gKiBVc2UgYW4gU1FTIFF1ZXVlIGFzIGEgdGFyZ2V0IGZvciBBbWF6b24gRXZlbnRCcmlkZ2UgcnVsZXMuXG4gKlxuICogQGV4YW1wbGVcbiAqICAgLy8vIGZpeHR1cmU9d2l0aFJlcG9BbmRTcXNRdWV1ZVxuICogICAvLyBwdWJsaXNoIHRvIGFuIFNRUyBxdWV1ZSBldmVyeSB0aW1lIGNvZGUgaXMgY29tbWl0dGVkXG4gKiAgIC8vIHRvIGEgQ29kZUNvbW1pdCByZXBvc2l0b3J5XG4gKiAgIHJlcG9zaXRvcnkub25Db21taXQoJ29uQ29tbWl0JywgeyB0YXJnZXQ6IG5ldyB0YXJnZXRzLlNxc1F1ZXVlKHF1ZXVlKSB9KTtcbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBTcXNRdWV1ZSBpbXBsZW1lbnRzIGV2ZW50cy5JUnVsZVRhcmdldCB7XG5cbiAgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHF1ZXVlOiBzcXMuSVF1ZXVlLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTcXNRdWV1ZVByb3BzID0ge30pIHtcbiAgICBpZiAocHJvcHMubWVzc2FnZUdyb3VwSWQgIT09IHVuZGVmaW5lZCAmJiAhcXVldWUuZmlmbykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdtZXNzYWdlR3JvdXBJZCBjYW5ub3QgYmUgc3BlY2lmaWVkIGZvciBub24tRklGTyBxdWV1ZXMnKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIFJ1bGVUYXJnZXQgdGhhdCBjYW4gYmUgdXNlZCB0byB0cmlnZ2VyIHRoaXMgU1FTIHF1ZXVlIGFzIGFcbiAgICogcmVzdWx0IGZyb20gYW4gRXZlbnRCcmlkZ2UgZXZlbnQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC91c2VyZ3VpZGUvcmVzb3VyY2UtYmFzZWQtcG9saWNpZXMtZXZlbnRicmlkZ2UuaHRtbCNzcXMtcGVybWlzc2lvbnNcbiAgICovXG4gIHB1YmxpYyBiaW5kKHJ1bGU6IGV2ZW50cy5JUnVsZSwgX2lkPzogc3RyaW5nKTogZXZlbnRzLlJ1bGVUYXJnZXRDb25maWcge1xuICAgIGNvbnN0IHJlc3RyaWN0VG9TYW1lQWNjb3VudCA9IEZlYXR1cmVGbGFncy5vZihydWxlKS5pc0VuYWJsZWQoY3hhcGkuRVZFTlRTX1RBUkdFVF9RVUVVRV9TQU1FX0FDQ09VTlQpO1xuXG4gICAgbGV0IGNvbmRpdGlvbnM6IGFueSA9IHt9O1xuICAgIGlmICghdGhpcy5xdWV1ZS5lbmNyeXB0aW9uTWFzdGVyS2V5KSB7XG4gICAgICBjb25kaXRpb25zID0ge1xuICAgICAgICBBcm5FcXVhbHM6IHsgJ2F3czpTb3VyY2VBcm4nOiBydWxlLnJ1bGVBcm4gfSxcbiAgICAgIH07XG4gICAgfSBlbHNlIGlmIChyZXN0cmljdFRvU2FtZUFjY291bnQpIHtcbiAgICAgIC8vIEFkZCBvbmx5IHRoZSBhY2NvdW50IGlkIGFzIGEgY29uZGl0aW9uLCB0byBhdm9pZCBjaXJjdWxhciBkZXBlbmRlbmN5LiBTZWUgaXNzdWUgIzExMTU4LlxuICAgICAgY29uZGl0aW9ucyA9IHtcbiAgICAgICAgU3RyaW5nRXF1YWxzOiB7ICdhd3M6U291cmNlQWNjb3VudCc6IHJ1bGUuZW52LmFjY291bnQgfSxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgLy8gZGVkdXBsaWNhdGVkIGF1dG9tYXRpY2FsbHlcbiAgICB0aGlzLnF1ZXVlLmdyYW50U2VuZE1lc3NhZ2VzKG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZXZlbnRzLmFtYXpvbmF3cy5jb20nLCB7IGNvbmRpdGlvbnMgfSkpO1xuXG4gICAgaWYgKHRoaXMucHJvcHMuZGVhZExldHRlclF1ZXVlKSB7XG4gICAgICBhZGRUb0RlYWRMZXR0ZXJRdWV1ZVJlc291cmNlUG9saWN5KHJ1bGUsIHRoaXMucHJvcHMuZGVhZExldHRlclF1ZXVlKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgLi4uYmluZEJhc2VUYXJnZXRDb25maWcodGhpcy5wcm9wcyksXG4gICAgICBhcm46IHRoaXMucXVldWUucXVldWVBcm4sXG4gICAgICBpbnB1dDogdGhpcy5wcm9wcy5tZXNzYWdlLFxuICAgICAgdGFyZ2V0UmVzb3VyY2U6IHRoaXMucXVldWUsXG4gICAgICBzcXNQYXJhbWV0ZXJzOiB0aGlzLnByb3BzLm1lc3NhZ2VHcm91cElkID8geyBtZXNzYWdlR3JvdXBJZDogdGhpcy5wcm9wcy5tZXNzYWdlR3JvdXBJZCB9IDogdW5kZWZpbmVkLFxuICAgIH07XG4gIH1cblxufVxuIl19