"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addToDeadLetterQueueResourcePolicy = exports.addLambdaPermission = exports.singletonEventRole = exports.bindBaseTargetConfig = void 0;
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const constructs_1 = require("constructs");
/**
 * Bind props to base rule target config.
 * @internal
 */
function bindBaseTargetConfig(props) {
    let { deadLetterQueue, retryAttempts, maxEventAge } = props;
    return {
        deadLetterConfig: deadLetterQueue ? { arn: deadLetterQueue?.queueArn } : undefined,
        retryPolicy: (retryAttempts !== undefined && retryAttempts >= 0) || maxEventAge
            ? {
                maximumRetryAttempts: retryAttempts,
                maximumEventAgeInSeconds: maxEventAge?.toSeconds({ integral: true }),
            }
            : undefined,
    };
}
exports.bindBaseTargetConfig = bindBaseTargetConfig;
/**
 * Obtain the Role for the EventBridge event
 *
 * If a role already exists, it will be returned. This ensures that if multiple
 * events have the same target, they will share a role.
 * @internal
 */
function singletonEventRole(scope) {
    const id = 'EventsRole';
    const existing = scope.node.tryFindChild(id);
    if (existing) {
        return existing;
    }
    const role = new iam.Role(scope, id, {
        roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
    });
    return role;
}
exports.singletonEventRole = singletonEventRole;
/**
 * Allows a Lambda function to be called from a rule
 * @internal
 */
function addLambdaPermission(rule, handler) {
    let scope;
    let node = handler.permissionsNode;
    let permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
    if (rule instanceof constructs_1.Construct) {
        // Place the Permission resource in the same stack as Rule rather than the Function
        // This is to reduce circular dependency when the lambda handler and the rule are across stacks.
        scope = rule;
        node = rule.node;
        permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(handler.node)}`;
    }
    if (!node.tryFindChild(permissionId)) {
        handler.addPermission(permissionId, {
            scope,
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('events.amazonaws.com'),
            sourceArn: rule.ruleArn,
        });
    }
}
exports.addLambdaPermission = addLambdaPermission;
/**
 * Allow a rule to send events with failed invocation to an Amazon SQS queue.
 * @internal
 */
function addToDeadLetterQueueResourcePolicy(rule, queue) {
    if (!sameEnvDimension(rule.env.region, queue.env.region)) {
        throw new Error(`Cannot assign Dead Letter Queue in region ${queue.env.region} to the rule ${core_1.Names.nodeUniqueId(rule.node)} in region ${rule.env.region}. Both the queue and the rule must be in the same region.`);
    }
    // Skip Resource Policy creation if the Queue is not in the same account.
    // There is no way to add a target onto an imported rule, so we can assume we will run the following code only
    // in the account where the rule is created.
    if (sameEnvDimension(rule.env.account, queue.env.account)) {
        const policyStatementId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
        queue.addToResourcePolicy(new iam.PolicyStatement({
            sid: policyStatementId,
            principals: [new iam.ServicePrincipal('events.amazonaws.com')],
            effect: iam.Effect.ALLOW,
            actions: ['sqs:SendMessage'],
            resources: [queue.queueArn],
            conditions: {
                ArnEquals: {
                    'aws:SourceArn': rule.ruleArn,
                },
            },
        }));
    }
    else {
        core_1.Annotations.of(rule).addWarning(`Cannot add a resource policy to your dead letter queue associated with rule ${rule.ruleName} because the queue is in a different account. You must add the resource policy manually to the dead letter queue in account ${queue.env.account}.`);
    }
}
exports.addToDeadLetterQueueResourcePolicy = addToDeadLetterQueueResourcePolicy;
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 * @internal
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
//# sourceMappingURL=data:application/json;base64,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