"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * An event pattern matcher
 */
class Match {
    /**
     * Matches a null value in the JSON of the event
     */
    static nullValue() {
        return this.fromObjects([null]);
    }
    /**
     * Matches when the field is present in the JSON of the event
     */
    static exists() {
        return this.fromObjects([{ exists: true }]);
    }
    /**
     * Matches when the field is absent from the JSON of the event
     */
    static doesNotExist() {
        return this.fromObjects([{ exists: false }]);
    }
    /**
     * Matches a string, exactly, in the JSON of the event
     */
    static exactString(value) {
        return this.fromObjects([value]);
    }
    /**
     * Matches a string, regardless of case, in the JSON of the event
     */
    static equalsIgnoreCase(value) {
        return this.fromObjects([{ 'equals-ignore-case': value }]);
    }
    /**
     * Matches strings with the given prefix in the JSON of the event
     */
    static prefix(value) {
        return this.fromObjects([{ prefix: value }]);
    }
    /**
     * Matches strings with the given suffix in the JSON of the event
     */
    static suffix(value) {
        return this.fromObjects([{ suffix: value }]);
    }
    /**
     * Matches IPv4 and IPv6 network addresses using the Classless Inter-Domain Routing (CIDR) format
     */
    static cidr(range) {
        const ipv4Regex = /^([0-9]{1,3}\.){3}[0-9]{1,3}(\/([0-9]|[1-2][0-9]|3[0-2]))?$/igm;
        const ipv6Regex = /^s*((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:)))(%.+)?s*(\/([0-9]|[1-9][0-9]|1[0-1][0-9]|12[0-8]))?$/igm;
        if (!ipv4Regex.test(range) && !ipv6Regex.test(range)) {
            throw new Error(`Invalid IP address range: ${range}`);
        }
        return this.fromObjects([{ cidr: range }]);
    }
    /**
     * Matches IPv4 and IPv6 network addresses using the Classless Inter-Domain Routing (CIDR) format.
     * Alias of `cidr()`.
     */
    static ipAddressRange(range) {
        return Match.cidr(range);
    }
    /**
     * Matches anything except what's provided in the rule. The list of provided values must contain
     * only strings or only numbers.
     */
    static anythingBut(...values) {
        if (values.length === 0) {
            throw new Error('anythingBut matchers must be non-empty lists');
        }
        const allNumbers = values.every(v => typeof (v) === 'number');
        const allStrings = values.every(v => typeof (v) === 'string');
        if (!(allNumbers || allStrings)) {
            throw new Error('anythingBut matchers must be lists that contain only strings or only numbers.');
        }
        return this.fromObjects([{ 'anything-but': values }]);
    }
    /**
     * Matches any string that doesn't start with the given prefix.
     */
    static anythingButPrefix(prefix) {
        return this.fromObjects([{ 'anything-but': { prefix: prefix } }]);
    }
    /**
     * Matches numbers greater than the provided value
     */
    static greaterThan(value) {
        return this.numeric('>', value);
    }
    /**
     * Matches numbers greater than, or equal to, the provided value
     */
    static greaterThanOrEqual(value) {
        return this.numeric('>=', value);
    }
    /**
     * Matches numbers less than the provided value
     */
    static lessThan(value) {
        return this.numeric('<', value);
    }
    /**
     * Matches numbers less than, or equal to, the provided value
     */
    static lessThanOrEqual(value) {
        return this.numeric('<=', value);
    }
    /**
     * Matches numbers equal to the provided value
     */
    static equal(value) {
        return this.numeric('=', value);
    }
    /**
     * Matches numbers inside a closed numeric interval. Equivalent to:
     *
     *    Match.allOf(Match.greaterThanOrEqual(lower), Match.lessThanOrEqual(upper))
     *
     * @param lower Lower bound (inclusive)
     * @param upper Upper bound (inclusive)
     */
    static interval(lower, upper) {
        if (lower > upper) {
            throw new Error(`Invalid interval: [${lower}, ${upper}]`);
        }
        return Match.allOf(Match.greaterThanOrEqual(lower), Match.lessThanOrEqual(upper));
    }
    /**
     * Matches an event if any of the provided matchers do. Only numeric matchers are accepted.
     */
    static allOf(...matchers) {
        if (matchers.length === 0) {
            throw new Error('A list of matchers must contain at least one element.');
        }
        return this.fromMergedObjects(matchers);
    }
    /**
     * Matches an event if any of the provided matchers does.
     */
    static anyOf(...matchers) {
        if (matchers.length === 0) {
            throw new Error('A list of matchers must contain at least one element.');
        }
        return this.fromObjects(matchers);
    }
    static numeric(operator, value) {
        return this.fromObjects([{ numeric: [operator, value] }]);
    }
    static fromObjects(values) {
        return new Match(values, { mergeMatchers: false }).asList();
    }
    static fromMergedObjects(values) {
        return new Match(values, { mergeMatchers: true }).asList();
    }
    constructor(matchers, options) {
        this.matchers = matchers;
        this.options = options;
        this.creationStack = (0, core_1.captureStackTrace)();
    }
    resolve(context) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_IResolveContext(context);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resolve);
            }
            throw error;
        }
        const matchers = this.matchers.flatMap(matcher => context.resolve(matcher));
        return this.options.mergeMatchers ? this.merge(matchers) : matchers;
    }
    merge(matchers) {
        // This is the only supported case for merging at the moment.
        // We can generalize this logic if EventBridge starts supporting more cases in the future.
        if (!matchers.every(matcher => matcher?.numeric)) {
            throw new Error('Only numeric matchers can be merged into a single matcher.');
        }
        return [{ numeric: matchers.flatMap(matcher => matcher.numeric) }];
    }
    toString() {
        return core_1.Token.asString(this);
    }
    /**
     * A representation of this matcher as a list of strings
     */
    asList() {
        return core_1.Token.asList(this);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "aws-cdk-lib.aws_events.Match", version: "2.74.0" };
exports.Match = Match;
//# sourceMappingURL=data:application/json;base64,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