"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * Import an existing EventBridge Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).splitArn(eventRuleArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(determineRuleScope(scope, props), id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        /** Set to keep track of what target accounts and regions we've already created event buses for */
        this._xEnvTargetsAdded = new Set();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_RuleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Rule);
            }
            throw error;
        }
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule?.expressionString;
        // add a warning on synth when minute is not defined in a cron schedule
        props.schedule?._bind(this);
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
        this.node.addValidation({ validate: () => this.validateRule() });
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_IRuleTarget(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTarget);
            }
            throw error;
        }
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = targetProps.role?.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetProps.targetResource.env?.account || targetStack.account;
            const targetRegion = targetProps.targetResource.env?.region || targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            // if the target is in a different account or region and is defined in this CDK App
            // we can generate all the needed components:
            // - forwarding rule in the source stack (target: default event bus of the receiver region)
            // - eventbus permissions policy (creating an extra stack)
            // - receiver rule in the target stack (target: the actual target)
            if (!this.sameEnvDimension(sourceAccount, targetAccount) || !this.sameEnvDimension(sourceRegion, targetRegion)) {
                // cross-account and/or cross-region event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cross-account.html
                // for cross-account or cross-region events, we require a concrete target account and region
                if (!targetAccount || core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account or cross-region events');
                }
                if (!targetRegion || core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account or cross-region events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account or cross-region events');
                }
                // Don't exactly understand why this code was here (seems unlikely this rule would be violated), but
                // let's leave it in nonetheless.
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account or cross-region targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account or cross-region event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                // The target of this Rule will be the default event bus of the target environment
                this.ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id);
                // The actual rule lives in the target stack. Other than the account, it's identical to this one,
                // but only evaluated at render time (via a special subclass).
                //
                // FIXME: the MirrorRule is a bit silly, forwarding the exact same event to another event bus
                // and trigger on it there (there will be issues with construct references, for example). Especially
                // in the case of scheduled events, we will just trigger both rules in parallel in both environments.
                //
                // A better solution would be to have the source rule add a unique token to the the event,
                // and have the mirror rule trigger on that token only (thereby properly separating triggering, which
                // happens in the source env; and activating, which happens in the target env).
                //
                // Don't have time to do that right now.
                const mirrorRuleScope = this.obtainMirrorRuleScope(targetStack, targetAccount, targetRegion);
                new MirrorRule(mirrorRuleScope, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                }, this);
                return;
            }
        }
        // Here only if the target does not have a targetResource defined.
        // In such case we don't have to generate any extra component.
        // Note that this can also be an imported resource (i.e: EventBus target)
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            httpParameters: targetProps.httpParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: inputProps?.inputTemplate !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_EventPattern(eventPattern);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventPattern);
            }
            throw error;
        }
        if (!eventPattern) {
            return;
        }
        (0, util_1.mergeEventPattern)(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        return (0, util_1.renderEventPattern)(this.eventPattern);
    }
    validateRule() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
    /**
     * Make sure we add the target environments event bus as a target, and the target has permissions set up to receive our events
     *
     * For cross-account rules, uses a support stack to set up a policy on the target event bus.
     */
    ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id) {
        // the _actual_ target is just the event bus of the target's account
        // make sure we only add it once per account per region
        const key = `${targetAccount}:${targetRegion}`;
        if (this._xEnvTargetsAdded.has(key)) {
            return;
        }
        this._xEnvTargetsAdded.add(key);
        const eventBusArn = targetStack.formatArn({
            service: 'events',
            resource: 'event-bus',
            resourceName: 'default',
            region: targetRegion,
            account: targetAccount,
        });
        // For some reason, cross-region requires a Role (with `PutEvents` on the
        // target event bus) while cross-account doesn't
        const roleArn = !this.sameEnvDimension(targetRegion, core_1.Stack.of(this).region)
            ? this.crossRegionPutEventsRole(eventBusArn).roleArn
            : undefined;
        this.targets.push({
            id,
            arn: eventBusArn,
            roleArn,
        });
        // Add a policy to the target Event Bus to allow the source account/region to publish into it.
        //
        // Since this Event Bus permission needs to be deployed before the stack containing the Rule is deployed
        // (as EventBridge verifies whether you have permissions to the targets on rule creation), this needs
        // to be in a support stack.
        const sourceApp = this.node.root;
        const sourceAccount = core_1.Stack.of(this).account;
        // If different accounts, we need to add the permissions to the target eventbus
        //
        // For different region, no need for a policy on the target event bus (but a need
        // for a role).
        if (!this.sameEnvDimension(sourceAccount, targetAccount)) {
            const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
            let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
            if (!eventBusPolicyStack) {
                eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                    env: {
                        account: targetAccount,
                        region: targetRegion,
                    },
                    // The region in the stack name is rather redundant (it will always be the target region)
                    // Leaving it in for backwards compatibility.
                    stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                });
                const statementPrefix = `Allow-account-${sourceAccount}-`;
                new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                    action: 'events:PutEvents',
                    statementId: statementPrefix + core_1.Names.uniqueResourceName(this, {
                        maxLength: 64 - statementPrefix.length,
                    }),
                    principal: sourceAccount,
                });
            }
            // deploy the event bus permissions before the source stack
            core_1.Stack.of(this).addDependency(eventBusPolicyStack);
        }
    }
    /**
     * Return the scope where the mirror rule should be created for x-env event targets
     *
     * This is the target resource's containing stack if it shares the same region (owned
     * resources), or should be a fresh support stack for imported resources.
     *
     * We don't implement the second yet, as I have to think long and hard on whether we
     * can reuse the existing support stack or not, and I don't have time for that right now.
     */
    obtainMirrorRuleScope(targetStack, targetAccount, targetRegion) {
        // for cross-account or cross-region events, we cannot create new components for an imported resource
        // because we don't have the target stack
        if (this.sameEnvDimension(targetStack.account, targetAccount) && this.sameEnvDimension(targetStack.region, targetRegion)) {
            return targetStack;
        }
        // For now, we don't do the work for the support stack yet
        throw new Error('Cannot create a cross-account or cross-region rule for an imported resource (create a stack with the right environment for the imported resource)');
    }
    /**
     * Obtain the Role for the EventBridge event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     * @internal
     */
    crossRegionPutEventsRole(eventBusArn) {
        const id = 'EventsRole';
        let role = this.node.tryFindChild(id);
        if (!role) {
            role = new aws_iam_1.Role(this, id, {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new aws_iam_1.ServicePrincipal('events.amazonaws.com'),
            });
        }
        role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [eventBusArn],
        }));
        return role;
    }
    /**
     * Whether two string probably contain the same environment dimension (region or account)
     *
     * Used to compare either accounts or regions, and also returns true if one or both
     * are unresolved (in which case both are expected to be "current region" or "current account").
     */
    sameEnvDimension(dim1, dim2) {
        switch (core_1.Token.compareStrings(dim1, dim2)) {
            case core_1.TokenComparison.ONE_UNRESOLVED:
                core_1.Annotations.of(this).addWarning('Either the Event Rule or target has an unresolved environment. \n \
          If they are being used in a cross-environment setup you need to specify the environment for both.');
                return true;
            case core_1.TokenComparison.BOTH_UNRESOLVED:
            case core_1.TokenComparison.SAME:
                return true;
            default:
                return false;
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "aws-cdk-lib.aws_events.Rule", version: "2.74.0" };
exports.Rule = Rule;
function determineRuleScope(scope, props) {
    if (!props.crossStackScope) {
        return scope;
    }
    const scopeStack = core_1.Stack.of(scope);
    const targetStack = core_1.Stack.of(props.crossStackScope);
    if (scopeStack === targetStack) {
        return scope;
    }
    // cross-region/account Events require their own setup,
    // so we use the base scope in that case
    const regionComparison = core_1.Token.compareStrings(scopeStack.region, targetStack.region);
    const accountComparison = core_1.Token.compareStrings(scopeStack.account, targetStack.account);
    const stacksInSameAccountAndRegion = (regionComparison === core_1.TokenComparison.SAME || regionComparison === core_1.TokenComparison.BOTH_UNRESOLVED) &&
        (accountComparison === core_1.TokenComparison.SAME || accountComparison === core_1.TokenComparison.BOTH_UNRESOLVED);
    return stacksInSameAccountAndRegion ? props.crossStackScope : scope;
}
/**
 * A rule that mirrors another rule
 */
class MirrorRule extends Rule {
    constructor(scope, id, props, source) {
        super(scope, id, props);
        this.source = source;
    }
    _renderEventPattern() {
        return this.source._renderEventPattern();
    }
    /**
     * Override validateRule to be a no-op
     *
     * The rules are never stored on this object so there's nothing to validate.
     *
     * Instead, we mirror the other rule at render time.
     */
    validateRule() {
        return [];
    }
}
//# sourceMappingURL=data:application/json;base64,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