"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDataCompressionType = exports.LustreAutoImportPolicy = exports.LustreDeploymentType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("../../aws-ec2");
const core_1 = require("../../core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
    /**
     * Newer type of long term storage with higher throughput tiers.
     * Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_2"] = "PERSISTENT_2";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The different auto import policies which are allowed
 */
var LustreAutoImportPolicy;
(function (LustreAutoImportPolicy) {
    /**
     * AutoImport is off. Amazon FSx only updates file and directory listings from the linked S3 bucket when the file system is created. FSx does not update file and directory listings for any new or changed objects after choosing this option.
     */
    LustreAutoImportPolicy["NONE"] = "NONE";
    /**
     * AutoImport is on. Amazon FSx automatically imports directory listings of any new objects added to the linked S3 bucket that do not currently exist in the FSx file system.
     */
    LustreAutoImportPolicy["NEW"] = "NEW";
    /**
     * AutoImport is on. Amazon FSx automatically imports file and directory listings of any new objects added to the S3 bucket and any existing objects that are changed in the S3 bucket after you choose this option.
     */
    LustreAutoImportPolicy["NEW_CHANGED"] = "NEW_CHANGED";
    /**
     * AutoImport is on. Amazon FSx automatically imports file and directory listings of any new objects added to the S3 bucket, any existing objects that are changed in the S3 bucket, and any objects that were deleted in the S3 bucket.
     * */
    LustreAutoImportPolicy["NEW_CHANGED_DELETED"] = "NEW_CHANGED_DELETED";
})(LustreAutoImportPolicy = exports.LustreAutoImportPolicy || (exports.LustreAutoImportPolicy = {}));
/**
  * The permitted Lustre data compression algorithms
*/
var LustreDataCompressionType;
(function (LustreDataCompressionType) {
    /**
    *
    * `NONE` - (Default) Data compression is turned off when the file system is created.
    */
    LustreDataCompressionType["NONE"] = "NONE";
    /**
    * `LZ4` - Data compression is turned on with the LZ4 algorithm.  Note: When you turn data compression on for an existing file system, only newly written files are compressed. Existing files are not compressed.
    */
    LustreDataCompressionType["LZ4"] = "LZ4";
})(LustreDataCompressionType = exports.LustreDataCompressionType || (exports.LustreDataCompressionType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_fsx_FileSystemAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLustreFileSystemAttributes);
            }
            throw error;
        }
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_fsx_LustreFileSystemProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LustreFileSystem);
            }
            throw error;
        }
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: props.lustreConfiguration.weeklyMaintenanceStartTime?.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.env.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validateAutoImportPolicy(deploymentType, lustreConfiguration.importPath, lustreConfiguration.autoImportPolicy);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the auto import policy
     */
    validateAutoImportPolicy(deploymentType, importPath, autoImportPolicy) {
        if (autoImportPolicy === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('autoImportPolicy requires importPath to be defined');
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_2) {
            throw new Error('autoImportPolicy is not supported with PERSISTENT_2 deployments');
        }
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1 && deploymentType !== LustreDeploymentType.PERSISTENT_2) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1/PERSISTENT_2 deployment types, received: ' + deploymentType);
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_1) {
            if (![50, 100, 200].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB for PERSISTENT_1 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_2) {
            if (![125, 250, 500, 1000].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 125, 250, 500 or 1000 MB/s/TiB for PERSISTENT_2 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
LustreFileSystem[_a] = { fqn: "aws-cdk-lib.aws_fsx.LustreFileSystem", version: "2.74.0" };
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
exports.LustreFileSystem = LustreFileSystem;
//# sourceMappingURL=data:application/json;base64,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