"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroupPeer = void 0;
const ec2 = require("../../aws-ec2");
const custom_resources_1 = require("../../custom-resources");
/**
 * The security group used by a Global Accelerator to send traffic to resources in a VPC.
 */
class AcceleratorSecurityGroupPeer {
    /**
     * Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            // APIs are available in 2.1055.0
            installLatestAwsSdk: false,
        });
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup.node.defaultChild);
        // Look up the security group ID
        return new AcceleratorSecurityGroupPeer(lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
    }
    constructor(securityGroupId) {
        this.securityGroupId = securityGroupId;
        this.canInlineRule = false;
        this.connections = new ec2.Connections({ peer: this });
        this.uniqueId = 'GlobalAcceleratorGroup';
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
exports.AcceleratorSecurityGroupPeer = AcceleratorSecurityGroupPeer;
//# sourceMappingURL=data:application/json;base64,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