"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedRole = void 0;
const core_1 = require("../../../core");
const cx_api_1 = require("../../../cx-api");
const grant_1 = require("../grant");
const policy_1 = require("../policy");
const principals_1 = require("../principals");
const util_1 = require("../util");
class ImportedRole extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            account: props.account,
        });
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.attachedPolicies = new util_1.AttachedPolicies();
        this.roleArn = props.roleArn;
        this.roleName = props.roleName;
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        this.defaultPolicyName = props.defaultPolicyName;
        this.principalAccount = props.account;
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        if (!this.defaultPolicy) {
            const useUniqueName = core_1.FeatureFlags.of(this).isEnabled(cx_api_1.IAM_IMPORTED_ROLE_STACK_SAFE_DEFAULT_POLICY_NAME);
            const defaultDefaultPolicyName = useUniqueName
                ? `Policy${core_1.Names.uniqueId(this)}`
                : 'Policy';
            const policyName = this.defaultPolicyName ?? defaultDefaultPolicyName;
            this.defaultPolicy = new policy_1.Policy(this, policyName, useUniqueName ? { policyName } : undefined);
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.defaultPolicy };
    }
    attachInlinePolicy(policy) {
        const thisAndPolicyAccountComparison = core_1.Token.compareStrings(this.env.account, policy.env.account);
        const equalOrAnyUnresolved = thisAndPolicyAccountComparison === core_1.TokenComparison.SAME ||
            thisAndPolicyAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
            thisAndPolicyAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
        if (equalOrAnyUnresolved) {
            this.attachedPolicies.attach(policy);
            policy.attachToRole(this);
        }
    }
    addManagedPolicy(policy) {
        core_1.Annotations.of(this).addWarning(`Not adding managed policy: ${policy.managedPolicyArn} to imported role: ${this.roleName}`);
    }
    grantPassRole(identity) {
        return this.grant(identity, 'iam:PassRole');
    }
    grantAssumeRole(identity) {
        return this.grant(identity, 'sts:AssumeRole');
    }
    grant(grantee, ...actions) {
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    dedupeString() {
        return `ImportedRole:${this.roleArn}`;
    }
}
exports.ImportedRole = ImportedRole;
//# sourceMappingURL=data:application/json;base64,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