"use strict";
// IAM Statement merging
//
// See docs/policy-merging.als for a formal model of the logic
// implemented here.
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeStatements = void 0;
const comparable_principal_1 = require("./comparable-principal");
const policy_statement_1 = require("../policy-statement");
const util_1 = require("../util");
/*
 * Don't produce any merged statements larger than this.
 *
 * They will become impossible to divide across managed policies if we do,
 * and this is the maximum size for User policies.
 */
const MAX_MERGE_SIZE = 2000;
/**
 * Merge as many statements as possible to shrink the total policy doc, modifying the input array in place
 *
 * We compare and merge all pairs of statements (O(N^2) complexity), opportunistically
 * merging them. This is not guaranteed to produce the optimal output, but it's probably
 * Good Enough(tm). If it merges anything, it's at least going to produce a smaller output
 * than the input.
 */
function mergeStatements(statements, options) {
    const sizeOptions = (0, policy_statement_1.deriveEstimateSizeOptions)(options.scope);
    const compStatements = statements.map(makeComparable);
    const mergeFn = options?.mergeIfCombinable ?? true ? mergeIfCombinable : mergeIfEqual;
    // Keep trying until nothing changes anymore
    while (onePass()) { /* again */ }
    const mergedStatements = new Array();
    const originsMap = new Map();
    for (const comp of compStatements) {
        const statement = renderComparable(comp);
        mergedStatements.push(statement);
        originsMap.set(statement, comp.originals);
    }
    return { mergedStatements, originsMap };
    // Do one optimization pass, return 'true' if we merged anything
    function onePass() {
        let ret = false;
        for (let i = 0; i < compStatements.length; i++) {
            let j = i + 1;
            while (j < compStatements.length) {
                const merged = mergeFn(compStatements[i], compStatements[j], !!options.limitSize, sizeOptions);
                if (merged) {
                    compStatements[i] = merged;
                    compStatements.splice(j, 1);
                    ret = true;
                }
                else {
                    j++;
                }
            }
        }
        return ret;
    }
}
exports.mergeStatements = mergeStatements;
/**
 * Given two statements, return their merging (if possible)
 *
 * We can merge two statements if:
 *
 * - Their effects are the same
 * - They don't have Sids (not really a hard requirement, but just a simplification and an escape hatch)
 * - Their Conditions are the same
 * - Their NotAction, NotResource and NotPrincipal sets are the same (empty sets is fine).
 * - From their Action, Resource and Principal sets, 2 are subsets of each other
 *   (empty sets are fine).
 */
function mergeIfCombinable(a, b, limitSize, options) {
    // Effects must be the same
    if (a.statement.effect !== b.statement.effect) {
        return;
    }
    // We don't merge Sids (for now)
    if (a.statement.sid || b.statement.sid) {
        return;
    }
    if (a.conditionString !== b.conditionString) {
        return;
    }
    if (!setEqual(a.statement.notActions, b.statement.notActions) ||
        !setEqual(a.statement.notResources, b.statement.notResources) ||
        !setEqualPrincipals(a.statement.notPrincipals, b.statement.notPrincipals)) {
        return;
    }
    // We can merge these statements if 2 out of the 3 sets of Action, Resource, Principal
    // are the same.
    const setsEqual = (setEqual(a.statement.actions, b.statement.actions) ? 1 : 0) +
        (setEqual(a.statement.resources, b.statement.resources) ? 1 : 0) +
        (setEqualPrincipals(a.statement.principals, b.statement.principals) ? 1 : 0);
    if (setsEqual < 2 || unmergeablePrincipals(a, b)) {
        return;
    }
    const combined = a.statement.copy({
        actions: setMerge(a.statement.actions, b.statement.actions),
        resources: setMerge(a.statement.resources, b.statement.resources),
        principals: setMergePrincipals(a.statement.principals, b.statement.principals),
    });
    if (limitSize && combined._estimateSize(options) > MAX_MERGE_SIZE) {
        return undefined;
    }
    return {
        originals: [...a.originals, ...b.originals],
        statement: combined,
        conditionString: a.conditionString,
    };
}
/**
 * We merge two statements only if they are exactly the same
 */
function mergeIfEqual(a, b) {
    if (a.statement.effect !== b.statement.effect) {
        return;
    }
    if (a.statement.sid !== b.statement.sid) {
        return;
    }
    if (a.conditionString !== b.conditionString) {
        return;
    }
    if (!setEqual(a.statement.notActions, b.statement.notActions) ||
        !setEqual(a.statement.notResources, b.statement.notResources) ||
        !setEqualPrincipals(a.statement.notPrincipals, b.statement.notPrincipals)) {
        return;
    }
    if (!setEqual(a.statement.actions, b.statement.actions) ||
        !setEqual(a.statement.resources, b.statement.resources) ||
        !setEqualPrincipals(a.statement.principals, b.statement.principals)) {
        return;
    }
    return {
        originals: [...a.originals, ...b.originals],
        statement: a.statement,
        conditionString: a.conditionString,
    };
}
/**
 * Calculate and return cached string set representation of the statement elements
 *
 * This is to be able to do comparisons on these sets quickly.
 */
function makeComparable(s) {
    return {
        originals: [s],
        statement: s,
        conditionString: JSON.stringify(s.conditions),
    };
}
/**
 * Return 'true' if the two principals are unmergeable
 *
 * This only happens if one of them is a literal, untyped principal (typically,
 * `Principal: '*'`) and the other one is typed.
 *
 * `Principal: '*'` behaves subtly different than `Principal: { AWS: '*' }` and must
 * therefore be preserved.
 */
function unmergeablePrincipals(a, b) {
    const aHasLiteral = a.statement.principals.some(v => util_1.LITERAL_STRING_KEY in v.policyFragment.principalJson);
    const bHasLiteral = b.statement.principals.some(v => util_1.LITERAL_STRING_KEY in v.policyFragment.principalJson);
    return aHasLiteral !== bHasLiteral;
}
/**
 * Turn a ComparableStatement back into a Statement
 */
function renderComparable(s) {
    return s.statement;
}
/**
 * Whether the given sets are equal
 */
function setEqual(a, b) {
    const bSet = new Set(b);
    return a.length === b.length && a.every(k => bSet.has(k));
}
/**
 * Merge two value sets
 */
function setMerge(x, y) {
    return Array.from(new Set([...x, ...y])).sort();
}
function setEqualPrincipals(xs, ys) {
    const xPrincipals = (0, comparable_principal_1.partitionPrincipals)(xs);
    const yPrincipals = (0, comparable_principal_1.partitionPrincipals)(ys);
    const nonComp = setEqual(xPrincipals.nonComparable, yPrincipals.nonComparable);
    const comp = setEqual(Object.keys(xPrincipals.comparable), Object.keys(yPrincipals.comparable));
    return nonComp && comp;
}
function setMergePrincipals(xs, ys) {
    const xPrincipals = (0, comparable_principal_1.partitionPrincipals)(xs);
    const yPrincipals = (0, comparable_principal_1.partitionPrincipals)(ys);
    const comparable = { ...xPrincipals.comparable, ...yPrincipals.comparable };
    return [...Object.values(comparable), ...xPrincipals.nonComparable, ...yPrincipals.nonComparable];
}
//# sourceMappingURL=data:application/json;base64,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