"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrecreatedRole = void 0;
const core_1 = require("../../../core");
const helpers_internal_1 = require("../../../core/lib/helpers-internal");
const constructs_1 = require("constructs");
/**
 * An IAM role that has been created outside of CDK and can be
 * used in place of a role that CDK _is_ creating.
 *
 * When any policy is attached to a precreated role the policy will be
 * synthesized into a separate report and will _not_ be synthesized in
 * the CloudFormation template.
 */
class PrecreatedRole extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            account: props.role.env.account,
            region: props.role.env.region,
        });
        this.grantPrincipal = this;
        this.policyStatements = [];
        this.managedPolicies = [];
        this.role = props.role;
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.policyFragment = this.role.policyFragment;
        this.principalAccount = this.role.principalAccount;
        this.roleArn = this.role.roleArn;
        this.roleName = this.role.roleName;
        this.stack = this.role.stack;
        const rolePath = props.rolePath ?? this.node.path;
        constructs_1.Dependable.implement(this, {
            dependencyRoots: [this.role],
        });
        // add a single PolicySynthesizer under the `App` scope
        this.policySynthesizer = helpers_internal_1.PolicySynthesizer.getOrCreate(this);
        this.policySynthesizer.addRole(rolePath, {
            roleName: this.roleName,
            managedPolicies: this.managedPolicies,
            policyStatements: this.policyStatements,
            assumeRolePolicy: core_1.Stack.of(this).resolve(props.assumeRolePolicy?.toJSON()?.Statement),
            missing: props.missing,
        });
    }
    attachInlinePolicy(policy) {
        const statements = policy.document.toJSON()?.Statement;
        if (statements && Array.isArray(statements)) {
            statements.forEach(statement => {
                this.policyStatements.push(statement);
            });
        }
    }
    addManagedPolicy(policy) {
        this.managedPolicies.push(policy);
    }
    addToPolicy(statement) {
        this.policyStatements.push(statement.toStatementJson());
        return false;
    }
    addToPrincipalPolicy(statement) {
        this.addToPolicy(statement);
        // If we return `false`, the grants will try to add the statement to the resource
        // (if possible).
        return { statementAdded: true, policyDependable: new constructs_1.DependencyGroup() };
    }
    grant(grantee, ...actions) {
        return this.role.grant(grantee, ...actions);
    }
    grantPassRole(grantee) {
        return this.role.grantPassRole(grantee);
    }
    grantAssumeRole(identity) {
        return this.role.grantAssumeRole(identity);
    }
}
exports.PrecreatedRole = PrecreatedRole;
//# sourceMappingURL=data:application/json;base64,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