"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Role = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const helpers_internal_1 = require("../../core/lib/helpers-internal");
const constructs_1 = require("constructs");
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const managed_policy_1 = require("./managed-policy");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const principals_1 = require("./principals");
const assume_role_policy_1 = require("./private/assume-role-policy");
const immutable_role_1 = require("./private/immutable-role");
const imported_role_1 = require("./private/imported-role");
const policydoc_adapter_1 = require("./private/policydoc-adapter");
const precreated_role_1 = require("./private/precreated-role");
const util_1 = require("./private/util");
const MAX_INLINE_SIZE = 10000;
const MAX_MANAGEDPOL_SIZE = 6000;
const IAM_ROLE_SYMBOL = Symbol.for('@aws-cdk/packages/aws-iam/lib/role.Role');
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends core_1.Resource {
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_FromRoleArnOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromRoleArn);
            }
            throw error;
        }
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.splitArn(roleArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const resourceName = parsedArn.resourceName;
        const roleAccount = parsedArn.account;
        // service roles have an ARN like 'arn:aws:iam::<account>:role/service-role/<roleName>'
        // or 'arn:aws:iam::<account>:role/service-role/servicename.amazonaws.com/service-role/<roleName>'
        // we want to support these as well, so we just use the element after the last slash as role name
        const roleName = resourceName.split('/').pop();
        if ((0, helpers_internal_1.getCustomizeRolesConfig)(scope).enabled) {
            return new precreated_role_1.PrecreatedRole(scope, id, {
                rolePath: `${scope.node.path}/${id}`,
                role: new imported_role_1.ImportedRole(scope, `Import${id}`, {
                    account: roleAccount,
                    roleArn,
                    roleName,
                    ...options,
                }),
            });
        }
        if (options.addGrantsToResources !== undefined && options.mutable !== false) {
            throw new Error('\'addGrantsToResources\' can only be passed if \'mutable: false\'');
        }
        const roleArnAndScopeStackAccountComparison = core_1.Token.compareStrings(roleAccount ?? '', scopeStack.account);
        const equalOrAnyUnresolved = roleArnAndScopeStackAccountComparison === core_1.TokenComparison.SAME ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
        // if we are returning an immutable role then the 'importedRole' is just a throwaway construct
        // so give it a different id
        const mutableRoleId = (options.mutable !== false && equalOrAnyUnresolved) ? id : `MutableRole${id}`;
        const importedRole = new imported_role_1.ImportedRole(scope, mutableRoleId, {
            roleArn,
            roleName,
            account: roleAccount,
            ...options,
        });
        // we only return an immutable Role if both accounts were explicitly provided, and different
        return options.mutable !== false && equalOrAnyUnresolved
            ? importedRole
            : new immutable_role_1.ImmutableRole(scope, id, importedRole, options.addGrantsToResources ?? false);
    }
    /**
      * Return whether the given object is a Role
     */
    static isRole(x) {
        return x !== null && typeof (x) === 'object' && IAM_ROLE_SYMBOL in x;
    }
    /**
     * Import an external role by name.
     *
     * The imported role is assumed to exist in the same account as the account
     * the scope's containing Stack is being deployed to.
  
     * @param scope construct scope
     * @param id construct id
     * @param roleName the name of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleName(scope, id, roleName, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_FromRoleNameOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromRoleName);
            }
            throw error;
        }
        // Validate the role name only if not a token
        if (!core_1.Token.isUnresolved(roleName)) {
            this.validateRoleName(roleName);
        }
        return Role.fromRoleArn(scope, id, core_1.Stack.of(scope).formatArn({
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: roleName,
        }), options);
    }
    /**
     * Customize the creation of IAM roles within the given scope
     *
     * It is recommended that you **do not** use this method and instead allow
     * CDK to manage role creation. This should only be used
     * in environments where CDK applications are not allowed to created IAM roles.
     *
     * This can be used to prevent the CDK application from creating roles
     * within the given scope and instead replace the references to the roles with
     * precreated role names. A report will be synthesized in the cloud assembly (i.e. cdk.out)
     * that will contain the list of IAM roles that would have been created along with the
     * IAM policy statements that the role should contain. This report can then be used
     * to create the IAM roles outside of CDK and then the created role names can be provided
     * in `usePrecreatedRoles`.
     *
     * @example
     * declare const app: App;
     * Role.customizeRoles(app, {
     *   usePrecreatedRoles: {
     *     'ConstructPath/To/Role': 'my-precreated-role-name',
     *   },
     * });
     *
     * @param scope construct scope to customize role creation
     * @param options options for configuring role creation
     */
    static customizeRoles(scope, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_CustomizeRolesOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.customizeRoles);
            }
            throw error;
        }
        const preventSynthesis = options?.preventSynthesis ?? true;
        const useRoles = {};
        for (const [constructPath, roleName] of Object.entries(options?.usePrecreatedRoles ?? {})) {
            const absPath = constructPath.startsWith(scope.node.path)
                ? constructPath
                : `${scope.node.path}/${constructPath}`;
            useRoles[absPath] = roleName;
        }
        scope.node.setContext(helpers_internal_1.CUSTOMIZE_ROLES_CONTEXT_KEY, {
            preventSynthesis,
            usePrecreatedRoles: useRoles,
        });
    }
    static validateRoleName(roleName) {
        // https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html
        const regexp = /[\w+=,.@-]+/;
        const matches = regexp.exec(roleName);
        if (!(matches && matches.length === 1 && matches[0] === roleName)) {
            throw new Error(`The role name ${roleName} does not match the IAM conventions.`);
        }
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.roleName,
        });
        this.grantPrincipal = this;
        this.principalAccount = this.env.account;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        this.dependables = new Map();
        this._didSplit = false;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_RoleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Role);
            }
            throw error;
        }
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.inlinePolicies = props.inlinePolicies || {};
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const description = (props.description && props.description?.length > 0) ? props.description : undefined;
        if (description && description.length > 1000) {
            throw new Error('Role description must be no longer than 1000 characters.');
        }
        validateRolePath(props.path);
        const config = this.getPrecreatedRoleConfig();
        const roleArn = core_1.Stack.of(scope).formatArn({
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: config.precreatedRoleName,
        });
        const importedRole = new imported_role_1.ImportedRole(this, 'Import' + id, {
            roleArn,
            roleName: config.precreatedRoleName ?? id,
            account: core_1.Stack.of(this).account,
        });
        this.roleName = importedRole.roleName;
        this.roleArn = importedRole.roleArn;
        if (config.enabled) {
            const role = new precreated_role_1.PrecreatedRole(this, 'PrecreatedRole' + id, {
                rolePath: this.node.path,
                role: importedRole,
                missing: !config.precreatedRoleName,
                assumeRolePolicy: this.assumeRolePolicy,
            });
            this.managedPolicies.forEach(policy => role.addManagedPolicy(policy));
            Object.entries(this.inlinePolicies).forEach(([name, policy]) => {
                role.attachInlinePolicy(new policy_1.Policy(this, name, { document: policy }));
            });
            this._precreatedRole = role;
        }
        // synthesize the resource if preventSynthesis=false
        if (!config.preventSynthesis) {
            const role = new iam_generated_1.CfnRole(this, 'Resource', {
                assumeRolePolicyDocument: this.assumeRolePolicy,
                managedPolicyArns: util_1.UniqueStringSet.from(() => this.managedPolicies.map(p => p.managedPolicyArn)),
                policies: _flatten(this.inlinePolicies),
                path: props.path,
                permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
                roleName: this.physicalName,
                maxSessionDuration,
                description,
            });
            this._roleId = role.attrRoleId;
            this.roleArn = this.getResourceArnAttribute(role.attrArn, {
                region: '',
                service: 'iam',
                resource: 'role',
                // Removes leading slash from path
                resourceName: `${props.path ? props.path.substr(props.path.charAt(0) === '/' ? 1 : 0) : ''}${this.physicalName}`,
            });
            this.roleName = this.getResourceNameAttribute(role.ref);
            core_1.Aspects.of(this).add({
                visit: (c) => {
                    if (c === this) {
                        this.splitLargePolicy();
                    }
                },
            });
        }
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
        this.node.addValidation({ validate: () => this.validateRole() });
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalPolicy);
            }
            throw error;
        }
        if (this._precreatedRole) {
            return this._precreatedRole.addToPrincipalPolicy(statement);
        }
        else {
            if (!this.defaultPolicy) {
                this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
                this.attachInlinePolicy(this.defaultPolicy);
            }
            this.defaultPolicy.addStatements(statement);
            // We might split this statement off into a different policy, so we'll need to
            // late-bind the dependable.
            const policyDependable = new constructs_1.DependencyGroup();
            this.dependables.set(statement, policyDependable);
            return { statementAdded: true, policyDependable };
        }
    }
    addToPolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPolicy);
            }
            throw error;
        }
        if (this._precreatedRole) {
            return this._precreatedRole.addToPolicy(statement);
        }
        else {
            return this.addToPrincipalPolicy(statement).statementAdded;
        }
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IManagedPolicy(policy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addManagedPolicy);
            }
            throw error;
        }
        if (this._precreatedRole) {
            return this._precreatedRole.addManagedPolicy(policy);
        }
        else {
            if (this.managedPolicies.find(mp => mp === policy)) {
                return;
            }
            this.managedPolicies.push(policy);
        }
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_Policy(policy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attachInlinePolicy);
            }
            throw error;
        }
        if (this._precreatedRole) {
            this._precreatedRole.attachInlinePolicy(policy);
        }
        else {
            this.attachedPolicies.attach(policy);
            policy.attachToRole(this);
        }
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee, ...actions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IPrincipal(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IPrincipal(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPassRole);
            }
            throw error;
        }
        return this.grant(identity, 'iam:PassRole');
    }
    /**
     * Grant permissions to the given principal to assume this role.
     */
    grantAssumeRole(identity) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IPrincipal(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantAssumeRole);
            }
            throw error;
        }
        return this.grant(identity, 'sts:AssumeRole');
    }
    /**
     * Returns the stable and unique string identifying the role. For example,
     * AIDAJQABLZS4A3QDU576Q.
     *
     * @attribute
     */
    get roleId() {
        if (!this._roleId) {
            throw new Error('"roleId" is not available on precreated roles');
        }
        return this._roleId;
    }
    /**
     * Return a copy of this Role object whose Policies will not be updated
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     */
    withoutPolicyUpdates(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_WithoutPolicyUpdatesOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.withoutPolicyUpdates);
            }
            throw error;
        }
        if (!this.immutableRole) {
            this.immutableRole = new immutable_role_1.ImmutableRole(constructs_1.Node.of(this).scope, `ImmutableRole${this.node.id}`, this, options.addGrantsToResources ?? false);
        }
        return this.immutableRole;
    }
    validateRole() {
        const errors = new Array();
        errors.push(...this.assumeRolePolicy?.validateForResourcePolicy() ?? []);
        for (const policy of Object.values(this.inlinePolicies)) {
            errors.push(...policy.validateForIdentityPolicy());
        }
        return errors;
    }
    /**
     * Split large inline policies into managed policies
     *
     * This gets around the 10k bytes limit on role policies.
     */
    splitLargePolicy() {
        if (!this.defaultPolicy || this._didSplit) {
            return;
        }
        this._didSplit = true;
        const self = this;
        const originalDoc = this.defaultPolicy.document;
        const splitOffDocs = originalDoc._splitDocument(this, MAX_INLINE_SIZE, MAX_MANAGEDPOL_SIZE);
        // Includes the "current" document
        const mpCount = this.managedPolicies.length + (splitOffDocs.size - 1);
        if (mpCount > 20) {
            core_1.Annotations.of(this).addWarning(`Policy too large: ${mpCount} exceeds the maximum of 20 managed policies attached to a Role`);
        }
        else if (mpCount > 10) {
            core_1.Annotations.of(this).addWarning(`Policy large: ${mpCount} exceeds 10 managed policies attached to a Role, this requires a quota increase`);
        }
        // Create the managed policies and fix up the dependencies
        markDeclaringConstruct(originalDoc, this.defaultPolicy);
        let i = 1;
        for (const newDoc of splitOffDocs.keys()) {
            if (newDoc === originalDoc) {
                continue;
            }
            const mp = new managed_policy_1.ManagedPolicy(this, `OverflowPolicy${i++}`, {
                description: `Part of the policies for ${this.node.path}`,
                document: newDoc,
                roles: [this],
            });
            markDeclaringConstruct(newDoc, mp);
        }
        /**
         * Update the Dependables for the statements in the given PolicyDocument to point to the actual declaring construct
         */
        function markDeclaringConstruct(doc, declaringConstruct) {
            for (const original of splitOffDocs.get(doc) ?? []) {
                self.dependables.get(original)?.add(declaringConstruct);
            }
        }
    }
    /**
     * Return configuration for precreated roles
     */
    getPrecreatedRoleConfig() {
        return (0, helpers_internal_1.getPrecreatedRoleConfig)(this);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Role[_a] = { fqn: "aws-cdk-lib.aws_iam.Role", version: "2.74.0" };
exports.Role = Role;
function createAssumeRolePolicy(principal, externalIds) {
    const actualDoc = new policy_document_1.PolicyDocument();
    // If requested, add externalIds to every statement added to this doc
    const addDoc = externalIds.length === 0
        ? actualDoc
        : new policydoc_adapter_1.MutatingPolicyDocumentAdapter(actualDoc, (statement) => {
            statement.addCondition('StringEquals', {
                'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds,
            });
            return statement;
        });
    (0, assume_role_policy_1.defaultAddPrincipalToAssumeRole)(principal, addDoc);
    return actualDoc;
}
function validateRolePath(path) {
    if (path === undefined || core_1.Token.isUnresolved(path)) {
        return;
    }
    const validRolePath = /^(\/|\/[\u0021-\u007F]+\/)$/;
    if (path.length == 0 || path.length > 512) {
        throw new Error(`Role path must be between 1 and 512 characters. The provided role path is ${path.length} characters.`);
    }
    else if (!validRolePath.test(path)) {
        throw new Error('Role path must be either a slash or valid characters (alphanumerics and symbols) surrounded by slashes. '
            + `Valid characters are unicode characters in [\\u0021-\\u007F]. However, ${path} is provided.`);
    }
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
Object.defineProperty(Role.prototype, IAM_ROLE_SYMBOL, {
    value: true,
    enumerable: false,
    writable: false,
});
//# sourceMappingURL=data:application/json;base64,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