"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const core_1 = require("../../core");
const helpers_internal_1 = require("../../core/lib/helpers-internal");
const constructs_1 = require("constructs");
const stream_1 = require("./stream");
/**
 * The authentication method to use with SelfManagedKafkaEventSource
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    /**
     * SASL_SCRAM_512_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    /**
     * SASL_SCRAM_256_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
    /**
     * BASIC_AUTH (SASL/PLAIN) authentication method for your Kafka cluster
     */
    AuthenticationMethod["BASIC_AUTH"] = "BASIC_AUTH";
    /**
     * CLIENT_CERTIFICATE_TLS_AUTH (mTLS) authentication method for your Kafka cluster
     */
    AuthenticationMethod["CLIENT_CERTIFICATE_TLS_AUTH"] = "CLIENT_CERTIFICATE_TLS_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        this._eventSourceMappingId = undefined;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_event_sources_ManagedKafkaEventSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ManagedKafkaEventSource);
            }
            throw error;
        }
        this.innerProps = props;
    }
    bind(target) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IFunction(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const eventSourceMapping = target.addEventSourceMapping(`KafkaEventSource:${core_1.Names.nodeUniqueId(target.node)}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
            kafkaConsumerGroupId: this.innerProps.consumerGroupId,
        }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        target.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
    /**
    * The identifier for this EventSourceMapping
    */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KafkaEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "aws-cdk-lib.aws_lambda_event_sources.ManagedKafkaEventSource", version: "2.74.0" };
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_event_sources_SelfManagedKafkaEventSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SelfManagedKafkaEventSource);
            }
            throw error;
        }
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    bind(target) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IFunction(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (!(target instanceof constructs_1.Construct)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            kafkaConsumerGroupId: this.innerProps.consumerGroupId,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        const idHash = (0, helpers_internal_1.md5hash)(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.BASIC_AUTH:
                authType = lambda.SourceAccessConfigurationType.BASIC_AUTH;
                break;
            case AuthenticationMethod.CLIENT_CERTIFICATE_TLS_AUTH:
                authType = lambda.SourceAccessConfigurationType.CLIENT_CERTIFICATE_TLS_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.rootCACertificate !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SERVER_ROOT_CA_CERTIFICATE,
                uri: this.innerProps.rootCACertificate.secretArn,
            });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            this.innerProps.vpc?.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "aws-cdk-lib.aws_lambda_event_sources.SelfManagedKafkaEventSource", version: "2.74.0" };
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
//# sourceMappingURL=data:application/json;base64,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