"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("../../aws-lambda");
const cdk = require("../../core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    constructor(props) {
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath, props.logLevel);
        Bundling.esbuildInstallation = Bundling.esbuildInstallation ?? package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = Bundling.tscInstallation ?? package_installation_1.PackageInstallation.detect('typescript');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        if (props.format === types_1.OutputFormat.ESM
            && (props.runtime === aws_lambda_1.Runtime.NODEJS_10_X || props.runtime === aws_lambda_1.Runtime.NODEJS_12_X)) {
            throw new Error(`ECMAScript module output format is not supported by the ${props.runtime.name} runtime`);
        }
        this.externals = [
            ...props.externalModules ?? (isSdkV2Runtime(props.runtime) ? ['aws-sdk'] : ['@aws-sdk/*']),
            ...props.nodeModules ?? [], // Mark the modules that we are going to install as externals also
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? props.dockerImage ?? cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs ?? {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: props.esbuildVersion ?? ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        })
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux', // linux docker image
        });
        this.command = props.command ?? ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = props.workingDirectory ?? '/';
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    createBundlingCommand(options) {
        const pathJoin = osPathJoin(options.osPlatform);
        let relativeEntryPath = pathJoin(options.inputDir, this.relativeEntryPath);
        let tscCommand = '';
        if (this.props.preCompilation) {
            const tsconfig = this.props.tsconfig ?? (0, util_1.findUp)('tsconfig.json', path.dirname(this.props.entry));
            if (!tsconfig) {
                throw new Error('Cannot find a `tsconfig.json` but `preCompilation` is set to `true`, please specify it via `tsconfig`');
            }
            const compilerOptions = (0, util_1.getTsconfigCompilerOptions)(tsconfig);
            tscCommand = `${options.tscRunner} "${relativeEntryPath}" ${compilerOptions}`;
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
        }
        const loaders = Object.entries(this.props.loader ?? {});
        const defines = Object.entries(this.props.define ?? {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = this.props.sourceMapMode ?? this.props.sourceMap;
        const sourceMapMode = this.props.sourceMapMode ?? types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = this.props.sourcesContent ?? true;
        const outFile = this.props.format === types_1.OutputFormat.ESM ? 'index.mjs' : 'index.js';
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${relativeEntryPath}"`,
            `--target=${this.props.target ?? toTarget(this.props.runtime)}`,
            '--platform=node',
            ...this.props.format ? [`--format=${this.props.format}`] : [],
            `--outfile="${pathJoin(options.outputDir, outFile)}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
            ...this.props.mainFields ? [`--main-fields=${this.props.mainFields.join(',')}`] : [],
            ...this.props.inject ? this.props.inject.map(i => `--inject:${i}`) : [],
            ...this.props.esbuildArgs ? [toCliArgs(this.props.esbuildArgs)] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = (0, util_1.findUp)('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = (0, util_1.extractDependencies)(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, this.relativeDepsLockFilePath ?? this.packageManager.lockFile);
            const isPnpm = this.packageManager.lockFile === package_manager_1.LockFile.PNPM;
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                isPnpm ? osCommand.write(pathJoin(options.outputDir, 'pnpm-workspace.yaml'), '') : '',
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
                isPnpm ? osCommand.remove(pathJoin(options.outputDir, 'node_modules', '.modules.yaml')) : '', // Remove '.modules.yaml' file which changes on each deployment
            ]);
        }
        return chain([
            ...this.props.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(this.props.nodeModules && this.props.commandHooks?.beforeInstall(options.inputDir, options.outputDir)) ?? [],
            depsCommand,
            ...this.props.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? [],
        ]);
    }
    getLocalBundlingProvider() {
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = this.props.environment ?? {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit', // inherit stderr
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    write(filePath, data) {
        if (this.osPlatform === 'win32') {
            if (!data) { // if `data` is empty, echo a blank line, otherwise the file will contain a `^` character
                return `echo. > "${filePath}"`;
            }
            return `echo ^${data}^ > "${filePath}"`;
        }
        return `echo '${data}' > "${filePath}"`;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        return this.write(filePath, stringifiedData);
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy "${src}" "${dest}"`;
        }
        return `cp "${src}" "${dest}"`;
    }
    changeDirectory(dir) {
        return `cd "${dir}"`;
    }
    remove(filePath) {
        if (this.osPlatform === 'win32') {
            return `del "${filePath}"`;
        }
        return `rm "${filePath}"`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
function toCliArgs(esbuildArgs) {
    const args = [];
    for (const [key, value] of Object.entries(esbuildArgs)) {
        if (value === true || value === '') {
            args.push(key);
        }
        else if (value) {
            args.push(`${key}="${value}"`);
        }
    }
    return args.join(' ');
}
function isSdkV2Runtime(runtime) {
    const sdkV2RuntimeList = [
        aws_lambda_1.Runtime.NODEJS,
        aws_lambda_1.Runtime.NODEJS_4_3,
        aws_lambda_1.Runtime.NODEJS_6_10,
        aws_lambda_1.Runtime.NODEJS_8_10,
        aws_lambda_1.Runtime.NODEJS_10_X,
        aws_lambda_1.Runtime.NODEJS_12_X,
        aws_lambda_1.Runtime.NODEJS_14_X,
        aws_lambda_1.Runtime.NODEJS_16_X,
    ];
    return sdkV2RuntimeList.some((r) => { return r.family === runtime.family && r.name === runtime.name; });
}
//# sourceMappingURL=data:application/json;base64,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