"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda");
const aws_lambda_1 = require("../../aws-lambda");
const bundling_1 = require("./bundling");
const package_manager_1 = require("./package-manager");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using esbuild
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_nodejs_NodejsFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, NodejsFunction);
            }
            throw error;
        }
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = props.handler ?? 'handler';
        const runtime = props.runtime ?? lambda.Runtime.NODEJS_14_X;
        const architecture = props.architecture ?? aws_lambda_1.Architecture.X86_64;
        const depsLockFilePath = findLockFile(props.depsLockFilePath);
        const projectRoot = props.projectRoot ?? path.dirname(depsLockFilePath);
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...props.bundling ?? {},
                entry,
                runtime,
                architecture,
                depsLockFilePath,
                projectRoot,
            }),
            handler: handler.indexOf('.') !== -1 ? `${handler}` : `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if (props.awsSdkConnectionReuse ?? true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
NodejsFunction[_a] = { fqn: "aws-cdk-lib.aws_lambda_nodejs.NodejsFunction", version: "2.74.0" };
exports.NodejsFunction = NodejsFunction;
/**
 * Checks given lock file or searches for a lock file
 */
function findLockFile(depsLockFilePath) {
    if (depsLockFilePath) {
        if (!fs.existsSync(depsLockFilePath)) {
            throw new Error(`Lock file at ${depsLockFilePath} doesn't exist`);
        }
        if (!fs.statSync(depsLockFilePath).isFile()) {
            throw new Error('`depsLockFilePath` should point to a file');
        }
        return path.resolve(depsLockFilePath);
    }
    const lockFiles = (0, util_1.findUpMultiple)([
        package_manager_1.LockFile.PNPM,
        package_manager_1.LockFile.YARN,
        package_manager_1.LockFile.NPM,
    ]);
    if (lockFiles.length === 0) {
        throw new Error('Cannot find a package lock file (`pnpm-lock.yaml`, `yarn.lock` or `package-lock.json`). Please specify it with `depsLockFilePath`.');
    }
    if (lockFiles.length > 1) {
        throw new Error(`Multiple package lock files found: ${lockFiles.join(', ')}. Please specify the desired one with \`depsLockFilePath\`.`);
    }
    return lockFiles[0];
}
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 * 4. A .mjs file name as the defining file with id as suffix (defining-file.id.mjs)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?|mjs)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    const mjsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.mjs`);
    if (fs.existsSync(mjsHandlerFile)) {
        return mjsHandlerFile;
    }
    throw new Error(`Cannot find handler file ${tsHandlerFile}, ${jsHandlerFile} or ${mjsHandlerFile}`);
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = (0, util_1.callsites)();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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