"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageManager = exports.LockFile = void 0;
const os = require("os");
const path = require("path");
const types_1 = require("./types");
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
    LockFile["PNPM"] = "pnpm-lock.yaml";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
/**
 * A node package manager
 */
class PackageManager {
    /**
     * Use a lock file path to determine the package manager to use. Optionally, specify a log level to
     * control its verbosity.
     * @param lockFilePath Path of the lock file
     * @param logLevel optional log level @default LogLevel.INFO
     * @returns the right PackageManager for that lock file
     */
    static fromLockFile(lockFilePath, logLevel) {
        const lockFile = path.basename(lockFilePath);
        switch (lockFile) {
            case LockFile.YARN:
                return new PackageManager({
                    lockFile: LockFile.YARN,
                    installCommand: logLevel && logLevel !== types_1.LogLevel.INFO ? ['yarn', 'install', '--no-immutable', '--silent'] : ['yarn', 'install', '--no-immutable'],
                    runCommand: ['yarn', 'run'],
                });
            case LockFile.PNPM:
                return new PackageManager({
                    lockFile: LockFile.PNPM,
                    installCommand: logLevel && logLevel !== types_1.LogLevel.INFO ? ['pnpm', 'install', '--reporter', 'silent', '--config.node-linker=hoisted', '--config.package-import-method=clone-or-copy', '--no-prefer-frozen-lockfile'] : ['pnpm', 'install', '--config.node-linker=hoisted', '--config.package-import-method=clone-or-copy', '--no-prefer-frozen-lockfile'],
                    // --config.node-linker=hoisted to create flat node_modules without symlinks
                    // --config.package-import-method=clone-or-copy to avoid hardlinking packages from the store
                    // --no-prefer-frozen-lockfile (works the same as yarn's --no-immutable) Disable --frozen-lockfile that is enabled by default in CI environments (https://github.com/pnpm/pnpm/issues/1994).
                    runCommand: ['pnpm', 'exec'],
                    argsSeparator: '--',
                });
            default:
                return new PackageManager({
                    lockFile: LockFile.NPM,
                    installCommand: logLevel ? ['npm', 'ci', '--loglevel', logLevel] : ['npm', 'ci'],
                    runCommand: ['npx', '--no-install'],
                });
        }
    }
    constructor(props) {
        this.lockFile = props.lockFile;
        this.installCommand = props.installCommand;
        this.runCommand = props.runCommand;
        this.argsSeparator = props.argsSeparator;
    }
    runBinCommand(bin) {
        const [runCommand, ...runArgs] = this.runCommand;
        return [
            os.platform() === 'win32' ? `${runCommand}.cmd` : runCommand,
            ...runArgs,
            ...(this.argsSeparator ? [this.argsSeparator] : []),
            bin,
        ].join(' ');
    }
}
exports.PackageManager = PackageManager;
//# sourceMappingURL=data:application/json;base64,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