"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Charset = exports.SourceMapMode = exports.LogLevel = exports.OutputFormat = void 0;
/**
 * Output format for the generated JavaScript files
 */
var OutputFormat;
(function (OutputFormat) {
    /**
     * CommonJS
     */
    OutputFormat["CJS"] = "cjs";
    /**
     * ECMAScript module
     *
     * Requires a running environment that supports `import` and `export` syntax.
     */
    OutputFormat["ESM"] = "esm";
})(OutputFormat = exports.OutputFormat || (exports.OutputFormat = {}));
/**
 * Log levels for esbuild and package managers' install commands.
 */
var LogLevel;
(function (LogLevel) {
    /** Show everything */
    LogLevel["INFO"] = "info";
    /** Show warnings and errors */
    LogLevel["WARNING"] = "warning";
    /** Show errors only */
    LogLevel["ERROR"] = "error";
    /** Show nothing */
    LogLevel["SILENT"] = "silent";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * SourceMap mode for esbuild
 * @see https://esbuild.github.io/api/#sourcemap
 */
var SourceMapMode;
(function (SourceMapMode) {
    /**
     * Default sourceMap mode - will generate a .js.map file alongside any generated .js file and add a special //# sourceMappingURL=
     * comment to the bottom of the .js file pointing to the .js.map file
     */
    SourceMapMode["DEFAULT"] = "default";
    /**
     *  External sourceMap mode - If you want to omit the special //# sourceMappingURL= comment from the generated .js file but you still
     *  want to generate the .js.map files
     */
    SourceMapMode["EXTERNAL"] = "external";
    /**
     * Inline sourceMap mode - If you want to insert the entire source map into the .js file instead of generating a separate .js.map file
     */
    SourceMapMode["INLINE"] = "inline";
    /**
     * Both sourceMap mode - If you want to have the effect of both inline and external simultaneously
     */
    SourceMapMode["BOTH"] = "both";
})(SourceMapMode = exports.SourceMapMode || (exports.SourceMapMode = {}));
/**
 * Charset for esbuild's output
 */
var Charset;
(function (Charset) {
    /**
     * ASCII
     *
     * Any non-ASCII characters are escaped using backslash escape sequences
     */
    Charset["ASCII"] = "ascii";
    /**
     * UTF-8
     *
     * Keep original characters without using escape sequences
     */
    Charset["UTF8"] = "utf8";
})(Charset = exports.Charset || (exports.Charset = {}));
//# sourceMappingURL=data:application/json;base64,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