"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTsconfigCompilerOptions = exports.extractDependencies = exports.tryGetModuleVersionFromPkg = exports.tryGetModuleVersionFromRequire = exports.exec = exports.findUpMultiple = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = new Error().stack?.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    return findUpMultiple([name], directory)[0];
}
exports.findUp = findUp;
/**
 * Find the lowest of multiple files by walking up parent directories. If
 * multiple files exist at the same level, they will all be returned.
 */
function findUpMultiple(names, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const files = [];
    for (const name of names) {
        const file = path.join(directory, name);
        if (fs.existsSync(file)) {
            files.push(file);
        }
    }
    if (files.length > 0) {
        return files;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return [];
    }
    return findUpMultiple(names, path.dirname(absoluteDirectory));
}
exports.findUpMultiple = findUpMultiple;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    const proc = (0, child_process_1.spawnSync)(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${proc.stdout?.toString().trim()}\n\n\nstderr: ${proc.stderr?.toString().trim()}`);
        }
        throw new Error(`${cmd} ${args.join(' ')} ${options?.cwd ? `run in directory ${options.cwd}` : ''} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersionFromRequire(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersionFromRequire = tryGetModuleVersionFromRequire;
/**
 * Gets module version from package.json content
 */
function tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath) {
    const dependencies = {
        ...pkgJson.dependencies ?? {},
        ...pkgJson.devDependencies ?? {},
        ...pkgJson.peerDependencies ?? {},
    };
    if (!dependencies[mod]) {
        return undefined;
    }
    // If it's a "file:" version, make it absolute
    const fileMatch = dependencies[mod].match(/file:(.+)/);
    if (fileMatch && !path.isAbsolute(fileMatch[1])) {
        const absoluteFilePath = path.join(path.dirname(pkgPath), fileMatch[1]);
        return `file:${absoluteFilePath}`;
    }
    ;
    return dependencies[mod];
}
exports.tryGetModuleVersionFromPkg = tryGetModuleVersionFromPkg;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    for (const mod of modules) {
        const version = tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath)
            ?? tryGetModuleVersionFromRequire(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
function getTsconfigCompilerOptions(tsconfigPath) {
    const compilerOptions = extractTsConfig(tsconfigPath);
    const excludedCompilerOptions = [
        'composite',
        'tsBuildInfoFile',
    ];
    const options = {
        ...compilerOptions,
        // Overrides
        incremental: false,
        // Intentionally Setting rootDir and outDir, so that the compiled js file always end up next to .ts file.
        rootDir: './',
        outDir: './',
    };
    let compilerOptionsString = '';
    Object.keys(options).sort().forEach((key) => {
        if (excludedCompilerOptions.includes(key)) {
            return;
        }
        const value = options[key];
        const option = '--' + key;
        const type = typeof value;
        if (type === 'boolean') {
            if (value) {
                compilerOptionsString += option + ' ';
            }
            else {
                compilerOptionsString += option + ' false ';
            }
        }
        else if (type === 'string') {
            compilerOptionsString += option + ' ' + value + ' ';
        }
        else if (type === 'object') {
            if (Array.isArray(value)) {
                compilerOptionsString += option + ' ' + value.join(',') + ' ';
            }
        }
        else {
            throw new Error(`Missing support for compilerOption: [${key}]: { ${type}, ${value}} \n`);
        }
    });
    return compilerOptionsString.trim();
}
exports.getTsconfigCompilerOptions = getTsconfigCompilerOptions;
function extractTsConfig(tsconfigPath, previousCompilerOptions) {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const { extends: extendedConfig, compilerOptions } = require(tsconfigPath);
    const updatedCompilerOptions = {
        ...compilerOptions,
        ...(previousCompilerOptions ?? {}),
    };
    if (extendedConfig) {
        return extractTsConfig(path.resolve(tsconfigPath.replace(/[^\/]+$/, ''), extendedConfig), updatedCompilerOptions);
    }
    return updatedCompilerOptions;
}
//# sourceMappingURL=data:application/json;base64,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