"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("../../aws-applicationautoscaling");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    static fromAliasAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_AliasAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAliasAttributes);
            }
            throw error;
        }
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.architecture = attrs.aliasVersion.lambda.architecture;
                this.canCreatePermissions = this._isStackAccount();
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        this.canCreatePermissions = true;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_AliasProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Alias);
            }
            throw error;
        }
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        this.architecture = this.lambda.architecture;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.qualifier = (0, lambda_version_1.extractQualifierFromArn)(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.splitArn(this.functionArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}:${this.aliasName}`;
    }
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    get role() {
        return this.version.role;
    }
    metric(metricName, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metric);
            }
            throw error;
        }
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differs from the base behavior.
        return super.metric(metricName, {
            dimensionsMap: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Configure provisioned concurrency autoscaling on a function alias. Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options
     */
    addAutoScaling(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_AutoScalingOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAutoScaling);
            }
            throw error;
        }
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: options.minCapacity ?? 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
_a = JSII_RTTI_SYMBOL_1;
Alias[_a] = { fqn: "aws-cdk-lib.aws_lambda.Alias", version: "2.74.0" };
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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