"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
/**
 * Represents the filesystem for the Lambda function
 */
class FileSystem {
    /**
     * mount the filesystem from Amazon EFS
     * @param ap the Amazon EFS access point
     * @param mountPath the target path in the lambda runtime environment
     */
    static fromEfsAccessPoint(ap, mountPath) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_efs_IAccessPoint(ap);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEfsAccessPoint);
            }
            throw error;
        }
        return new FileSystem({
            localMountPath: mountPath,
            arn: ap.accessPointArn,
            dependency: [ap.fileSystem.mountTargetsAvailable],
            connections: ap.fileSystem.connections,
            policies: [
                new iam.PolicyStatement({
                    actions: ['elasticfilesystem:ClientMount'],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'elasticfilesystem:AccessPointArn': ap.accessPointArn,
                        },
                    },
                }),
                new iam.PolicyStatement({
                    actions: ['elasticfilesystem:ClientWrite'],
                    resources: [core_1.Stack.of(ap).formatArn({
                            service: 'elasticfilesystem',
                            resource: 'file-system',
                            resourceName: ap.fileSystem.fileSystemId,
                        })],
                }),
            ],
        });
    }
    /**
     * @param config the FileSystem configurations for the Lambda function
     */
    constructor(config) {
        this.config = config;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_FileSystemConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FileSystem);
            }
            throw error;
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
FileSystem[_a] = { fqn: "aws-cdk-lib.aws_lambda.FileSystem", version: "2.74.0" };
exports.FileSystem = FileSystem;
//# sourceMappingURL=data:application/json;base64,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