"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto_1 = require("crypto");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const function_url_1 = require("./function-url");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Flag to delay adding a warning message until current version is invoked.
         * @internal
         */
        this._warnIfCurrentVersionCalled = false;
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
        /**
         * Mapping of fucntion URL invocation principals to grants. Used to de-dupe `grantInvokeUrl()` calls.
         * @internal
         */
        this._functionUrlInvocationGrants = {};
    }
    /**
     * A warning will be added to functions under the following conditions:
     * - permissions that include `lambda:InvokeFunction` are added to the unqualified function.
     * - function.currentVersion is invoked before or after the permission is created.
     *
     * This applies only to permissions on Lambda functions, not versions or aliases.
     * This function is overridden as a noOp for QualifiedFunctionBase.
     */
    considerWarningOnInvokeFunctionPermissions(scope, action) {
        const affectedPermissions = ['lambda:InvokeFunction', 'lambda:*', 'lambda:Invoke*'];
        if (affectedPermissions.includes(action)) {
            if (scope.node.tryFindChild('CurrentVersion')) {
                this.warnInvokeFunctionPermissions(scope);
            }
            else {
                this._warnIfCurrentVersionCalled = true;
            }
        }
    }
    warnInvokeFunctionPermissions(scope) {
        core_1.Annotations.of(scope).addWarning([
            "AWS Lambda has changed their authorization strategy, which may cause client invocations using the 'Qualifier' parameter of the lambda function to fail with Access Denied errors.",
            "If you are using a lambda Version or Alias, make sure to call 'grantInvoke' or 'addPermission' on the Version or Alias, not the underlying Function",
            'See: https://github.com/aws/aws-cdk/issues/19273',
        ].join('\n'));
    }
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id for the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_Permission(permission);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPermission);
            }
            throw error;
        }
        if (!this.canCreatePermissions) {
            // FIXME: @deprecated(v2) - throw an error if calling `addPermission` on a resource that doesn't support it.
            return;
        }
        let principal = this.parsePermissionPrincipal(permission.principal);
        let { sourceArn, sourceAccount, principalOrgID } = this.validateConditionCombinations(permission.principal) ?? {};
        const action = permission.action ?? 'lambda:InvokeFunction';
        const scope = permission.scope ?? this;
        this.considerWarningOnInvokeFunctionPermissions(scope, action);
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount ?? sourceAccount,
            sourceArn: permission.sourceArn ?? sourceArn,
            principalOrgId: permission.organizationId ?? principalOrgID,
            functionUrlAuthType: permission.functionUrlAuthType,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToRolePolicy);
            }
            throw error;
        }
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_EventSourceMappingOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventSourceMapping);
            }
            throw error;
        }
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantInvoke);
            }
            throw error;
        }
        const hash = (0, crypto_1.createHash)('sha256')
            .update(JSON.stringify({
            principal: grantee.grantPrincipal.toString(),
            conditions: grantee.grantPrincipal.policyFragment.conditions,
        }), 'utf8')
            .digest('base64');
        const identifier = `Invoke${hash}`;
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = this.grant(grantee, identifier, 'lambda:InvokeFunction', this.resourceArnsForGrantInvoke);
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * Grant the given identity permissions to invoke this Lambda Function URL
     */
    grantInvokeUrl(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantInvokeUrl);
            }
            throw error;
        }
        const identifier = `InvokeFunctionUrl${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._functionUrlInvocationGrants[identifier];
        if (!grant) {
            grant = this.grant(grantee, identifier, 'lambda:InvokeFunctionUrl', [this.functionArn], {
                functionUrlAuthType: function_url_1.FunctionUrlAuthType.AWS_IAM,
            });
            this._functionUrlInvocationGrants[identifier] = grant;
        }
        return grant;
    }
    addEventSource(source) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_IEventSource(source);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventSource);
            }
            throw error;
        }
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_EventInvokeConfigOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureAsyncInvoke);
            }
            throw error;
        }
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    addFunctionUrl(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_FunctionUrlOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFunctionUrl);
            }
            throw error;
        }
        return new function_url_1.FunctionUrl(this, 'FunctionUrl', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    /**
     * Given the function arn, check if the account id matches this account
     *
     * Function ARNs look like this:
     *
     *   arn:aws:lambda:region:account-id:function:function-name
     *
     * ..which means that in order to extract the `account-id` component from the ARN, we can
     * split the ARN using ":" and select the component in index 4.
     *
     * @returns true if account id of function matches the account specified on the stack, false otherwise.
     *
     * @internal
     */
    _isStackAccount() {
        if (core_1.Token.isUnresolved(this.stack.account) || core_1.Token.isUnresolved(this.functionArn)) {
            return false;
        }
        return this.stack.splitArn(this.functionArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).account === this.stack.account;
    }
    grant(grantee, identifier, action, resourceArns, permissionOverrides) {
        const grant = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: [action],
            resourceArns,
            // Fake resource-like object on which to call addToResourcePolicy(), which actually
            // calls addPermission()
            resource: {
                addToResourcePolicy: (_statement) => {
                    // Couldn't add permissions to the principal, so add them locally.
                    this.addPermission(identifier, {
                        principal: grantee.grantPrincipal,
                        action: action,
                        ...permissionOverrides,
                    });
                    const permissionNode = this._functionNode().tryFindChild(identifier);
                    if (!permissionNode && !this._skipPermissions) {
                        throw new Error('Cannot modify permission to lambda function. Function is either imported or $LATEST version.\n'
                            + 'If the function is imported from the same account use `fromFunctionAttributes()` API with the `sameEnvironment` flag.\n'
                            + 'If the function is imported from a different account and already has the correct permissions use `fromFunctionAttributes()` API with the `skipPermissions` flag.');
                    }
                    return { statementAdded: true, policyDependable: permissionNode };
                },
                node: this.node,
                stack: this.stack,
                env: this.env,
                applyRemovalPolicy: this.applyRemovalPolicy,
            },
        });
        return grant;
    }
    /**
     * Translate IPrincipal to something we can pass to AWS::Lambda::Permissions
     *
     * Do some nasty things because `Permission` supports a subset of what the
     * full IAM principal language supports, and we may not be able to parse strings
     * outright because they may be tokens.
     *
     * Try to recognize some specific Principal classes first, then try a generic
     * fallback.
     */
    parsePermissionPrincipal(principal) {
        // Try some specific common classes first.
        // use duck-typing, not instance of
        if ('wrapped' in principal) {
            // eslint-disable-next-line dot-notation
            principal = principal['wrapped'];
        }
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        const stringEquals = matchSingleKey('StringEquals', principal.policyFragment.conditions);
        if (stringEquals) {
            const orgId = matchSingleKey('aws:PrincipalOrgID', stringEquals);
            if (orgId) {
                // we will move the organization id to the `principalOrgId` property of `Permissions`.
                return '*';
            }
        }
        // Try a best-effort approach to support simple principals that are not any of the predefined
        // classes, but are simple enough that they will fit into the Permission model. Main target
        // here: imported Roles, Users, Groups.
        //
        // The principal cannot have conditions and must have a single { AWS: [arn] } entry.
        const json = principal.policyFragment.principalJson;
        if (Object.keys(principal.policyFragment.conditions).length === 0 && json.AWS) {
            if (typeof json.AWS === 'string') {
                return json.AWS;
            }
            if (Array.isArray(json.AWS) && json.AWS.length === 1 && typeof json.AWS[0] === 'string') {
                return json.AWS[0];
            }
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal, OrganizationPrincipal');
        /**
         * Returns the value at the key if the object contains the key and nothing else. Otherwise,
         * returns undefined.
         */
        function matchSingleKey(key, obj) {
            if (Object.keys(obj).length !== 1) {
                return undefined;
            }
            return obj[key];
        }
    }
    validateConditionCombinations(principal) {
        const conditions = this.validateConditions(principal);
        if (!conditions) {
            return undefined;
        }
        const sourceArn = requireString(requireObject(conditions.ArnLike)?.['aws:SourceArn']);
        const sourceAccount = requireString(requireObject(conditions.StringEquals)?.['aws:SourceAccount']);
        const principalOrgID = requireString(requireObject(conditions.StringEquals)?.['aws:PrincipalOrgID']);
        // PrincipalOrgID cannot be combined with any other conditions
        if (principalOrgID && (sourceArn || sourceAccount)) {
            throw new Error('PrincipalWithConditions had unsupported condition combinations for Lambda permission statement: principalOrgID cannot be set with other conditions.');
        }
        return {
            sourceArn,
            sourceAccount,
            principalOrgID,
        };
    }
    validateConditions(principal) {
        if (this.isPrincipalWithConditions(principal)) {
            const conditions = principal.policyFragment.conditions;
            const conditionPairs = (0, util_1.flatMap)(Object.entries(conditions), ([operator, conditionObjs]) => Object.keys(conditionObjs).map(key => { return { operator, key }; }));
            // These are all the supported conditions. Some combinations are not supported,
            // like only 'aws:SourceArn' or 'aws:PrincipalOrgID' and 'aws:SourceAccount'.
            // These will be validated through `this.validateConditionCombinations`.
            const supportedPrincipalConditions = [{
                    operator: 'ArnLike',
                    key: 'aws:SourceArn',
                },
                {
                    operator: 'StringEquals',
                    key: 'aws:SourceAccount',
                }, {
                    operator: 'StringEquals',
                    key: 'aws:PrincipalOrgID',
                }];
            const unsupportedConditions = conditionPairs.filter((condition) => !supportedPrincipalConditions.some((supportedCondition) => supportedCondition.operator === condition.operator && supportedCondition.key === condition.key));
            if (unsupportedConditions.length == 0) {
                return conditions;
            }
            else {
                throw new Error(`PrincipalWithConditions had unsupported conditions for Lambda permission statement: ${JSON.stringify(unsupportedConditions)}. ` +
                    `Supported operator/condition pairs: ${JSON.stringify(supportedPrincipalConditions)}`);
            }
        }
        return undefined;
    }
    isPrincipalWithConditions(principal) {
        return Object.keys(principal.policyFragment.conditions).length > 0;
    }
}
_a = JSII_RTTI_SYMBOL_1;
FunctionBase[_a] = { fqn: "aws-cdk-lib.aws_lambda.FunctionBase", version: "2.74.0" };
exports.FunctionBase = FunctionBase;
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    get resourceArnsForGrantInvoke() {
        return [this.functionArn];
    }
    configureAsyncInvoke(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_lambda_EventInvokeConfigOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureAsyncInvoke);
            }
            throw error;
        }
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
    considerWarningOnInvokeFunctionPermissions(_scope, _action) {
        // noOp
        return;
    }
}
_b = JSII_RTTI_SYMBOL_1;
QualifiedFunctionBase[_b] = { fqn: "aws-cdk-lib.aws_lambda.QualifiedFunctionBase", version: "2.74.0" };
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = false;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get architecture() {
        return this.lambda.architecture;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
    get resourceArnsForGrantInvoke() {
        return [this.functionArn];
    }
    addAlias(aliasName, options = {}) {
        return (0, util_1.addAlias)(this, this, aliasName, options);
    }
}
function requireObject(x) {
    return x && typeof x === 'object' && !Array.isArray(x) ? x : undefined;
}
function requireString(x) {
    return x && typeof x === 'string' ? x : undefined;
}
//# sourceMappingURL=data:application/json;base64,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