"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("../../aws-certificatemanager");
const aws_cloudwatch_1 = require("../../aws-cloudwatch");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const route53 = require("../../aws-route53");
const secretsmanager = require("../../aws-secretsmanager");
const cdk = require("../../core");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const opensearch_access_policy_1 = require("./opensearch-access-policy");
const opensearchservice_generated_1 = require("./opensearchservice.generated");
const perms = require("./perms");
/**
 * The minimum TLS version required for traffic to the domain.
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    /** Cipher suite TLS 1.2 */
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this domain.
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensionsMap: {
                DomainName: this.domainName,
                ClientId: this.env.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Amazon OpenSearch Service domain.
 */
class Domain extends DomainBase {
    /**
     * Creates a domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_opensearchservice_DomainAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDomainAttributes);
            }
            throw error;
        }
        const { domainArn, domainEndpoint } = attrs;
        const domainName = cdk.Stack.of(scope).splitArn(domainArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName
            ?? extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainId = domainName;
                this.domainEndpoint = domainEndpoint.replace(/^https?:\/\//, '');
            }
        };
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.domainName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_opensearchservice_DomainProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Domain);
            }
            throw error;
        }
        const defaultInstanceType = 'r5.large.search';
        const warmDefaultInstanceType = 'ultrawarm1.medium.search';
        const dedicatedMasterType = initializeInstanceType(defaultInstanceType, props.capacity?.masterNodeInstanceType);
        const dedicatedMasterCount = props.capacity?.masterNodes ?? 0;
        const dedicatedMasterEnabled = cdk.Token.isUnresolved(dedicatedMasterCount) ? true : dedicatedMasterCount > 0;
        const instanceType = initializeInstanceType(defaultInstanceType, props.capacity?.dataNodeInstanceType);
        const instanceCount = props.capacity?.dataNodes ?? 1;
        const warmType = initializeInstanceType(warmDefaultInstanceType, props.capacity?.warmInstanceType);
        const warmCount = props.capacity?.warmNodes ?? 0;
        const warmEnabled = cdk.Token.isUnresolved(warmCount) ? true : warmCount > 0;
        const availabilityZoneCount = props.zoneAwareness?.availabilityZoneCount ?? 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = props.zoneAwareness?.enabled ??
            props.zoneAwareness?.availabilityZoneCount != null;
        let securityGroups;
        let subnets;
        let skipZoneAwarenessCheck = false;
        if (props.vpc) {
            const subnetSelections = props.vpcSubnets ?? [{ subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS }];
            subnets = selectSubnets(props.vpc, subnetSelections);
            skipZoneAwarenessCheck = zoneAwarenessCheckShouldBeSkipped(props.vpc, subnetSelections);
            securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            if (props.enforceHttps) {
                this._connections = new ec2.Connections({ securityGroups, defaultPort: ec2.Port.tcp(443) });
            }
            else {
                this._connections = new ec2.Connections({ securityGroups });
            }
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ (only if the vpc is not imported from another stack)
        if (subnets &&
            zoneAwarenessEnabled &&
            !skipZoneAwarenessCheck &&
            new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => (!cdk.Token.isUnresolved(t) && !t.endsWith('.search')))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".search".');
        }
        if (!cdk.Token.isUnresolved(warmType) && !warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const unsignedBasicAuthEnabled = props.useUnsignedBasicAuth ?? false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (props.encryptionAtRest?.enabled == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = props.fineGrainedAccessControl?.masterUserArn;
        const masterUserNameProps = props.fineGrainedAccessControl?.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps ?? 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn ?? masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = props.fineGrainedAccessControl?.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp ?? createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = props.encryptionAtRest?.enabled ?? (props.encryptionAtRest?.kmsKey != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = props.nodeToNodeEncryption ?? unsignedBasicAuthEnabled;
        const volumeSize = props.ebs?.volumeSize ?? 10;
        const volumeType = props.ebs?.volumeType ?? ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = props.ebs?.enabled ?? true;
        const enforceHttps = props.enforceHttps ?? unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryDatanodeInstanceType(...instanceTypes) {
            return instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch/OpenSearch version, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/features-by-version.html
        const { versionNum: versionNum, isElasticsearchVersion } = parseVersion(props.version);
        if (isElasticsearchVersion) {
            if (versionNum <= 7.7 &&
                ![
                    1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                    6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                    7.7,
                ].includes(versionNum)) {
                throw new Error(`Unknown Elasticsearch version: ${versionNum}`);
            }
            if (versionNum < 5.1) {
                if (props.logging?.appLogEnabled) {
                    throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.encryptionAtRest?.enabled) {
                    throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.cognitoDashboardsAuth != null) {
                    throw new Error('Cognito authentication for OpenSearch Dashboards requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                    throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.0) {
                if (props.nodeToNodeEncryption) {
                    throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.7) {
                if (unsignedBasicAuthEnabled) {
                    throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
                if (advancedSecurityEnabled) {
                    throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.8 && warmEnabled) {
                throw new Error('UltraWarm requires Elasticsearch version 6.8 or later or OpenSearch version 1.0 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && !(isElasticsearchVersion && versionNum <= 2.3)) {
            throw new Error('The t2.micro.search instance type supports only Elasticsearch versions 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/opensearch-service/pricing/
        if (!ebsEnabled && !isEveryDatanodeInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (props.logging?.auditLogEnabled && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if (props.logging?.slowSearchLogEnabled) {
            this.slowSearchLogGroup = props.logging.slowSearchLogGroup ??
                new logs.LogGroup(this, 'SlowSearchLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if (props.logging?.slowIndexLogEnabled) {
            this.slowIndexLogGroup = props.logging.slowIndexLogGroup ??
                new logs.LogGroup(this, 'SlowIndexLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if (props.logging?.appLogEnabled) {
            this.appLogGroup = props.logging.appLogGroup ??
                new logs.LogGroup(this, 'AppLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.appLogGroup);
        }
        ;
        if (props.logging?.auditLogEnabled) {
            this.auditLogGroup = props.logging.auditLogGroup ??
                new logs.LogGroup(this, 'AuditLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: this.auditLogGroup?.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new opensearchservice_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            engineVersion: props.version.version,
            clusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? props.ebs?.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? props.encryptionAtRest?.kmsKey?.keyId
                    : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: props.cognitoDashboardsAuth ? {
                enabled: true,
                identityPoolId: props.cognitoDashboardsAuth?.identityPoolId,
                roleArn: props.cognitoDashboardsAuth?.role.roleArn,
                userPoolId: props.cognitoDashboardsAuth?.userPoolId,
            } : undefined,
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: props.tlsSecurityPolicy ?? TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: this.masterUserPassword?.unsafeUnwrap(), // Safe usage
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainId = this.domain.getAtt('Id').toString();
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if (props.customEndpoint?.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        this.encryptionAtRestOptions = props.encryptionAtRest;
        if (props.accessPolicies) {
            this.addAccessPolicies(...props.accessPolicies);
        }
        if (unsignedBasicAuthEnabled) {
            this.addAccessPolicies(unsignedAccessPolicy);
        }
    }
    /**
     * Manages network connections to the domain. This will throw an error in case the domain
     * is not placed inside a VPC.
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
    /**
     * Add policy statements to the domain access policy
     */
    addAccessPolicies(...accessPolicyStatements) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(accessPolicyStatements);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAccessPolicies);
            }
            throw error;
        }
        if (accessPolicyStatements.length > 0) {
            if (!this.accessPolicy) {
                // Only create the custom resource after there are statements to set.
                this.accessPolicy = new opensearch_access_policy_1.OpenSearchAccessPolicy(this, 'AccessPolicy', {
                    domainName: this.domainName,
                    domainArn: this.domainArn,
                    accessPolicies: accessPolicyStatements,
                });
                if (this.encryptionAtRestOptions?.kmsKey) {
                    // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html
                    // these permissions are documented as required during domain creation.
                    // while not strictly documented for updates as well, it stands to reason that an update
                    // operation might require these in case the cluster uses a kms key.
                    // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                    this.accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                        actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                        resources: [this.encryptionAtRestOptions.kmsKey.keyArn],
                        effect: iam.Effect.ALLOW,
                    }));
                }
            }
            else {
                this.accessPolicy.addAccessPolicies(...accessPolicyStatements);
            }
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "aws-cdk-lib.aws_opensearchservice.Domain", version: "2.74.0" };
exports.Domain = Domain;
/**
 * Given an Amazon OpenSearch Service domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Amazon OpenSearch Service domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an engine version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The engine version object
 */
function parseVersion(version) {
    const elasticsearchPrefix = 'Elasticsearch_';
    const openSearchPrefix = 'OpenSearch_';
    const isElasticsearchVersion = version.version.startsWith(elasticsearchPrefix);
    const versionStr = isElasticsearchVersion
        ? version.version.substring(elasticsearchPrefix.length)
        : version.version.substring(openSearchPrefix.length);
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return { versionNum: parseFloat(versionStr), isElasticsearchVersion };
        }
        else {
            return { versionNum: parseFloat(versionStr.substring(0, secondDot)), isElasticsearchVersion };
        }
    }
    catch {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
/**
 * Check if any of the subnets are pending lookups. If so, the zone awareness check should be skipped, otherwise it will always throw an error
 *
 * @param vpc The vpc to which the subnets apply
 * @param vpcSubnets The vpc subnets that should be checked
 * @returns true if there are pending lookups for the subnets
 */
function zoneAwarenessCheckShouldBeSkipped(vpc, vpcSubnets) {
    for (const selection of vpcSubnets) {
        if (vpc.selectSubnets(selection).isPendingLookup) {
            return true;
        }
        ;
    }
    return false;
}
/**
 * Initializes an instance type.
 *
 * @param defaultInstanceType Default instance type which is used if no instance type is provided
 * @param instanceType Instance type
 * @returns Instance type in lowercase (if provided) or default instance type
 */
function initializeInstanceType(defaultInstanceType, instanceType) {
    if (instanceType) {
        return cdk.Token.isUnresolved(instanceType) ? instanceType : instanceType.toLowerCase();
    }
    else {
        return defaultInstanceType;
    }
}
//# sourceMappingURL=data:application/json;base64,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