"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const secretsmanager = require("../../aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        const parameterGroup = options.parameterGroup ?? this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
        this.combineImportAndExportRoles = props.combineImportAndExportRoles;
    }
    bindToCluster(scope, options) {
        const config = super.bindToCluster(scope, options);
        const parameterGroup = options.parameterGroup ?? (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            // versions which combine the import and export Roles (right now, this is only 8.0)
            // require a different parameter name (identical for both import and export)
            const s3ImportParam = this.combineImportAndExportRoles
                ? 'aws_default_s3_role'
                : 'aurora_load_from_s3_role';
            parameterGroup?.addParameter(s3ImportParam, options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            const s3ExportParam = this.combineImportAndExportRoles
                ? 'aws_default_s3_role'
                : 'aurora_select_into_s3_role';
            parameterGroup?.addParameter(s3ExportParam, options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by `DatabaseClusterEngine.aurora`).
 */
class AuroraEngineVersion {
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
}
_a = JSII_RTTI_SYMBOL_1;
AuroraEngineVersion[_a] = { fqn: "aws-cdk-lib.aws_rds.AuroraEngineVersion", version: "2.74.0" };
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
/** Version "5.6.mysql_aurora.1.22.3". */
AuroraEngineVersion.VER_1_22_3 = AuroraEngineVersion.builtIn_5_6('1.22.3');
/** Version "5.6.mysql_aurora.1.22.4". */
AuroraEngineVersion.VER_1_22_4 = AuroraEngineVersion.builtIn_5_6('1.22.4');
/** Version "5.6.mysql_aurora.1.22.5". */
AuroraEngineVersion.VER_1_22_5 = AuroraEngineVersion.builtIn_5_6('1.22.5');
/** Version "5.6.mysql_aurora.1.23.0". */
AuroraEngineVersion.VER_1_23_0 = AuroraEngineVersion.builtIn_5_6('1.23.0');
/** Version "5.6.mysql_aurora.1.23.1". */
AuroraEngineVersion.VER_1_23_1 = AuroraEngineVersion.builtIn_5_6('1.23.1');
/** Version "5.6.mysql_aurora.1.23.2". */
AuroraEngineVersion.VER_1_23_2 = AuroraEngineVersion.builtIn_5_6('1.23.2');
/** Version "5.6.mysql_aurora.1.23.3". */
AuroraEngineVersion.VER_1_23_3 = AuroraEngineVersion.builtIn_5_6('1.23.3');
/** Version "5.6.mysql_aurora.1.23.4". */
AuroraEngineVersion.VER_1_23_4 = AuroraEngineVersion.builtIn_5_6('1.23.4');
exports.AuroraEngineVersion = AuroraEngineVersion;
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by `DatabaseClusterEngine.auroraMysql`).
 *
 * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraMySQLReleaseNotes/Welcome.html
 */
class AuroraMysqlEngineVersion {
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
    static builtIn_8_0(minorVersion) {
        // 8.0 of the MySQL engine needs to combine the import and export Roles
        return new AuroraMysqlEngineVersion(`8.0.mysql_aurora.${minorVersion}`, '8.0', true);
    }
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7', combineImportAndExportRoles) {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
        this._combineImportAndExportRoles = combineImportAndExportRoles;
    }
}
_b = JSII_RTTI_SYMBOL_1;
AuroraMysqlEngineVersion[_b] = { fqn: "aws-cdk-lib.aws_rds.AuroraMysqlEngineVersion", version: "2.74.0" };
/**
 * Version "5.7.12".
 * @deprecated Version 5.7.12 is no longer supported by Amazon RDS.
 */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/**
 * Version "5.7.mysql_aurora.2.03.2".
 * @deprecated Version 5.7.mysql_aurora.2.03.2 is no longer supported by Amazon RDS.
 */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/**
 * Version "5.7.mysql_aurora.2.03.3".
 * @deprecated Version 5.7.mysql_aurora.2.03.3 is no longer supported by Amazon RDS.
 */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/**
 * Version "5.7.mysql_aurora.2.03.4".
 * @deprecated Version 5.7.mysql_aurora.2.03.4 is no longer supported by Amazon RDS.
 */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/**
 * Version "5.7.mysql_aurora.2.05.0".
 * @deprecated Version 5.7.mysql_aurora.2.05.0 is no longer supported by Amazon RDS.
 */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/**
 * Version "5.7.mysql_aurora.2.06.0".
 * @deprecated Version 5.7.mysql_aurora.2.06.0 is no longer supported by Amazon RDS.
 */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.07.3". */
AuroraMysqlEngineVersion.VER_2_07_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.3');
/** Version "5.7.mysql_aurora.2.07.4". */
AuroraMysqlEngineVersion.VER_2_07_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.4');
/** Version "5.7.mysql_aurora.2.07.5". */
AuroraMysqlEngineVersion.VER_2_07_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.5');
/** Version "5.7.mysql_aurora.2.07.6". */
AuroraMysqlEngineVersion.VER_2_07_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.6');
/** Version "5.7.mysql_aurora.2.07.7". */
AuroraMysqlEngineVersion.VER_2_07_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.7');
/** Version "5.7.mysql_aurora.2.07.8". */
AuroraMysqlEngineVersion.VER_2_07_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.8');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/** Version "5.7.mysql_aurora.2.08.2". */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/** Version "5.7.mysql_aurora.2.08.3". */
AuroraMysqlEngineVersion.VER_2_08_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.3');
/** Version "5.7.mysql_aurora.2.08.4". */
AuroraMysqlEngineVersion.VER_2_08_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.4');
/** Version "5.7.mysql_aurora.2.09.0". */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
/** Version "5.7.mysql_aurora.2.09.1". */
AuroraMysqlEngineVersion.VER_2_09_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.1');
/** Version "5.7.mysql_aurora.2.09.2". */
AuroraMysqlEngineVersion.VER_2_09_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.2');
/** Version "5.7.mysql_aurora.2.09.3". */
AuroraMysqlEngineVersion.VER_2_09_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.3');
/** Version "5.7.mysql_aurora.2.10.0". */
AuroraMysqlEngineVersion.VER_2_10_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.0');
/** Version "5.7.mysql_aurora.2.10.1". */
AuroraMysqlEngineVersion.VER_2_10_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.1');
/** Version "5.7.mysql_aurora.2.10.2". */
AuroraMysqlEngineVersion.VER_2_10_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.2');
/** Version "5.7.mysql_aurora.2.10.3". */
AuroraMysqlEngineVersion.VER_2_10_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.3');
/** Version "5.7.mysql_aurora.2.11.0". */
AuroraMysqlEngineVersion.VER_2_11_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.11.0');
/** Version "5.7.mysql_aurora.2.11.1". */
AuroraMysqlEngineVersion.VER_2_11_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.11.1');
/** Version "5.7.mysql_aurora.2.11.2". */
AuroraMysqlEngineVersion.VER_2_11_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.11.2');
/** Version "8.0.mysql_aurora.3.01.0". */
AuroraMysqlEngineVersion.VER_3_01_0 = AuroraMysqlEngineVersion.builtIn_8_0('3.01.0');
/** Version "8.0.mysql_aurora.3.01.1". */
AuroraMysqlEngineVersion.VER_3_01_1 = AuroraMysqlEngineVersion.builtIn_8_0('3.01.1');
/** Version "8.0.mysql_aurora.3.02.0". */
AuroraMysqlEngineVersion.VER_3_02_0 = AuroraMysqlEngineVersion.builtIn_8_0('3.02.0');
/** Version "8.0.mysql_aurora.3.02.1". */
AuroraMysqlEngineVersion.VER_3_02_1 = AuroraMysqlEngineVersion.builtIn_8_0('3.02.1');
/** Version "8.0.mysql_aurora.3.02.2". */
AuroraMysqlEngineVersion.VER_3_02_2 = AuroraMysqlEngineVersion.builtIn_8_0('3.02.2');
/** Version "8.0.mysql_aurora.3.03.0". */
AuroraMysqlEngineVersion.VER_3_03_0 = AuroraMysqlEngineVersion.builtIn_8_0('3.03.0');
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
            combineImportAndExportRoles: version?._combineImportAndExportRoles,
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by `DatabaseClusterEngine.auroraPostgres`).
 *
 * https://docs.aws.amazon.com/AmazonRDS/latest/AuroraPostgreSQLReleaseNotes/AuroraPostgreSQL.Updates.html
 */
class AuroraPostgresEngineVersion {
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_AuroraPostgresEngineFeatures(auroraPostgresFeatures);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: auroraPostgresFeatures?.s3Import ? 's3Import' : undefined,
            s3Export: auroraPostgresFeatures?.s3Export ? 's3Export' : undefined,
        };
    }
}
_c = JSII_RTTI_SYMBOL_1;
AuroraPostgresEngineVersion[_c] = { fqn: "aws-cdk-lib.aws_rds.AuroraPostgresEngineVersion", version: "2.74.0" };
/**
 * Version "9.6.8".
 * @deprecated Version 9.6.8 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/**
 * Version "9.6.9".
 * @deprecated Version 9.6.9 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/**
 * Version "9.6.11".
 * @deprecated Version 9.6.11 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/**
 * Version "9.6.12".
 * @deprecated Version 9.6.12 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/**
 * Version "9.6.16".
 * @deprecated Version 9.6.16 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/**
 * Version "9.6.17".
 * @deprecated Version 9.6.17 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/**
 * Version "9.6.18".
 * @deprecated Version 9.6.18 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_18 = AuroraPostgresEngineVersion.of('9.6.18', '9.6');
/**
 * Version "9.6.19".
 * @deprecated Version 9.6.19 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_9_6_19 = AuroraPostgresEngineVersion.of('9.6.19', '9.6');
/**
 *  Version "10.4".
 * @deprecated Version 10.4 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/**
 *  Version "10.5".
 * @deprecated Version 10.5 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/**
 *  Version "10.6".
 * @deprecated Version 10.6 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/**
 *  Version "10.7".
 * @deprecated Version 10.7 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 *  Version "10.11".
 * @deprecated Version 10.11 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/**
 *  Version "10.12".
 * @deprecated Version 10.12 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/**
 *  Version "10.13".
 * @deprecated Version 10.13 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_13 = AuroraPostgresEngineVersion.of('10.13', '10', { s3Import: true, s3Export: true });
/**
 *  Version "10.14".
 * @deprecated Version 10.14 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_10_14 = AuroraPostgresEngineVersion.of('10.14', '10', { s3Import: true, s3Export: true });
/** Version "10.16". */
AuroraPostgresEngineVersion.VER_10_16 = AuroraPostgresEngineVersion.of('10.16', '10', { s3Import: true, s3Export: true });
/** Version "10.17". */
AuroraPostgresEngineVersion.VER_10_17 = AuroraPostgresEngineVersion.of('10.17', '10', { s3Import: true, s3Export: true });
/** Version "10.18". */
AuroraPostgresEngineVersion.VER_10_18 = AuroraPostgresEngineVersion.of('10.18', '10', { s3Import: true, s3Export: true });
/** Version "10.19". */
AuroraPostgresEngineVersion.VER_10_19 = AuroraPostgresEngineVersion.of('10.19', '10', { s3Import: true, s3Export: true });
/** Version "10.20". */
AuroraPostgresEngineVersion.VER_10_20 = AuroraPostgresEngineVersion.of('10.20', '10', { s3Import: true, s3Export: true });
/** Version "10.21". */
AuroraPostgresEngineVersion.VER_10_21 = AuroraPostgresEngineVersion.of('10.21', '10', { s3Import: true, s3Export: true });
/**
 *  Version "11.4".
 * @deprecated Version 11.4 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 *  Version "11.6".
 * @deprecated Version 11.6 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/**
 *  Version "11.7".
 * @deprecated Version 11.7 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
/**
 *  Version "11.8".
 * @deprecated Version 11.8 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_11_8 = AuroraPostgresEngineVersion.of('11.8', '11', { s3Import: true, s3Export: true });
/** Version "11.9". */
AuroraPostgresEngineVersion.VER_11_9 = AuroraPostgresEngineVersion.of('11.9', '11', { s3Import: true, s3Export: true });
/** Version "11.11". */
AuroraPostgresEngineVersion.VER_11_11 = AuroraPostgresEngineVersion.of('11.11', '11', { s3Import: true, s3Export: true });
/** Version "11.12". */
AuroraPostgresEngineVersion.VER_11_12 = AuroraPostgresEngineVersion.of('11.12', '11', { s3Import: true, s3Export: true });
/** Version "11.13". */
AuroraPostgresEngineVersion.VER_11_13 = AuroraPostgresEngineVersion.of('11.13', '11', { s3Import: true, s3Export: true });
/** Version "11.14". */
AuroraPostgresEngineVersion.VER_11_14 = AuroraPostgresEngineVersion.of('11.14', '11', { s3Import: true, s3Export: true });
/** Version "11.15". */
AuroraPostgresEngineVersion.VER_11_15 = AuroraPostgresEngineVersion.of('11.15', '11', { s3Import: true, s3Export: true });
/** Version "11.16". */
AuroraPostgresEngineVersion.VER_11_16 = AuroraPostgresEngineVersion.of('11.16', '11', { s3Import: true, s3Export: true });
/** Version "11.17". */
AuroraPostgresEngineVersion.VER_11_17 = AuroraPostgresEngineVersion.of('11.17', '11', { s3Import: true, s3Export: true });
/** Version "11.18". */
AuroraPostgresEngineVersion.VER_11_18 = AuroraPostgresEngineVersion.of('11.18', '11', { s3Import: true, s3Export: true });
/** Version "11.19". */
AuroraPostgresEngineVersion.VER_11_19 = AuroraPostgresEngineVersion.of('11.19', '11', { s3Import: true, s3Export: true });
/** Version "12.4". */
AuroraPostgresEngineVersion.VER_12_4 = AuroraPostgresEngineVersion.of('12.4', '12', { s3Import: true, s3Export: true });
/** Version "12.6". */
AuroraPostgresEngineVersion.VER_12_6 = AuroraPostgresEngineVersion.of('12.6', '12', { s3Import: true, s3Export: true });
/** Version "12.7". */
AuroraPostgresEngineVersion.VER_12_7 = AuroraPostgresEngineVersion.of('12.7', '12', { s3Import: true, s3Export: true });
/** Version "12.8". */
AuroraPostgresEngineVersion.VER_12_8 = AuroraPostgresEngineVersion.of('12.8', '12', { s3Import: true, s3Export: true });
/** Version "12.9". */
AuroraPostgresEngineVersion.VER_12_9 = AuroraPostgresEngineVersion.of('12.9', '12', { s3Import: true, s3Export: true });
/** Version "12.10". */
AuroraPostgresEngineVersion.VER_12_10 = AuroraPostgresEngineVersion.of('12.10', '12', { s3Import: true, s3Export: true });
/** Version "12.11". */
AuroraPostgresEngineVersion.VER_12_11 = AuroraPostgresEngineVersion.of('12.11', '12', { s3Import: true, s3Export: true });
/** Version "12.12". */
AuroraPostgresEngineVersion.VER_12_12 = AuroraPostgresEngineVersion.of('12.12', '12', { s3Import: true, s3Export: true });
/** Version "12.13". */
AuroraPostgresEngineVersion.VER_12_13 = AuroraPostgresEngineVersion.of('12.13', '12', { s3Import: true, s3Export: true });
/** Version "12.14". */
AuroraPostgresEngineVersion.VER_12_14 = AuroraPostgresEngineVersion.of('12.14', '12', { s3Import: true, s3Export: true });
/** Version "13.3". */
AuroraPostgresEngineVersion.VER_13_3 = AuroraPostgresEngineVersion.of('13.3', '13', { s3Import: true, s3Export: true });
/** Version "13.4". */
AuroraPostgresEngineVersion.VER_13_4 = AuroraPostgresEngineVersion.of('13.4', '13', { s3Import: true, s3Export: true });
/** Version "13.5". */
AuroraPostgresEngineVersion.VER_13_5 = AuroraPostgresEngineVersion.of('13.5', '13', { s3Import: true, s3Export: true });
/** Version "13.6". */
AuroraPostgresEngineVersion.VER_13_6 = AuroraPostgresEngineVersion.of('13.6', '13', { s3Import: true, s3Export: true });
/** Version "13.7". */
AuroraPostgresEngineVersion.VER_13_7 = AuroraPostgresEngineVersion.of('13.7', '13', { s3Import: true, s3Export: true });
/** Version "13.8". */
AuroraPostgresEngineVersion.VER_13_8 = AuroraPostgresEngineVersion.of('13.8', '13', { s3Import: true, s3Export: true });
/** Version "13.9". */
AuroraPostgresEngineVersion.VER_13_9 = AuroraPostgresEngineVersion.of('13.9', '13', { s3Import: true, s3Export: true });
/** Version "13.10". */
AuroraPostgresEngineVersion.VER_13_10 = AuroraPostgresEngineVersion.of('13.10', '13', { s3Import: true, s3Export: true });
/** Version "14.3". */
AuroraPostgresEngineVersion.VER_14_3 = AuroraPostgresEngineVersion.of('14.3', '14', { s3Import: true, s3Export: true });
/**
 *  Version "14.4".
 * @deprecated Version 14.4 is no longer supported by Amazon RDS.
 */
AuroraPostgresEngineVersion.VER_14_4 = AuroraPostgresEngineVersion.of('14.4', '14', { s3Import: true, s3Export: true });
/** Version "14.5". */
AuroraPostgresEngineVersion.VER_14_5 = AuroraPostgresEngineVersion.of('14.5', '14', { s3Import: true, s3Export: true });
/** Version "14.6". */
AuroraPostgresEngineVersion.VER_14_6 = AuroraPostgresEngineVersion.of('14.6', '14', { s3Import: true, s3Export: true });
/** Version "14.7". */
AuroraPostgresEngineVersion.VER_14_7 = AuroraPostgresEngineVersion.of('14.7', '14', { s3Import: true, s3Export: true });
/** Version "15.2". */
AuroraPostgresEngineVersion.VER_15_2 = AuroraPostgresEngineVersion.of('15.2', '15', { s3Import: true, s3Export: true });
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if (this.engineVersion?.fullVersion) {
            if (options.s3ImportRole && !(config.features?.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !(config.features?.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_AuroraClusterEngineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.aurora);
            }
            throw error;
        }
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_AuroraMysqlClusterEngineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.auroraMysql);
            }
            throw error;
        }
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_AuroraPostgresClusterEngineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.auroraPostgres);
            }
            throw error;
        }
        return new AuroraPostgresClusterEngine(props.version);
    }
}
_d = JSII_RTTI_SYMBOL_1;
DatabaseClusterEngine[_d] = { fqn: "aws-cdk-lib.aws_rds.DatabaseClusterEngine", version: "2.74.0" };
/**
 * The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the `aurora()` method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the `auroraMysql()` method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *   as that can pose an availability risk.
 *   We recommend using versioned engines created using the `auroraPostgres()` method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
exports.DatabaseClusterEngine = DatabaseClusterEngine;
//# sourceMappingURL=data:application/json;base64,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