"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterFromSnapshot = exports.DatabaseCluster = exports.DatabaseClusterBase = exports.InstanceUpdateBehaviour = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const aws_iam_1 = require("../../aws-iam");
const logs = require("../../aws-logs");
const secretsmanager = require("../../aws-secretsmanager");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * The orchestration of updates of multiple instances
 */
var InstanceUpdateBehaviour;
(function (InstanceUpdateBehaviour) {
    /**
     * In a bulk update, all instances of the cluster are updated at the same time.
     * This results in a faster update procedure.
     * During the update, however, all instances might be unavailable at the same time and thus a downtime might occur.
     */
    InstanceUpdateBehaviour["BULK"] = "BULK";
    /**
     * In a rolling update, one instance after another is updated.
     * This results in at most one instance being unavailable during the update.
     * If your cluster consists of more than 1 instance, the downtime periods are limited to the time a primary switch needs.
     */
    InstanceUpdateBehaviour["ROLLING"] = "ROLLING";
})(InstanceUpdateBehaviour = exports.InstanceUpdateBehaviour || (exports.InstanceUpdateBehaviour = {}));
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseProxyOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addProxy);
            }
            throw error;
        }
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_a] = { fqn: "aws-cdk-lib.aws_rds.DatabaseClusterBase", version: "2.74.0" };
exports.DatabaseClusterBase = DatabaseClusterBase;
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
class DatabaseClusterNew extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = props.subnetGroup ?? new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.instanceProps.vpc,
            vpcSubnets: props.instanceProps.vpcSubnets,
            removalPolicy: (0, util_1.renderUnless)((0, util_1.helperRemovalPolicy)(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        this.securityGroups = props.instanceProps.securityGroups ?? [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        const combineRoles = props.engine.combineImportAndExportRoles ?? false;
        let { s3ImportRole, s3ExportRole } = (0, util_1.setupS3ImportExport)(this, props, combineRoles);
        if (props.parameterGroup && props.parameters) {
            throw new Error('You cannot specify both parameterGroup and parameters');
        }
        const parameterGroup = props.parameterGroup ?? (props.parameters
            ? new parameter_group_1.ParameterGroup(this, 'ParameterGroup', {
                engine: props.engine,
                parameters: props.parameters,
            })
            : undefined);
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup,
        });
        const clusterAssociatedRoles = [];
        if (s3ImportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: clusterEngineBindConfig.features?.s3Import });
        }
        if (s3ExportRole &&
            // only add the second associated Role if it's different than the first
            // (duplicates in the associated Roles array are not allowed by the RDS service)
            (s3ExportRole !== s3ImportRole ||
                clusterEngineBindConfig.features?.s3Import !== clusterEngineBindConfig.features?.s3Export)) {
            clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: clusterEngineBindConfig.features?.s3Export });
        }
        const clusterParameterGroup = props.parameterGroup ?? clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup?.bindToCluster({});
        this.engine = props.engine;
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER) && !core_1.Token.isUnresolved(props.clusterIdentifier)
            ? props.clusterIdentifier?.toLowerCase()
            : props.clusterIdentifier;
        this.newCfnProps = {
            // Basic
            engine: props.engine.engineType,
            engineVersion: props.engine.engineVersion?.fullVersion,
            dbClusterIdentifier: clusterIdentifier,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            port: props.port ?? clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig?.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            deletionProtection: (0, util_1.defaultDeletionProtection)(props.deletionProtection, props.removalPolicy),
            enableIamDatabaseAuthentication: props.iamAuthentication,
            networkType: props.networkType,
            // Admin
            backtrackWindow: props.backtrackWindow?.toSeconds(),
            backupRetentionPeriod: props.backup?.retention?.toDays(),
            preferredBackupWindow: props.backup?.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            enableCloudwatchLogsExports: props.cloudwatchLogsExports,
            // Encryption
            kmsKeyId: props.storageEncryptionKey?.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
            // Tags
            copyTagsToSnapshot: props.copyTagsToSnapshot ?? true,
        };
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     * See [Single user rotation strategy](https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets_strategies.html#rotating-secrets-one-user-one-password)
     */
    addRotationSingleUser(options = {}) {
        if (!this.secret) {
            throw new Error('Cannot add a single user rotation for a cluster without a secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...(0, util_1.applyDefaultRotationOptions)(options, this.vpcSubnets),
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     * See [Alternating users rotation strategy](https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets_strategies.html#rotating-secrets-two-users)
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add a multi user rotation for a cluster without a secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...(0, util_1.applyDefaultRotationOptions)(options, this.vpcSubnets),
            secret: options.secret,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
}
/**
 * Represents an imported database cluster.
 */
class ImportedDatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        this._clusterResourceIdentifier = attrs.clusterResourceIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.engine = attrs.engine;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
        this._instanceIdentifiers = attrs.instanceIdentifiers;
        this._instanceEndpoints = (attrs.instanceEndpointAddresses && attrs.port)
            ? attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port))
            : undefined;
    }
    get clusterResourceIdentifier() {
        if (!this._clusterResourceIdentifier) {
            throw new Error('Cannot access `clusterResourceIdentifier` of an imported cluster without a clusterResourceIdentifier');
        }
        return this._clusterResourceIdentifier;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
    get instanceIdentifiers() {
        if (!this._instanceIdentifiers) {
            throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
        }
        return this._instanceIdentifiers;
    }
    get instanceEndpoints() {
        if (!this._instanceEndpoints) {
            throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
        }
        return this._instanceEndpoints;
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterNew {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseClusterAttributes);
            }
            throw error;
        }
        return new ImportedDatabaseCluster(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseCluster);
            }
            throw error;
        }
        const credentials = (0, util_1.renderCredentials)(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            // Admin
            masterUsername: credentials.username,
            masterUserPassword: credentials.password?.unsafeUnwrap(),
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrDbClusterResourceId;
        if (secret) {
            this.secret = secret.attach(this);
        }
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
}
_b = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_b] = { fqn: "aws-cdk-lib.aws_rds.DatabaseCluster", version: "2.74.0" };
exports.DatabaseCluster = DatabaseCluster;
/**
 * A database cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseClusterFromSnapshotProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseClusterFromSnapshot);
            }
            throw error;
        }
        if (props.credentials && !props.credentials.password && !props.credentials.secret) {
            core_1.Annotations.of(this).addWarning('Use `snapshotCredentials` to modify password of a cluster created from a snapshot.');
        }
        if (!props.credentials && !props.snapshotCredentials) {
            core_1.Annotations.of(this).addWarning('Generated credentials will not be applied to cluster. Use `snapshotCredentials` instead. `addRotationSingleUser()` and `addRotationMultiUser()` cannot be used on this cluster.');
        }
        const deprecatedCredentials = (0, util_1.renderCredentials)(this, props.engine, props.credentials);
        let credentials = props.snapshotCredentials;
        let secret = credentials?.secret;
        if (!secret && credentials?.generatePassword) {
            if (!credentials.username) {
                throw new Error('`snapshotCredentials` `username` must be specified when `generatePassword` is set to true');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'SnapshotSecret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
                replaceOnPasswordCriteriaChanges: credentials.replaceOnPasswordCriteriaChanges,
                replicaRegions: credentials.replicaRegions,
            });
        }
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret?.secretValueFromJson('password')?.unsafeUnwrap() ?? credentials?.password?.unsafeUnwrap(), // Safe usage
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrDbClusterResourceId;
        if (secret) {
            this.secret = secret.attach(this);
        }
        if (deprecatedCredentials.secret) {
            const deprecatedSecret = deprecatedCredentials.secret.attach(this);
            if (!this.secret) {
                this.secret = deprecatedSecret;
            }
        }
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
}
_c = JSII_RTTI_SYMBOL_1;
DatabaseClusterFromSnapshot[_c] = { fqn: "aws-cdk-lib.aws_rds.DatabaseClusterFromSnapshot", version: "2.74.0" };
exports.DatabaseClusterFromSnapshot = DatabaseClusterFromSnapshot;
/**
 * Sets up CloudWatch log retention if configured.
 * A function rather than protected member to prevent exposing ``DatabaseClusterBaseProps``.
 */
function setLogRetention(cluster, props) {
    if (props.cloudwatchLogsExports) {
        const unsupportedLogTypes = props.cloudwatchLogsExports.filter(logType => !props.engine.supportedLogTypes.includes(logType));
        if (unsupportedLogTypes.length > 0) {
            throw new Error(`Unsupported logs for the current engine type: ${unsupportedLogTypes.join(',')}`);
        }
        if (props.cloudwatchLogsRetention) {
            for (const log of props.cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/cluster/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudwatchLogsRetention,
                    role: props.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * Creates the instances for the cluster.
 * A function rather than a protected method on ``DatabaseClusterNew`` to avoid exposing
 * ``DatabaseClusterNew`` and ``DatabaseClusterBaseProps`` in the API.
 */
function createInstances(cluster, props, subnetGroup) {
    const instanceCount = props.instances != null ? props.instances : 2;
    const instanceUpdateBehaviour = props.instanceUpdateBehaviour ?? InstanceUpdateBehaviour.BULK;
    if (core_1.Token.isUnresolved(instanceCount)) {
        throw new Error('The number of instances an RDS Cluster consists of cannot be provided as a deploy-time only value!');
    }
    if (instanceCount < 1) {
        throw new Error('At least one instance is required');
    }
    const instanceIdentifiers = [];
    const instanceEndpoints = [];
    const portAttribute = cluster.clusterEndpoint.port;
    const instanceProps = props.instanceProps;
    // Get the actual subnet objects so we can depend on internet connectivity.
    const internetConnected = instanceProps.vpc.selectSubnets(instanceProps.vpcSubnets).internetConnectivityEstablished;
    let monitoringRole;
    if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
        monitoringRole = props.monitoringRole || new aws_iam_1.Role(cluster, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
    }
    const enablePerformanceInsights = instanceProps.enablePerformanceInsights
        || instanceProps.performanceInsightRetention !== undefined || instanceProps.performanceInsightEncryptionKey !== undefined;
    if (enablePerformanceInsights && instanceProps.enablePerformanceInsights === false) {
        throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
    }
    const instanceType = instanceProps.instanceType ?? ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
    if (instanceProps.parameterGroup && instanceProps.parameters) {
        throw new Error('You cannot specify both parameterGroup and parameters');
    }
    const instanceParameterGroup = instanceProps.parameterGroup ?? (instanceProps.parameters
        ? new parameter_group_1.ParameterGroup(cluster, 'InstanceParameterGroup', {
            engine: props.engine,
            parameters: instanceProps.parameters,
        })
        : undefined);
    const instanceParameterGroupConfig = instanceParameterGroup?.bindToInstance({});
    const instances = [];
    for (let i = 0; i < instanceCount; i++) {
        const instanceIndex = i + 1;
        const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
            props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                undefined;
        const instance = new rds_generated_1.CfnDBInstance(cluster, `Instance${instanceIndex}`, {
            // Link to cluster
            engine: props.engine.engineType,
            dbClusterIdentifier: cluster.clusterIdentifier,
            dbInstanceIdentifier: instanceIdentifier,
            // Instance properties
            dbInstanceClass: databaseInstanceType(instanceType),
            publiclyAccessible: instanceProps.publiclyAccessible ??
                (instanceProps.vpcSubnets && instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC),
            enablePerformanceInsights: enablePerformanceInsights || instanceProps.enablePerformanceInsights,
            performanceInsightsKmsKeyId: instanceProps.performanceInsightEncryptionKey?.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (instanceProps.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            dbParameterGroupName: instanceParameterGroupConfig?.parameterGroupName,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            autoMinorVersionUpgrade: props.instanceProps.autoMinorVersionUpgrade,
            allowMajorVersionUpgrade: props.instanceProps.allowMajorVersionUpgrade,
            deleteAutomatedBackups: props.instanceProps.deleteAutomatedBackups,
        });
        // For instances that are part of a cluster:
        //
        //  Cluster DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
        //  Cluster RETAIN              -> RETAIN (otherwise cluster state will disappear)
        instance.applyRemovalPolicy((0, util_1.helperRemovalPolicy)(props.removalPolicy));
        // We must have a dependency on the NAT gateway provider here to create
        // things in the right order.
        instance.node.addDependency(internetConnected);
        instanceIdentifiers.push(instance.ref);
        instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
        instances.push(instance);
    }
    // Adding dependencies here to ensure that the instances are updated one after the other.
    if (instanceUpdateBehaviour === InstanceUpdateBehaviour.ROLLING) {
        for (let i = 1; i < instanceCount; i++) {
            instances[i].node.addDependency(instances[i - 1]);
        }
    }
    return { instanceEndpoints, instanceIdentifiers };
}
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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