"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.NetworkType = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const secretsmanager = require("../../aws-secretsmanager");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseInstanceAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseInstanceAttributes);
            }
            throw error;
        }
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = core_1.Tokenization.stringifyNumber(attrs.port);
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
                this.engine = attrs.engine;
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a new db proxy to this instance.
     */
    addProxy(id, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseProxyOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addProxy);
            }
            throw error;
        }
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    grantConnect(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantConnect);
            }
            throw error;
        }
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.instanceArn],
        });
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_events_OnEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onEvent);
            }
            throw error;
        }
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        const commonAnComponents = {
            service: 'rds',
            resource: 'db',
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        };
        const localArn = core_1.Stack.of(this).formatArn({
            ...commonAnComponents,
            resourceName: this.instanceIdentifier,
        });
        return this.getResourceArnAttribute(localArn, {
            ...commonAnComponents,
            resourceName: this.physicalName,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstanceBase[_a] = { fqn: "aws-cdk-lib.aws_rds.DatabaseInstanceBase", version: "2.74.0" };
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     *
     * Amazon RDS supports magnetic storage for backward compatibility. It is recommended to use
     * General Purpose SSD or Provisioned IOPS SSD for any new storage needs.
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#CHAP_Storage.Magnetic
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose SSD (gp2).
     *
     * Baseline performance determined by volume size
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#Concepts.Storage.GeneralSSD
     */
    StorageType["GP2"] = "gp2";
    /**
     * General purpose SSD (gp3).
     *
     * Performance scales independently from storage
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#Concepts.Storage.GeneralSSD
     */
    StorageType["GP3"] = "gp3";
    /**
     * Provisioned IOPS (SSD).
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The network type of the DB instance.
 */
var NetworkType;
(function (NetworkType) {
    /**
     * IPv4 only network type.
     */
    NetworkType["IPV4"] = "IPV4";
    /**
     * Dual-stack network type.
     */
    NetworkType["DUAL"] = "DUAL";
})(NetworkType = exports.NetworkType || (exports.NetworkType = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        // RDS always lower-cases the ID of the database, so use that for the physical name
        // (which is the name used for cross-environment access, so it needs to be correct,
        // regardless of the feature flag that changes it in the template for the L1)
        const instancePhysicalName = core_1.Token.isUnresolved(props.instanceIdentifier)
            ? props.instanceIdentifier
            : props.instanceIdentifier?.toLowerCase();
        super(scope, id, {
            physicalName: instancePhysicalName,
        });
        this.vpc = props.vpc;
        if (props.vpcSubnets && props.vpcPlacement) {
            throw new Error('Only one of `vpcSubnets` or `vpcPlacement` can be specified');
        }
        this.vpcPlacement = props.vpcSubnets ?? props.vpcPlacement;
        if (props.multiAz === true && props.availabilityZone) {
            throw new Error('Requesting a specific availability zone is not valid for Multi-AZ instances');
        }
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.SubnetGroup(this, 'SubnetGroup', {
            description: `Subnet group for ${this.node.id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            removalPolicy: (0, util_1.renderUnless)((0, util_1.helperRemovalPolicy)(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.number({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const storageType = props.storageType ?? StorageType.GP2;
        const iops = defaultIops(storageType, props.iops);
        if (props.storageThroughput && storageType !== StorageType.GP3) {
            throw new Error(`The storage throughput can only be specified with GP3 storage type. Got ${storageType}.`);
        }
        if (storageType === StorageType.GP3 && props.storageThroughput && iops
            && !core_1.Token.isUnresolved(props.storageThroughput) && !core_1.Token.isUnresolved(iops)
            && props.storageThroughput / iops > 0.25) {
            throw new Error(`The maximum ratio of storage throughput to IOPS is 0.25. Got ${props.storageThroughput / iops}.`);
        }
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.enableIamAuthentication = props.iamAuthentication;
        const enablePerformanceInsights = props.enablePerformanceInsights
            || props.performanceInsightRetention !== undefined || props.performanceInsightEncryptionKey !== undefined;
        if (enablePerformanceInsights && props.enablePerformanceInsights === false) {
            throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
        }
        if (props.domain) {
            this.domainId = props.domain;
            this.domainRole = props.domainRole || new iam.Role(this, 'RDSDirectoryServiceRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSDirectoryServiceAccess'),
                ],
            });
        }
        const maybeLowercasedInstanceId = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            && !core_1.Token.isUnresolved(props.instanceIdentifier)
            ? props.instanceIdentifier?.toLowerCase()
            : props.instanceIdentifier;
        const instanceParameterGroupConfig = props.parameterGroup?.bindToInstance({});
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention?.toDays(),
            copyTagsToSnapshot: props.copyTagsToSnapshot ?? true,
            dbInstanceClass: core_1.Lazy.string({ produce: () => `db.${this.instanceType}` }),
            dbInstanceIdentifier: core_1.Token.isUnresolved(props.instanceIdentifier)
                // if the passed identifier is a Token,
                // we need to use the physicalName of the database
                // (we cannot change its case anyway),
                // as it might be used in a cross-environment fashion
                ? this.physicalName
                : maybeLowercasedInstanceId,
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection: (0, util_1.defaultDeletionProtection)(props.deletionProtection, props.removalPolicy),
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: core_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            enablePerformanceInsights: enablePerformanceInsights || props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval?.toSeconds(),
            monitoringRoleArn: monitoringRole?.roleArn,
            multiAz: props.multiAz,
            dbParameterGroupName: instanceParameterGroupConfig?.parameterGroupName,
            optionGroupName: props.optionGroup?.optionGroupName,
            performanceInsightsKmsKeyId: props.performanceInsightEncryptionKey?.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (props.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port !== undefined ? core_1.Tokenization.stringifyNumber(props.port) : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.publiclyAccessible ?? (this.vpcPlacement && this.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC),
            storageType,
            storageThroughput: props.storageThroughput,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
            domain: this.domainId,
            domainIamRoleName: this.domainRole?.roleName,
            networkType: props.networkType,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new logs.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.engine = props.engine;
        const engineType = props.engine.engineType;
        // only Oracle and SQL Server require the import and export Roles to be the same
        const combineRoles = engineType.startsWith('oracle-') || engineType.startsWith('sqlserver-');
        let { s3ImportRole, s3ExportRole } = (0, util_1.setupS3ImportExport)(this, props, combineRoles);
        const engineConfig = props.engine.bindToInstance(this, {
            ...props,
            s3ImportRole,
            s3ExportRole,
        });
        const instanceAssociatedRoles = [];
        const engineFeatures = engineConfig.features;
        if (s3ImportRole) {
            if (!engineFeatures?.s3Import) {
                throw new Error(`Engine '${(0, util_1.engineDescription)(props.engine)}' does not support S3 import`);
            }
            instanceAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: engineFeatures?.s3Import });
        }
        if (s3ExportRole) {
            if (!engineFeatures?.s3Export) {
                throw new Error(`Engine '${(0, util_1.engineDescription)(props.engine)}' does not support S3 export`);
            }
            // only add the export feature if it's different from the import feature
            if (engineFeatures.s3Import !== engineFeatures?.s3Export) {
                instanceAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: engineFeatures?.s3Export });
            }
        }
        this.instanceType = props.instanceType ?? ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
        if (props.parameterGroup && props.parameters) {
            throw new Error('You cannot specify both parameterGroup and parameters');
        }
        const dbParameterGroupName = props.parameters
            ? new parameter_group_1.ParameterGroup(this, 'ParameterGroup', {
                engine: props.engine,
                parameters: props.parameters,
            }).bindToInstance({}).parameterGroupName
            : this.newCfnProps.dbParameterGroupName;
        this.sourceCfnProps = {
            ...this.newCfnProps,
            associatedRoles: instanceAssociatedRoles.length > 0 ? instanceAssociatedRoles : undefined,
            optionGroupName: engineConfig.optionGroup?.optionGroupName,
            allocatedStorage: props.allocatedStorage?.toString() ?? '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            engine: engineType,
            engineVersion: props.engine.engineVersion?.fullVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
            dbParameterGroupName,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param options the options for the rotation,
     *                if you want to override the defaults
     */
    addRotationSingleUser(options = {}) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...(0, util_1.applyDefaultRotationOptions)(options, this.vpcPlacement),
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...(0, util_1.applyDefaultRotationOptions)(options, this.vpcPlacement),
            secret: options.secret,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseInstanceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstance);
            }
            throw error;
        }
        const credentials = (0, util_1.renderCredentials)(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: credentials.password?.unsafeUnwrap(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = this.getResourceNameAttribute(instance.ref);
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
_b = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_b] = { fqn: "aws-cdk-lib.aws_rds.DatabaseInstance", version: "2.74.0" };
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseInstanceFromSnapshotProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstanceFromSnapshot);
            }
            throw error;
        }
        let credentials = props.credentials;
        let secret = credentials?.secret;
        if (!secret && credentials?.generatePassword) {
            if (!credentials.username) {
                throw new Error('`credentials` `username` must be specified when `generatePassword` is set to true');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
                replaceOnPasswordCriteriaChanges: credentials.replaceOnPasswordCriteriaChanges,
                replicaRegions: credentials.replicaRegions,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret?.secretValueFromJson('password')?.unsafeUnwrap() ?? credentials?.password?.unsafeUnwrap(), // Safe usage
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
_c = JSII_RTTI_SYMBOL_1;
DatabaseInstanceFromSnapshot[_c] = { fqn: "aws-cdk-lib.aws_rds.DatabaseInstanceFromSnapshot", version: "2.74.0" };
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.engine = undefined;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseInstanceReadReplicaProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstanceReadReplica);
            }
            throw error;
        }
        if (props.sourceDatabaseInstance.engine
            && !props.sourceDatabaseInstance.engine.supportsReadReplicaBackups
            && props.backupRetention) {
            throw new Error(`Cannot set 'backupRetention', as engine '${(0, util_1.engineDescription)(props.sourceDatabaseInstance.engine)}' does not support automatic backups for read replicas`);
        }
        // The read replica instance always uses the same engine as the source instance
        // but some CF validations require the engine to be explicitely passed when some
        // properties are specified.
        const shouldPassEngine = props.domain != null;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey?.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
            engine: shouldPassEngine ? props.sourceDatabaseInstance.engine?.engineType : undefined,
        });
        this.instanceType = props.instanceType;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        this.setLogRetention();
    }
}
_d = JSII_RTTI_SYMBOL_1;
DatabaseInstanceReadReplica[_d] = { fqn: "aws-cdk-lib.aws_rds.DatabaseInstanceReadReplica", version: "2.74.0" };
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function defaultIops(storageType, iops) {
    switch (storageType) {
        case StorageType.STANDARD:
        case StorageType.GP2:
            return undefined;
        case StorageType.GP3:
            return iops;
        case StorageType.IO1:
            return iops ?? 1000;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5zdGFuY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnN0YW5jZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxxQ0FBcUM7QUFDckMsMkNBQTJDO0FBQzNDLHFDQUFxQztBQUVyQyx1Q0FBdUM7QUFFdkMsMkRBQTJEO0FBQzNELHFDQUFvSjtBQUNwSixzQ0FBc0M7QUFFdEMsdURBQW1EO0FBQ25ELHlDQUFzQztBQUd0Qyx1REFBb0U7QUFDcEUseUNBQXNMO0FBQ3RMLG1DQUE2STtBQUM3SSxtQ0FBMkU7QUFDM0UsbURBQW9FO0FBQ3BFLGlEQUEyRDtBQTZGM0Q7O0dBRUc7QUFDSCxNQUFzQixvQkFBcUIsU0FBUSxlQUFRO0lBQ3pEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLDhCQUE4QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlDOzs7Ozs7Ozs7O1FBQzFHLE1BQU0sTUFBTyxTQUFRLG9CQUFvQjtZQUF6Qzs7Z0JBQ2tCLGdCQUFXLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUN2QyxnQkFBVyxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQztvQkFDaEQsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO29CQUNwQyxXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7aUJBQzlCLENBQUMsQ0FBQztnQkFDYSx1QkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7Z0JBQzlDLDhCQUF5QixHQUFHLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztnQkFDMUQsMkJBQXNCLEdBQUcsbUJBQVksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNsRSxxQkFBZ0IsR0FBRyxJQUFJLG1CQUFRLENBQUMsS0FBSyxDQUFDLHVCQUF1QixFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDM0UsV0FBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7Z0JBQzVCLDRCQUF1QixHQUFHLElBQUksQ0FBQztZQUMzQyxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5QjtJQWVEOztPQUVHO0lBQ0ksUUFBUSxDQUFDLEVBQVUsRUFBRSxPQUE2Qjs7Ozs7Ozs7OztRQUN2RCxPQUFPLElBQUkscUJBQWEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ2pDLFdBQVcsRUFBRSxtQkFBVyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7WUFDM0MsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO0tBQ0o7SUFFTSxZQUFZLENBQUMsT0FBdUI7Ozs7Ozs7Ozs7UUFDekMsSUFBSSxJQUFJLENBQUMsdUJBQXVCLEtBQUssS0FBSyxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELENBQUMsQ0FBQztTQUM3RTtRQUVELElBQUksQ0FBQyx1QkFBdUIsR0FBRyxJQUFJLENBQUM7UUFDcEMsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUM5QixPQUFPO1lBQ1AsT0FBTyxFQUFFLENBQUMsZ0JBQWdCLENBQUM7WUFDM0IsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQztTQUNqQyxDQUFDLENBQUM7S0FDSjtJQUVEOzs7T0FHRztJQUNJLE9BQU8sQ0FBQyxFQUFVLEVBQUUsVUFBaUMsRUFBRTs7Ozs7Ozs7OztRQUM1RCxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUNuQixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO1NBQzlCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFFRDs7T0FFRztJQUNILElBQVcsV0FBVztRQUNwQixNQUFNLGtCQUFrQixHQUFrQjtZQUN4QyxPQUFPLEVBQUUsS0FBSztZQUNkLFFBQVEsRUFBRSxJQUFJO1lBQ2QsU0FBUyxFQUFFLGdCQUFTLENBQUMsbUJBQW1CO1NBQ3pDLENBQUM7UUFDRixNQUFNLFFBQVEsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUN4QyxHQUFHLGtCQUFrQjtZQUNyQixZQUFZLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtTQUN0QyxDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxRQUFRLEVBQUU7WUFDNUMsR0FBRyxrQkFBa0I7WUFDckIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTztZQUNMLFFBQVEsRUFBRSxJQUFJLENBQUMsa0JBQWtCO1lBQ2pDLFVBQVUsRUFBRSxjQUFjLENBQUMsb0JBQW9CLENBQUMsZUFBZTtTQUNoRSxDQUFDO0tBQ0g7Ozs7QUFuR21CLG9EQUFvQjtBQXNHMUM7O0dBRUc7QUFDSCxJQUFZLFlBZVg7QUFmRCxXQUFZLFlBQVk7SUFDdEI7O09BRUc7SUFDSCxxREFBcUMsQ0FBQTtJQUVyQzs7T0FFRztJQUNILGlFQUFpRCxDQUFBO0lBRWpEOztPQUVHO0lBQ0gsaUVBQWlELENBQUE7QUFDbkQsQ0FBQyxFQWZXLFlBQVksR0FBWixvQkFBWSxLQUFaLG9CQUFZLFFBZXZCO0FBcUJEOzs7O0dBSUc7QUFDSCxJQUFZLFdBbUNYO0FBbkNELFdBQVksV0FBVztJQUNyQjs7Ozs7OztPQU9HO0lBQ0gsb0NBQXFCLENBQUE7SUFFckI7Ozs7OztPQU1HO0lBQ0gsMEJBQVcsQ0FBQTtJQUVYOzs7Ozs7T0FNRztJQUNILDBCQUFXLENBQUE7SUFFWDs7OztPQUlHO0lBQ0gsMEJBQVcsQ0FBQTtBQUNiLENBQUMsRUFuQ1csV0FBVyxHQUFYLG1CQUFXLEtBQVgsbUJBQVcsUUFtQ3RCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLFdBVVg7QUFWRCxXQUFZLFdBQVc7SUFDckI7O09BRUc7SUFDSCw0QkFBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCw0QkFBYSxDQUFBO0FBQ2YsQ0FBQyxFQVZXLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBVXRCO0FBa1hEOztHQUVHO0FBQ0gsTUFBZSxtQkFBb0IsU0FBUSxvQkFBb0I7SUFzQjdELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBK0I7UUFDdkUsbUZBQW1GO1FBQ25GLG1GQUFtRjtRQUNuRiw2RUFBNkU7UUFDN0UsTUFBTSxvQkFBb0IsR0FBRyxZQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztZQUN2RSxDQUFDLENBQUMsS0FBSyxDQUFDLGtCQUFrQjtZQUMxQixDQUFDLENBQUMsS0FBSyxDQUFDLGtCQUFrQixFQUFFLFdBQVcsRUFBRSxDQUFDO1FBQzVDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLG9CQUFvQjtTQUNuQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7UUFDckIsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO1NBQ2hGO1FBQ0QsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFFM0QsSUFBSSxLQUFLLENBQUMsT0FBTyxLQUFLLElBQUksSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RUFBNkUsQ0FBQyxDQUFDO1NBQ2hHO1FBRUQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxJQUFJLDBCQUFXLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUM1RSxXQUFXLEVBQUUsb0JBQW9CLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxXQUFXO1lBQ3hELEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLFVBQVUsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUM3QixhQUFhLEVBQUUsSUFBQSxtQkFBWSxFQUFDLElBQUEsMEJBQW1CLEVBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxFQUFFLG9CQUFhLENBQUMsT0FBTyxDQUFDO1NBQzdGLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDM0YsV0FBVyxFQUFFLHNCQUFzQixJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsV0FBVztnQkFDMUQsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2FBQ2YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQztZQUNyQyxjQUFjO1lBQ2QsV0FBVyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7U0FDdEYsQ0FBQyxDQUFDO1FBRUgsSUFBSSxjQUFjLENBQUM7UUFDbkIsSUFBSSxLQUFLLENBQUMsa0JBQWtCLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLFNBQVMsRUFBRSxFQUFFO1lBQ3BFLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7Z0JBQzVFLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyw4QkFBOEIsQ0FBQztnQkFDbkUsZUFBZSxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO2FBQzlHLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxXQUFXLENBQUMsR0FBRyxDQUFDO1FBQ3pELE1BQU0sSUFBSSxHQUFHLFdBQVcsQ0FBQyxXQUFXLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xELElBQUksS0FBSyxDQUFDLGlCQUFpQixJQUFJLFdBQVcsS0FBSyxXQUFXLENBQUMsR0FBRyxFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLFdBQVcsR0FBRyxDQUFDLENBQUM7U0FDNUc7UUFDRCxJQUFJLFdBQVcsS0FBSyxXQUFXLENBQUMsR0FBRyxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxJQUFJO2VBQy9ELENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDO2VBQ3pFLEtBQUssQ0FBQyxpQkFBaUIsR0FBQyxJQUFJLEdBQUcsSUFBSSxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLEtBQUssQ0FBQyxpQkFBaUIsR0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQ2xIO1FBRUQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztRQUN6RCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixDQUFDO1FBQzdELElBQUksQ0FBQywyQkFBMkIsR0FBRyxLQUFLLENBQUMsMkJBQTJCLENBQUM7UUFDckUsSUFBSSxDQUFDLHVCQUF1QixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUV2RCxNQUFNLHlCQUF5QixHQUFHLEtBQUssQ0FBQyx5QkFBeUI7ZUFDNUQsS0FBSyxDQUFDLDJCQUEyQixLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsK0JBQStCLEtBQUssU0FBUyxDQUFDO1FBQzVHLElBQUkseUJBQXlCLElBQUksS0FBSyxDQUFDLHlCQUF5QixLQUFLLEtBQUssRUFBRTtZQUMxRSxNQUFNLElBQUksS0FBSyxDQUFDLHNIQUFzSCxDQUFDLENBQUM7U0FDekk7UUFFRCxJQUFJLEtBQUssQ0FBQyxNQUFNLEVBQUU7WUFDaEIsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1lBQzdCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFO2dCQUNsRixTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7Z0JBQ3hELGVBQWUsRUFBRTtvQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDhDQUE4QyxDQUFDO2lCQUMzRjthQUNGLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSx5QkFBeUIsR0FBRyxtQkFBWSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLDJCQUEyQixDQUFDO2VBQ2pHLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUM7WUFDOUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxXQUFXLEVBQUU7WUFDekMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztRQUU3QixNQUFNLDRCQUE0QixHQUFHLEtBQUssQ0FBQyxjQUFjLEVBQUUsY0FBYyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQzlFLElBQUksQ0FBQyxXQUFXLEdBQUc7WUFDakIsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLHVCQUF1QjtZQUN0RCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0I7WUFDcEUscUJBQXFCLEVBQUUsS0FBSyxDQUFDLGVBQWUsRUFBRSxNQUFNLEVBQUU7WUFDdEQsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixJQUFJLElBQUk7WUFDcEQsZUFBZSxFQUFFLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsTUFBTSxJQUFJLENBQUMsWUFBWSxFQUFFLEVBQUUsQ0FBQztZQUMxRSxvQkFBb0IsRUFBRSxZQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztnQkFDaEUsdUNBQXVDO2dCQUN2QyxrREFBa0Q7Z0JBQ2xELHNDQUFzQztnQkFDdEMscURBQXFEO2dCQUNyRCxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVk7Z0JBQ25CLENBQUMsQ0FBQyx5QkFBeUI7WUFDN0IsaUJBQWlCLEVBQUUsV0FBVyxDQUFDLGVBQWU7WUFDOUMsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxrQkFBa0IsRUFBRSxJQUFBLGdDQUF5QixFQUFDLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDO1lBQzVGLDJCQUEyQixFQUFFLElBQUksQ0FBQyxxQkFBcUI7WUFDdkQsK0JBQStCLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsQ0FBQztZQUMxRix5QkFBeUIsRUFBRSx5QkFBeUIsSUFBSSxLQUFLLENBQUMseUJBQXlCO1lBQ3ZGLElBQUk7WUFDSixrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCLEVBQUUsU0FBUyxFQUFFO1lBQ3pELGlCQUFpQixFQUFFLGNBQWMsRUFBRSxPQUFPO1lBQzFDLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixvQkFBb0IsRUFBRSw0QkFBNEIsRUFBRSxrQkFBa0I7WUFDdEUsZUFBZSxFQUFFLEtBQUssQ0FBQyxXQUFXLEVBQUUsZUFBZTtZQUNuRCwyQkFBMkIsRUFBRSxLQUFLLENBQUMsK0JBQStCLEVBQUUsTUFBTTtZQUMxRSxrQ0FBa0MsRUFBRSx5QkFBeUI7Z0JBQzNELENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQywyQkFBMkIsSUFBSSxtQ0FBMkIsQ0FBQyxPQUFPLENBQUM7Z0JBQzVFLENBQUMsQ0FBQyxTQUFTO1lBQ2IsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxtQkFBWSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDckYscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtZQUNsRCwwQkFBMEIsRUFBRSxLQUFLLENBQUMsMEJBQTBCO1lBQzVELGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUM7WUFDOUYsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsS0FBSyxHQUFHLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM3SCxXQUFXO1lBQ1gsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxpQkFBaUIsRUFBRSxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztZQUM3RCxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzlDLE1BQU0sRUFBRSxJQUFJLENBQUMsUUFBUTtZQUNyQixpQkFBaUIsRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLFFBQVE7WUFDNUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQy9CLENBQUM7S0FDSDtJQUVTLGVBQWU7UUFDdkIsSUFBSSxJQUFJLENBQUMscUJBQXFCLElBQUksSUFBSSxDQUFDLHVCQUF1QixFQUFFO1lBQzlELEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxDQUFDLHFCQUFxQixFQUFFO2dCQUM1QyxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLGVBQWUsR0FBRyxFQUFFLEVBQUU7b0JBQ2hELFlBQVksRUFBRSxxQkFBcUIsSUFBSSxDQUFDLGtCQUFrQixJQUFJLEdBQUcsRUFBRTtvQkFDbkUsU0FBUyxFQUFFLElBQUksQ0FBQyx1QkFBdUI7b0JBQ3ZDLElBQUksRUFBRSxJQUFJLENBQUMsMkJBQTJCO2lCQUN2QyxDQUFDLENBQUM7YUFDSjtTQUNGO0tBQ0Y7Q0FDRjtBQWdFRDs7R0FFRztBQUNILE1BQWUsc0JBQXVCLFNBQVEsbUJBQW1CO0lBYS9ELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBa0M7UUFDMUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEIsSUFBSSxDQUFDLDZCQUE2QixHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsNkJBQTZCLENBQUM7UUFDaEYsSUFBSSxDQUFDLDRCQUE0QixHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsNEJBQTRCLENBQUM7UUFDOUUsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1FBRTNCLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDO1FBQzNDLGdGQUFnRjtRQUNoRixNQUFNLFlBQVksR0FBRyxVQUFVLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDN0YsSUFBSSxFQUFFLFlBQVksRUFBRSxZQUFZLEVBQUUsR0FBRyxJQUFBLDBCQUFtQixFQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDcEYsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFO1lBQ3JELEdBQUcsS0FBSztZQUNSLFlBQVk7WUFDWixZQUFZO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsTUFBTSx1QkFBdUIsR0FBMkMsRUFBRSxDQUFDO1FBQzNFLE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQyxRQUFRLENBQUM7UUFDN0MsSUFBSSxZQUFZLEVBQUU7WUFDaEIsSUFBSSxDQUFDLGNBQWMsRUFBRSxRQUFRLEVBQUU7Z0JBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFBLHdCQUFpQixFQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsOEJBQThCLENBQUMsQ0FBQzthQUMzRjtZQUNELHVCQUF1QixDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxZQUFZLENBQUMsT0FBTyxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztTQUN4RztRQUNELElBQUksWUFBWSxFQUFFO1lBQ2hCLElBQUksQ0FBQyxjQUFjLEVBQUUsUUFBUSxFQUFFO2dCQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLFdBQVcsSUFBQSx3QkFBaUIsRUFBQyxLQUFLLENBQUMsTUFBTSxDQUFDLDhCQUE4QixDQUFDLENBQUM7YUFDM0Y7WUFDRCx3RUFBd0U7WUFDeEUsSUFBSSxjQUFjLENBQUMsUUFBUSxLQUFLLGNBQWMsRUFBRSxRQUFRLEVBQUU7Z0JBQ3hELHVCQUF1QixDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxZQUFZLENBQUMsT0FBTyxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQzthQUN4RztTQUNGO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFNUcsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO1NBQzFFO1FBRUQsTUFBTSxvQkFBb0IsR0FBRyxLQUFLLENBQUMsVUFBVTtZQUMzQyxDQUFDLENBQUMsSUFBSSxnQ0FBYyxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDM0MsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO2dCQUNwQixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7YUFDN0IsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxrQkFBa0I7WUFDeEMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsb0JBQW9CLENBQUM7UUFFMUMsSUFBSSxDQUFDLGNBQWMsR0FBRztZQUNwQixHQUFHLElBQUksQ0FBQyxXQUFXO1lBQ25CLGVBQWUsRUFBRSx1QkFBdUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN6RixlQUFlLEVBQUUsWUFBWSxDQUFDLFdBQVcsRUFBRSxlQUFlO1lBQzFELGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxRQUFRLEVBQUUsSUFBSSxLQUFLO1lBQzdELHdCQUF3QixFQUFFLEtBQUssQ0FBQyx3QkFBd0I7WUFDeEQsTUFBTSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQzFCLE1BQU0sRUFBRSxVQUFVO1lBQ2xCLGFBQWEsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxXQUFXO1lBQ3RELFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsb0JBQW9CO1NBQ3JCLENBQUM7S0FDSDtJQUVEOzs7OztPQUtHO0lBQ0kscUJBQXFCLENBQUMsVUFBcUMsRUFBRTtRQUNsRSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNoQixNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxDQUFDLENBQUM7U0FDcEY7UUFFRCxNQUFNLEVBQUUsR0FBRyxvQkFBb0IsQ0FBQztRQUNoQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM1QyxJQUFJLFFBQVEsRUFBRTtZQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztTQUMvRTtRQUVELE9BQU8sSUFBSSxjQUFjLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDakQsR0FBRyxJQUFBLGtDQUEyQixFQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDO1lBQzFELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtZQUNuQixXQUFXLEVBQUUsSUFBSSxDQUFDLDZCQUE2QjtZQUMvQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixNQUFNLEVBQUUsSUFBSTtTQUNiLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxvQkFBb0IsQ0FBQyxFQUFVLEVBQUUsT0FBaUM7UUFDdkUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQ25GO1FBRUQsT0FBTyxJQUFJLGNBQWMsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNqRCxHQUFHLElBQUEsa0NBQTJCLEVBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUM7WUFDMUQsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNO1lBQ3RCLFlBQVksRUFBRSxJQUFJLENBQUMsTUFBTTtZQUN6QixXQUFXLEVBQUUsSUFBSSxDQUFDLDRCQUE0QjtZQUM5QyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixNQUFNLEVBQUUsSUFBSTtTQUNiLENBQUMsQ0FBQztLQUNKO0NBQ0Y7QUFvQ0Q7Ozs7R0FJRztBQUNILE1BQWEsZ0JBQWlCLFNBQVEsc0JBQXNCO0lBTzFELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OzsrQ0FSZixnQkFBZ0I7Ozs7UUFVekIsTUFBTSxXQUFXLEdBQUcsSUFBQSx3QkFBaUIsRUFBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDN0UsTUFBTSxNQUFNLEdBQUcsV0FBVyxDQUFDLE1BQU0sQ0FBQztRQUVsQyxNQUFNLFFBQVEsR0FBRyxJQUFJLDZCQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNuRCxHQUFHLElBQUksQ0FBQyxjQUFjO1lBQ3RCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsUUFBUSxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsTUFBTTtZQUN6RSxjQUFjLEVBQUUsV0FBVyxDQUFDLFFBQVE7WUFDcEMsa0JBQWtCLEVBQUUsV0FBVyxDQUFDLFFBQVEsRUFBRSxZQUFZLEVBQUU7WUFDeEQsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0I7U0FDN0UsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdEUsSUFBSSxDQUFDLHlCQUF5QixHQUFHLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztRQUM5RCxJQUFJLENBQUMsc0JBQXNCLEdBQUcsUUFBUSxDQUFDLGdCQUFnQixDQUFDO1FBRXhELGlFQUFpRTtRQUNqRSxNQUFNLGFBQWEsR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2hFLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLG1CQUFRLENBQUMsUUFBUSxDQUFDLG1CQUFtQixFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRWxGLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxJQUFJLG9CQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFM0UsSUFBSSxNQUFNLEVBQUU7WUFDVixJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDbkM7UUFFRCxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7S0FDeEI7Ozs7QUFyQ1UsNENBQWdCO0FBOEQ3Qjs7OztHQUlHO0FBQ0gsTUFBYSw0QkFBNkIsU0FBUSxzQkFBc0I7SUFPdEUsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3QztRQUNoRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQzs7Ozs7OytDQVJmLDRCQUE0Qjs7OztRQVVyQyxJQUFJLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3BDLElBQUksTUFBTSxHQUFHLFdBQVcsRUFBRSxNQUFNLENBQUM7UUFDakMsSUFBSSxDQUFDLE1BQU0sSUFBSSxXQUFXLEVBQUUsZ0JBQWdCLEVBQUU7WUFDNUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUU7Z0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQzthQUN0RztZQUVELE1BQU0sR0FBRyxJQUFJLGdDQUFjLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtnQkFDMUMsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFRO2dCQUM5QixhQUFhLEVBQUUsV0FBVyxDQUFDLGFBQWE7Z0JBQ3hDLGlCQUFpQixFQUFFLFdBQVcsQ0FBQyxpQkFBaUI7Z0JBQ2hELGdDQUFnQyxFQUFFLFdBQVcsQ0FBQyxnQ0FBZ0M7Z0JBQzlFLGNBQWMsRUFBRSxXQUFXLENBQUMsY0FBYzthQUMzQyxDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sUUFBUSxHQUFHLElBQUksNkJBQWEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ25ELEdBQUcsSUFBSSxDQUFDLGNBQWM7WUFDdEIsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUM5QyxrQkFBa0IsRUFBRSxNQUFNLEVBQUUsbUJBQW1CLENBQUMsVUFBVSxDQUFDLEVBQUUsWUFBWSxFQUFFLElBQUksV0FBVyxFQUFFLFFBQVEsRUFBRSxZQUFZLEVBQUUsRUFBRSxhQUFhO1NBQ3BJLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxrQkFBa0IsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyx5QkFBeUIsR0FBRyxRQUFRLENBQUMsbUJBQW1CLENBQUM7UUFDOUQsSUFBSSxDQUFDLHNCQUFzQixHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQztRQUV4RCxpRUFBaUU7UUFDakUsTUFBTSxhQUFhLEdBQUcsWUFBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxtQkFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUVsRixRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSxvQkFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTNFLElBQUksTUFBTSxFQUFFO1lBQ1YsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25DO1FBRUQsSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO0tBQ3hCOzs7O0FBL0NVLG9FQUE0QjtBQW1GekM7Ozs7R0FJRztBQUNILE1BQWEsMkJBQTRCLFNBQVEsbUJBQW1CO0lBUWxFLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUM7UUFDL0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFKVixXQUFNLEdBQXFCLFNBQVMsQ0FBQzs7Ozs7OytDQUwxQywyQkFBMkI7Ozs7UUFXcEMsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsTUFBTTtlQUNoQyxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsMEJBQTBCO2VBQy9ELEtBQUssQ0FBQyxlQUFlLEVBQUU7WUFDNUIsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsSUFBQSx3QkFBaUIsRUFBQyxLQUFLLENBQUMsc0JBQXNCLENBQUMsTUFBTSxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDN0s7UUFFRCwrRUFBK0U7UUFDL0UsZ0ZBQWdGO1FBQ2hGLDRCQUE0QjtRQUM1QixNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDO1FBRTlDLE1BQU0sUUFBUSxHQUFHLElBQUksNkJBQWEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ25ELEdBQUcsSUFBSSxDQUFDLFdBQVc7WUFDbkIsdUlBQXVJO1lBQ3ZJLDBCQUEwQixFQUFFLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXO1lBQ3BFLFFBQVEsRUFBRSxLQUFLLENBQUMsb0JBQW9CLEVBQUUsTUFBTTtZQUM1QyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQjtZQUM1RSxNQUFNLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ3ZGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUN2QyxJQUFJLENBQUMseUJBQXlCLEdBQUcsUUFBUSxDQUFDLG1CQUFtQixDQUFDO1FBQzlELElBQUksQ0FBQyxzQkFBc0IsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUM7UUFFeEQsaUVBQWlFO1FBQ2pFLE1BQU0sYUFBYSxHQUFHLFlBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDaEUsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksbUJBQVEsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFFbEYsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxhQUFhLElBQUksb0JBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUUzRSxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7S0FDeEI7Ozs7QUEzQ1Usa0VBQTJCO0FBOEN4Qzs7OztHQUlHO0FBQ0gsU0FBUyx1QkFBdUIsQ0FBQyxRQUEyQjtJQUMxRCxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFMUcsT0FBTyxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUM7QUFDOUQsQ0FBQztBQUVELFNBQVMsV0FBVyxDQUFDLFdBQXdCLEVBQUUsSUFBYTtJQUMxRCxRQUFRLFdBQVcsRUFBRTtRQUNuQixLQUFLLFdBQVcsQ0FBQyxRQUFRLENBQUM7UUFDMUIsS0FBSyxXQUFXLENBQUMsR0FBRztZQUNsQixPQUFPLFNBQVMsQ0FBQztRQUNuQixLQUFLLFdBQVcsQ0FBQyxHQUFHO1lBQ2xCLE9BQU8sSUFBSSxDQUFDO1FBQ2QsS0FBSyxXQUFXLENBQUMsR0FBRztZQUNsQixPQUFPLElBQUksSUFBSSxJQUFJLENBQUM7S0FDdkI7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gJy4uLy4uL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJy4uLy4uL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJy4uLy4uL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICcuLi8uLi9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICcuLi8uLi9hd3MtczMnO1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSAnLi4vLi4vYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCB7IEFybkNvbXBvbmVudHMsIEFybkZvcm1hdCwgRHVyYXRpb24sIEZlYXR1cmVGbGFncywgSVJlc291cmNlLCBMYXp5LCBSZW1vdmFsUG9saWN5LCBSZXNvdXJjZSwgU3RhY2ssIFRva2VuLCBUb2tlbml6YXRpb24gfSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCAqIGFzIGN4YXBpIGZyb20gJy4uLy4uL2N4LWFwaSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IERhdGFiYXNlU2VjcmV0IH0gZnJvbSAnLi9kYXRhYmFzZS1zZWNyZXQnO1xuaW1wb3J0IHsgRW5kcG9pbnQgfSBmcm9tICcuL2VuZHBvaW50JztcbmltcG9ydCB7IElJbnN0YW5jZUVuZ2luZSB9IGZyb20gJy4vaW5zdGFuY2UtZW5naW5lJztcbmltcG9ydCB7IElPcHRpb25Hcm91cCB9IGZyb20gJy4vb3B0aW9uLWdyb3VwJztcbmltcG9ydCB7IElQYXJhbWV0ZXJHcm91cCwgUGFyYW1ldGVyR3JvdXAgfSBmcm9tICcuL3BhcmFtZXRlci1ncm91cCc7XG5pbXBvcnQgeyBhcHBseURlZmF1bHRSb3RhdGlvbk9wdGlvbnMsIGRlZmF1bHREZWxldGlvblByb3RlY3Rpb24sIGVuZ2luZURlc2NyaXB0aW9uLCByZW5kZXJDcmVkZW50aWFscywgc2V0dXBTM0ltcG9ydEV4cG9ydCwgaGVscGVyUmVtb3ZhbFBvbGljeSwgcmVuZGVyVW5sZXNzIH0gZnJvbSAnLi9wcml2YXRlL3V0aWwnO1xuaW1wb3J0IHsgQ3JlZGVudGlhbHMsIFBlcmZvcm1hbmNlSW5zaWdodFJldGVudGlvbiwgUm90YXRpb25NdWx0aVVzZXJPcHRpb25zLCBSb3RhdGlvblNpbmdsZVVzZXJPcHRpb25zLCBTbmFwc2hvdENyZWRlbnRpYWxzIH0gZnJvbSAnLi9wcm9wcyc7XG5pbXBvcnQgeyBEYXRhYmFzZVByb3h5LCBEYXRhYmFzZVByb3h5T3B0aW9ucywgUHJveHlUYXJnZXQgfSBmcm9tICcuL3Byb3h5JztcbmltcG9ydCB7IENmbkRCSW5zdGFuY2UsIENmbkRCSW5zdGFuY2VQcm9wcyB9IGZyb20gJy4vcmRzLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBJU3VibmV0R3JvdXAsIFN1Ym5ldEdyb3VwIH0gZnJvbSAnLi9zdWJuZXQtZ3JvdXAnO1xuXG4vKipcbiAqIEEgZGF0YWJhc2UgaW5zdGFuY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJRGF0YWJhc2VJbnN0YW5jZSBleHRlbmRzIElSZXNvdXJjZSwgZWMyLklDb25uZWN0YWJsZSwgc2VjcmV0c21hbmFnZXIuSVNlY3JldEF0dGFjaG1lbnRUYXJnZXQge1xuICAvKipcbiAgICogVGhlIGluc3RhbmNlIGlkZW50aWZpZXIuXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZUlkZW50aWZpZXI6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGluc3RhbmNlIGFybi5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSBlbmRwb2ludCBhZGRyZXNzLlxuICAgKlxuICAgKiBAYXR0cmlidXRlIEVuZHBvaW50QWRkcmVzc1xuICAgKi9cbiAgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50QWRkcmVzczogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgaW5zdGFuY2UgZW5kcG9pbnQgcG9ydC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZSBFbmRwb2ludFBvcnRcbiAgICovXG4gIHJlYWRvbmx5IGRiSW5zdGFuY2VFbmRwb2ludFBvcnQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGluc3RhbmNlIGVuZHBvaW50LlxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VFbmRwb2ludDogRW5kcG9pbnQ7XG5cbiAgLyoqXG4gICAqIFRoZSBlbmdpbmUgb2YgdGhpcyBkYXRhYmFzZSBJbnN0YW5jZS5cbiAgICogTWF5IGJlIG5vdCBrbm93biBmb3IgaW1wb3J0ZWQgSW5zdGFuY2VzIGlmIGl0IHdhc24ndCBwcm92aWRlZCBleHBsaWNpdGx5LFxuICAgKiBvciBmb3IgcmVhZCByZXBsaWNhcy5cbiAgICovXG4gIHJlYWRvbmx5IGVuZ2luZT86IElJbnN0YW5jZUVuZ2luZTtcblxuICAvKipcbiAgICogQWRkIGEgbmV3IGRiIHByb3h5IHRvIHRoaXMgaW5zdGFuY2UuXG4gICAqL1xuICBhZGRQcm94eShpZDogc3RyaW5nLCBvcHRpb25zOiBEYXRhYmFzZVByb3h5T3B0aW9ucyk6IERhdGFiYXNlUHJveHk7XG5cbiAgLyoqXG4gICAqIEdyYW50IHRoZSBnaXZlbiBpZGVudGl0eSBjb25uZWN0aW9uIGFjY2VzcyB0byB0aGUgZGF0YWJhc2UuXG4gICAqICoqTm90ZSoqOiB0aGlzIG1ldGhvZCBkb2VzIG5vdCBjdXJyZW50bHkgd29yaywgc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvMTE4NTEgZm9yIGRldGFpbHMuXG4gICAqIEBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy8xMTg1MVxuICAgKi9cbiAgZ3JhbnRDb25uZWN0KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGEgQ2xvdWRXYXRjaCBldmVudCBydWxlIHdoaWNoIHRyaWdnZXJzIGZvciBpbnN0YW5jZSBldmVudHMuIFVzZVxuICAgKiBgcnVsZS5hZGRFdmVudFBhdHRlcm4ocGF0dGVybilgIHRvIHNwZWNpZnkgYSBmaWx0ZXIuXG4gICAqL1xuICBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBldmVudHMuT25FdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRoYXQgZGVzY3JpYmUgYW4gZXhpc3RpbmcgaW5zdGFuY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZUluc3RhbmNlQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBUaGUgaW5zdGFuY2UgaWRlbnRpZmllci5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlSWRlbnRpZmllcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZW5kcG9pbnQgYWRkcmVzcy5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlRW5kcG9pbnRBZGRyZXNzOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkYXRhYmFzZSBwb3J0LlxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgc2VjdXJpdHkgZ3JvdXBzIG9mIHRoZSBpbnN0YW5jZS5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzOiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAvKipcbiAgICogVGhlIGVuZ2luZSBvZiB0aGUgZXhpc3RpbmcgZGF0YWJhc2UgSW5zdGFuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGltcG9ydGVkIEluc3RhbmNlJ3MgZW5naW5lIGlzIHVua25vd25cbiAgICovXG4gIHJlYWRvbmx5IGVuZ2luZT86IElJbnN0YW5jZUVuZ2luZTtcbn1cblxuLyoqXG4gKiBBIG5ldyBvciBpbXBvcnRlZCBkYXRhYmFzZSBpbnN0YW5jZS5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIERhdGFiYXNlSW5zdGFuY2VCYXNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRGF0YWJhc2VJbnN0YW5jZSB7XG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgZGF0YWJhc2UgaW5zdGFuY2UuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21EYXRhYmFzZUluc3RhbmNlQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogRGF0YWJhc2VJbnN0YW5jZUF0dHJpYnV0ZXMpOiBJRGF0YWJhc2VJbnN0YW5jZSB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgRGF0YWJhc2VJbnN0YW5jZUJhc2UgaW1wbGVtZW50cyBJRGF0YWJhc2VJbnN0YW5jZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdFBvcnQgPSBlYzIuUG9ydC50Y3AoYXR0cnMucG9ydCk7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnMgPSBuZXcgZWMyLkNvbm5lY3Rpb25zKHtcbiAgICAgICAgc2VjdXJpdHlHcm91cHM6IGF0dHJzLnNlY3VyaXR5R3JvdXBzLFxuICAgICAgICBkZWZhdWx0UG9ydDogdGhpcy5kZWZhdWx0UG9ydCxcbiAgICAgIH0pO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlSWRlbnRpZmllciA9IGF0dHJzLmluc3RhbmNlSWRlbnRpZmllcjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBkYkluc3RhbmNlRW5kcG9pbnRBZGRyZXNzID0gYXR0cnMuaW5zdGFuY2VFbmRwb2ludEFkZHJlc3M7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50UG9ydCA9IFRva2VuaXphdGlvbi5zdHJpbmdpZnlOdW1iZXIoYXR0cnMucG9ydCk7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VFbmRwb2ludCA9IG5ldyBFbmRwb2ludChhdHRycy5pbnN0YW5jZUVuZHBvaW50QWRkcmVzcywgYXR0cnMucG9ydCk7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZW5naW5lID0gYXR0cnMuZW5naW5lO1xuICAgICAgcHJvdGVjdGVkIGVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uID0gdHJ1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGluc3RhbmNlSWRlbnRpZmllcjogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50QWRkcmVzczogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50UG9ydDogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgaW5zdGFuY2VFbmRwb2ludDogRW5kcG9pbnQ7XG4gIC8vIG9ubHkgcmVxdWlyZWQgYmVjYXVzZSBvZiBKU0lJIGJ1ZzogaHR0cHM6Ly9naXRodWIuY29tL2F3cy9qc2lpL2lzc3Vlcy8yMDQwXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBlbmdpbmU/OiBJSW5zdGFuY2VFbmdpbmU7XG4gIHByb3RlY3RlZCBhYnN0cmFjdCBlbmFibGVJYW1BdXRoZW50aWNhdGlvbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEFjY2VzcyB0byBuZXR3b3JrIGNvbm5lY3Rpb25zLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIEFkZCBhIG5ldyBkYiBwcm94eSB0byB0aGlzIGluc3RhbmNlLlxuICAgKi9cbiAgcHVibGljIGFkZFByb3h5KGlkOiBzdHJpbmcsIG9wdGlvbnM6IERhdGFiYXNlUHJveHlPcHRpb25zKTogRGF0YWJhc2VQcm94eSB7XG4gICAgcmV0dXJuIG5ldyBEYXRhYmFzZVByb3h5KHRoaXMsIGlkLCB7XG4gICAgICBwcm94eVRhcmdldDogUHJveHlUYXJnZXQuZnJvbUluc3RhbmNlKHRoaXMpLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBncmFudENvbm5lY3QoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIGlmICh0aGlzLmVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uID09PSBmYWxzZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZ3JhbnQgY29ubmVjdCB3aGVuIElBTSBhdXRoZW50aWNhdGlvbiBpcyBkaXNhYmxlZCcpO1xuICAgIH1cblxuICAgIHRoaXMuZW5hYmxlSWFtQXV0aGVudGljYXRpb24gPSB0cnVlO1xuICAgIHJldHVybiBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnM6IFsncmRzLWRiOmNvbm5lY3QnXSxcbiAgICAgIHJlc291cmNlQXJuczogW3RoaXMuaW5zdGFuY2VBcm5dLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYSBDbG91ZFdhdGNoIGV2ZW50IHJ1bGUgd2hpY2ggdHJpZ2dlcnMgZm9yIGluc3RhbmNlIGV2ZW50cy4gVXNlXG4gICAqIGBydWxlLmFkZEV2ZW50UGF0dGVybihwYXR0ZXJuKWAgdG8gc3BlY2lmeSBhIGZpbHRlci5cbiAgICovXG4gIHB1YmxpYyBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyA9IHt9KSB7XG4gICAgY29uc3QgcnVsZSA9IG5ldyBldmVudHMuUnVsZSh0aGlzLCBpZCwgb3B0aW9ucyk7XG4gICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgc291cmNlOiBbJ2F3cy5yZHMnXSxcbiAgICAgIHJlc291cmNlczogW3RoaXMuaW5zdGFuY2VBcm5dLFxuICAgIH0pO1xuICAgIHJ1bGUuYWRkVGFyZ2V0KG9wdGlvbnMudGFyZ2V0KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgaW5zdGFuY2UgYXJuLlxuICAgKi9cbiAgcHVibGljIGdldCBpbnN0YW5jZUFybigpOiBzdHJpbmcge1xuICAgIGNvbnN0IGNvbW1vbkFuQ29tcG9uZW50czogQXJuQ29tcG9uZW50cyA9IHtcbiAgICAgIHNlcnZpY2U6ICdyZHMnLFxuICAgICAgcmVzb3VyY2U6ICdkYicsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgIH07XG4gICAgY29uc3QgbG9jYWxBcm4gPSBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgLi4uY29tbW9uQW5Db21wb25lbnRzLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLmluc3RhbmNlSWRlbnRpZmllcixcbiAgICB9KTtcbiAgICByZXR1cm4gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShsb2NhbEFybiwge1xuICAgICAgLi4uY29tbW9uQW5Db21wb25lbnRzLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXJzIHRoZSBzZWNyZXQgYXR0YWNobWVudCB0YXJnZXQgc3BlY2lmaWNhdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgYXNTZWNyZXRBdHRhY2htZW50VGFyZ2V0KCk6IHNlY3JldHNtYW5hZ2VyLlNlY3JldEF0dGFjaG1lbnRUYXJnZXRQcm9wcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHRhcmdldElkOiB0aGlzLmluc3RhbmNlSWRlbnRpZmllcixcbiAgICAgIHRhcmdldFR5cGU6IHNlY3JldHNtYW5hZ2VyLkF0dGFjaG1lbnRUYXJnZXRUeXBlLlJEU19EQl9JTlNUQU5DRSxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogVGhlIGxpY2Vuc2UgbW9kZWwuXG4gKi9cbmV4cG9ydCBlbnVtIExpY2Vuc2VNb2RlbCB7XG4gIC8qKlxuICAgKiBMaWNlbnNlIGluY2x1ZGVkLlxuICAgKi9cbiAgTElDRU5TRV9JTkNMVURFRCA9ICdsaWNlbnNlLWluY2x1ZGVkJyxcblxuICAvKipcbiAgICogQnJpbmcgeW91ciBvd24gbGljZW5jc2UuXG4gICAqL1xuICBCUklOR19ZT1VSX09XTl9MSUNFTlNFID0gJ2JyaW5nLXlvdXItb3duLWxpY2Vuc2UnLFxuXG4gIC8qKlxuICAgKiBHZW5lcmFsIHB1YmxpYyBsaWNlbnNlLlxuICAgKi9cbiAgR0VORVJBTF9QVUJMSUNfTElDRU5TRSA9ICdnZW5lcmFsLXB1YmxpYy1saWNlbnNlJ1xufVxuXG4vKipcbiAqIFRoZSBwcm9jZXNzb3IgZmVhdHVyZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUHJvY2Vzc29yRmVhdHVyZXMge1xuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBDUFUgY29yZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgZGVmYXVsdCBudW1iZXIgb2YgQ1BVIGNvcmVzIGZvciB0aGUgY2hvc2VuIGluc3RhbmNlIGNsYXNzLlxuICAgKi9cbiAgcmVhZG9ubHkgY29yZUNvdW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHRocmVhZHMgcGVyIGNvcmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgbnVtYmVyIG9mIHRocmVhZHMgcGVyIGNvcmUgZm9yIHRoZSBjaG9zZW4gaW5zdGFuY2UgY2xhc3MuXG4gICAqL1xuICByZWFkb25seSB0aHJlYWRzUGVyQ29yZT86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBUaGUgdHlwZSBvZiBzdG9yYWdlLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL0NIQVBfU3RvcmFnZS5odG1sXG4gKi9cbmV4cG9ydCBlbnVtIFN0b3JhZ2VUeXBlIHtcbiAgLyoqXG4gICAqIFN0YW5kYXJkLlxuICAgKlxuICAgKiBBbWF6b24gUkRTIHN1cHBvcnRzIG1hZ25ldGljIHN0b3JhZ2UgZm9yIGJhY2t3YXJkIGNvbXBhdGliaWxpdHkuIEl0IGlzIHJlY29tbWVuZGVkIHRvIHVzZVxuICAgKiBHZW5lcmFsIFB1cnBvc2UgU1NEIG9yIFByb3Zpc2lvbmVkIElPUFMgU1NEIGZvciBhbnkgbmV3IHN0b3JhZ2UgbmVlZHMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL0NIQVBfU3RvcmFnZS5odG1sI0NIQVBfU3RvcmFnZS5NYWduZXRpY1xuICAgKi9cbiAgU1RBTkRBUkQgPSAnc3RhbmRhcmQnLFxuXG4gIC8qKlxuICAgKiBHZW5lcmFsIHB1cnBvc2UgU1NEIChncDIpLlxuICAgKlxuICAgKiBCYXNlbGluZSBwZXJmb3JtYW5jZSBkZXRlcm1pbmVkIGJ5IHZvbHVtZSBzaXplXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL0NIQVBfU3RvcmFnZS5odG1sI0NvbmNlcHRzLlN0b3JhZ2UuR2VuZXJhbFNTRFxuICAgKi9cbiAgR1AyID0gJ2dwMicsXG5cbiAgLyoqXG4gICAqIEdlbmVyYWwgcHVycG9zZSBTU0QgKGdwMykuXG4gICAqXG4gICAqIFBlcmZvcm1hbmNlIHNjYWxlcyBpbmRlcGVuZGVudGx5IGZyb20gc3RvcmFnZVxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9DSEFQX1N0b3JhZ2UuaHRtbCNDb25jZXB0cy5TdG9yYWdlLkdlbmVyYWxTU0RcbiAgICovXG4gIEdQMyA9ICdncDMnLFxuXG4gIC8qKlxuICAgKiBQcm92aXNpb25lZCBJT1BTIChTU0QpLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9DSEFQX1N0b3JhZ2UuaHRtbCNVU0VSX1BJT1BTXG4gICAqL1xuICBJTzEgPSAnaW8xJ1xufVxuXG4vKipcbiAqIFRoZSBuZXR3b3JrIHR5cGUgb2YgdGhlIERCIGluc3RhbmNlLlxuICovXG5leHBvcnQgZW51bSBOZXR3b3JrVHlwZSB7XG4gIC8qKlxuICAgKiBJUHY0IG9ubHkgbmV0d29yayB0eXBlLlxuICAgKi9cbiAgSVBWNCA9ICdJUFY0JyxcblxuICAvKipcbiAgICogRHVhbC1zdGFjayBuZXR3b3JrIHR5cGUuXG4gICAqL1xuICBEVUFMID0gJ0RVQUwnXG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgRGF0YWJhc2VJbnN0YW5jZU5ld1xuICovXG5leHBvcnQgaW50ZXJmYWNlIERhdGFiYXNlSW5zdGFuY2VOZXdQcm9wcyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgaWYgdGhlIGRhdGFiYXNlIGluc3RhbmNlIGlzIGEgbXVsdGlwbGUgQXZhaWxhYmlsaXR5IFpvbmUgZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG11bHRpQXo/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgQXZhaWxhYmlsaXR5IFpvbmUgd2hlcmUgdGhlIERCIGluc3RhbmNlIHdpbGwgYmUgbG9jYXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBwcmVmZXJlbmNlXG4gICAqL1xuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgc3RvcmFnZSB0eXBlLiBTdG9yYWdlIHR5cGVzIHN1cHBvcnRlZCBhcmUgZ3AyLCBpbzEsIHN0YW5kYXJkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9DSEFQX1N0b3JhZ2UuaHRtbCNDb25jZXB0cy5TdG9yYWdlLkdlbmVyYWxTU0RcbiAgICpcbiAgICogQGRlZmF1bHQgR1AyXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlVHlwZT86IFN0b3JhZ2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgc3RvcmFnZSB0aHJvdWdocHV0LCBzcGVjaWZpZWQgaW4gbWViaWJ5dGVzIHBlciBzZWNvbmQgKE1pQnBzKS5cbiAgICpcbiAgICogT25seSBhcHBsaWNhYmxlIGZvciBHUDMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tLy9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9DSEFQX1N0b3JhZ2UuaHRtbCNncDMtc3RvcmFnZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEyNSBNaUJwcyBpZiBhbGxvY2F0ZWQgc3RvcmFnZSBpcyBsZXNzIHRoYW4gNDAwIEdpQiBmb3IgTWFyaWFEQiwgTXlTUUwsIGFuZCBQb3N0Z3JlU1FMLFxuICAgKiBsZXNzIHRoYW4gMjAwIEdpQiBmb3IgT3JhY2xlIGFuZCBsZXNzIHRoYW4gMjAgR2lCIGZvciBTUUwgU2VydmVyLiA1MDAgTWlCcHMgb3RoZXJ3aXNlIChleGNlcHQgZm9yXG4gICAqIFNRTCBTZXJ2ZXIgd2hlcmUgdGhlIGRlZmF1bHQgaXMgYWx3YXlzIDEyNSBNaUJwcykuXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlVGhyb3VnaHB1dD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBJL08gb3BlcmF0aW9ucyBwZXIgc2Vjb25kIChJT1BTKSB0aGF0IHRoZSBkYXRhYmFzZSBwcm92aXNpb25zLlxuICAgKiBUaGUgdmFsdWUgbXVzdCBiZSBlcXVhbCB0byBvciBncmVhdGVyIHRoYW4gMTAwMC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBwcm92aXNpb25lZCBpb3BzIGlmIHN0b3JhZ2UgdHlwZSBpcyBub3Qgc3BlY2lmaWVkLiBGb3IgR1AzOiAzLDAwMCBJT1BTIGlmIGFsbG9jYXRlZFxuICAgKiBzdG9yYWdlIGlzIGxlc3MgdGhhbiA0MDAgR2lCIGZvciBNYXJpYURCLCBNeVNRTCwgYW5kIFBvc3RncmVTUUwsIGxlc3MgdGhhbiAyMDAgR2lCIGZvciBPcmFjbGUgYW5kXG4gICAqIGxlc3MgdGhhbiAyMCBHaUIgZm9yIFNRTCBTZXJ2ZXIuIDEyLDAwMCBJT1BTIG90aGVyd2lzZSAoZXhjZXB0IGZvciBTUUwgU2VydmVyIHdoZXJlIHRoZSBkZWZhdWx0IGlzXG4gICAqIGFsd2F5cyAzLDAwMCBJT1BTKS5cbiAgICovXG4gIHJlYWRvbmx5IGlvcHM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgQ1BVIGNvcmVzIGFuZCB0aGUgbnVtYmVyIG9mIHRocmVhZHMgcGVyIGNvcmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgbnVtYmVyIG9mIENQVSBjb3JlcyBhbmQgdGhyZWFkcyBwZXIgY29yZSBmb3IgdGhlXG4gICAqIGNob3NlbiBpbnN0YW5jZSBjbGFzcy5cbiAgICpcbiAgICogU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9Db25jZXB0cy5EQkluc3RhbmNlQ2xhc3MuaHRtbCNVU0VSX0NvbmZpZ3VyZVByb2Nlc3NvclxuICAgKi9cbiAgcmVhZG9ubHkgcHJvY2Vzc29yRmVhdHVyZXM/OiBQcm9jZXNzb3JGZWF0dXJlcztcblxuICAvKipcbiAgICogQSBuYW1lIGZvciB0aGUgREIgaW5zdGFuY2UuIElmIHlvdSBzcGVjaWZ5IGEgbmFtZSwgQVdTIENsb3VkRm9ybWF0aW9uXG4gICAqIGNvbnZlcnRzIGl0IHRvIGxvd2VyY2FzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIENsb3VkRm9ybWF0aW9uIGdlbmVyYXRlZCBuYW1lXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZUlkZW50aWZpZXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgbmV0d29yayB3aGVyZSB0aGUgREIgc3VibmV0IGdyb3VwIHNob3VsZCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2Ygc3VibmV0cyB0byBhZGQgdG8gdGhlIGNyZWF0ZWQgREIgc3VibmV0IGdyb3VwLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHZwY1N1Ym5ldHNgXG4gICAqIEBkZWZhdWx0IC0gcHJpdmF0ZSBzdWJuZXRzXG4gICAqL1xuICByZWFkb25seSB2cGNQbGFjZW1lbnQ/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBzdWJuZXRzIHRvIGFkZCB0byB0aGUgY3JlYXRlZCBEQiBzdWJuZXQgZ3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcHJpdmF0ZSBzdWJuZXRzXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGdyb3VwcyB0byBhc3NpZ24gdG8gdGhlIERCIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHNlY3VyaXR5IGdyb3VwIGlzIGNyZWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogZWMyLklTZWN1cml0eUdyb3VwW107XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0IGZvciB0aGUgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgcG9ydCBmb3IgdGhlIGNob3NlbiBlbmdpbmUuXG4gICAqL1xuICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgREIgcGFyYW1ldGVyIGdyb3VwIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBwYXJhbWV0ZXIgZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFtZXRlckdyb3VwPzogSVBhcmFtZXRlckdyb3VwO1xuXG4gIC8qKlxuICAgKiBUaGUgb3B0aW9uIGdyb3VwIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBvcHRpb24gZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IG9wdGlvbkdyb3VwPzogSU9wdGlvbkdyb3VwO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGVuYWJsZSBtYXBwaW5nIG9mIEFXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgYWNjb3VudHNcbiAgICogdG8gZGF0YWJhc2UgYWNjb3VudHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpYW1BdXRoZW50aWNhdGlvbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGF5cyBkdXJpbmcgd2hpY2ggYXV0b21hdGljIERCIHNuYXBzaG90cyBhcmUgcmV0YWluZWQuXG4gICAqIFNldCB0byB6ZXJvIHRvIGRpc2FibGUgYmFja3Vwcy5cbiAgICogV2hlbiBjcmVhdGluZyBhIHJlYWQgcmVwbGljYSwgeW91IG11c3QgZW5hYmxlIGF1dG9tYXRpYyBiYWNrdXBzIG9uIHRoZSBzb3VyY2VcbiAgICogZGF0YWJhc2UgaW5zdGFuY2UgYnkgc2V0dGluZyB0aGUgYmFja3VwIHJldGVudGlvbiB0byBhIHZhbHVlIG90aGVyIHRoYW4gemVyby5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEdXJhdGlvbi5kYXlzKDEpIGZvciBzb3VyY2UgaW5zdGFuY2VzLCBkaXNhYmxlZCBmb3IgcmVhZCByZXBsaWNhc1xuICAgKi9cbiAgcmVhZG9ubHkgYmFja3VwUmV0ZW50aW9uPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBkYWlseSB0aW1lIHJhbmdlIGR1cmluZyB3aGljaCBhdXRvbWF0ZWQgYmFja3VwcyBhcmUgcGVyZm9ybWVkLlxuICAgKlxuICAgKiBDb25zdHJhaW50czpcbiAgICogLSBNdXN0IGJlIGluIHRoZSBmb3JtYXQgYGhoMjQ6bWktaGgyNDptaWAuXG4gICAqIC0gTXVzdCBiZSBpbiBVbml2ZXJzYWwgQ29vcmRpbmF0ZWQgVGltZSAoVVRDKS5cbiAgICogLSBNdXN0IG5vdCBjb25mbGljdCB3aXRoIHRoZSBwcmVmZXJyZWQgbWFpbnRlbmFuY2Ugd2luZG93LlxuICAgKiAtIE11c3QgYmUgYXQgbGVhc3QgMzAgbWludXRlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIDMwLW1pbnV0ZSB3aW5kb3cgc2VsZWN0ZWQgYXQgcmFuZG9tIGZyb20gYW4gOC1ob3VyIGJsb2NrIG9mXG4gICAqIHRpbWUgZm9yIGVhY2ggQVdTIFJlZ2lvbi4gVG8gc2VlIHRoZSB0aW1lIGJsb2NrcyBhdmFpbGFibGUsIHNlZVxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvVVNFUl9Xb3JraW5nV2l0aEF1dG9tYXRlZEJhY2t1cHMuaHRtbCNVU0VSX1dvcmtpbmdXaXRoQXV0b21hdGVkQmFja3Vwcy5CYWNrdXBXaW5kb3dcbiAgICovXG4gIHJlYWRvbmx5IHByZWZlcnJlZEJhY2t1cFdpbmRvdz86IHN0cmluZztcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdG8gY29weSBhbGwgb2YgdGhlIHVzZXItZGVmaW5lZCB0YWdzIGZyb20gdGhlXG4gICAqIERCIGluc3RhbmNlIHRvIHNuYXBzaG90cyBvZiB0aGUgREIgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGNvcHlUYWdzVG9TbmFwc2hvdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIGF1dG9tYXRlZCBiYWNrdXBzIHNob3VsZCBiZSBkZWxldGVkIG9yIHJldGFpbmVkIHdoZW5cbiAgICogeW91IGRlbGV0ZSBhIERCIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVsZXRlQXV0b21hdGVkQmFja3Vwcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBpbnRlcnZhbCwgaW4gc2Vjb25kcywgYmV0d2VlbiBwb2ludHMgd2hlbiBBbWF6b24gUkRTIGNvbGxlY3RzIGVuaGFuY2VkXG4gICAqIG1vbml0b3JpbmcgbWV0cmljcyBmb3IgdGhlIERCIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGVuaGFuY2VkIG1vbml0b3JpbmdcbiAgICovXG4gIHJlYWRvbmx5IG1vbml0b3JpbmdJbnRlcnZhbD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBSb2xlIHRoYXQgd2lsbCBiZSB1c2VkIHRvIG1hbmFnZSBEQiBpbnN0YW5jZSBtb25pdG9yaW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgcm9sZSBpcyBhdXRvbWF0aWNhbGx5IGNyZWF0ZWQgZm9yIHlvdVxuICAgKi9cbiAgcmVhZG9ubHkgbW9uaXRvcmluZ1JvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gZW5hYmxlIFBlcmZvcm1hbmNlIEluc2lnaHRzIGZvciB0aGUgREIgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2UsIHVubGVzcyBgYHBlcmZvcm1hbmNlSW5zaWdodFJlbnRlbnRpb25gYCBvciBgYHBlcmZvcm1hbmNlSW5zaWdodEVuY3J5cHRpb25LZXlgYCBpcyBzZXQuXG4gICAqL1xuICByZWFkb25seSBlbmFibGVQZXJmb3JtYW5jZUluc2lnaHRzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGFtb3VudCBvZiB0aW1lLCBpbiBkYXlzLCB0byByZXRhaW4gUGVyZm9ybWFuY2UgSW5zaWdodHMgZGF0YS5cbiAgICpcbiAgICogQGRlZmF1bHQgN1xuICAgKi9cbiAgcmVhZG9ubHkgcGVyZm9ybWFuY2VJbnNpZ2h0UmV0ZW50aW9uPzogUGVyZm9ybWFuY2VJbnNpZ2h0UmV0ZW50aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgQVdTIEtNUyBrZXkgZm9yIGVuY3J5cHRpb24gb2YgUGVyZm9ybWFuY2UgSW5zaWdodHMgZGF0YS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG1hc3RlciBrZXlcbiAgICovXG4gIHJlYWRvbmx5IHBlcmZvcm1hbmNlSW5zaWdodEVuY3J5cHRpb25LZXk/OiBrbXMuSUtleTtcblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2YgbG9nIHR5cGVzIHRoYXQgbmVlZCB0byBiZSBlbmFibGVkIGZvciBleHBvcnRpbmcgdG9cbiAgICogQ2xvdWRXYXRjaCBMb2dzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGxvZyBleHBvcnRzXG4gICAqL1xuICByZWFkb25seSBjbG91ZHdhdGNoTG9nc0V4cG9ydHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBkYXlzIGxvZyBldmVudHMgYXJlIGtlcHQgaW4gQ2xvdWRXYXRjaCBMb2dzLiBXaGVuIHVwZGF0aW5nXG4gICAqIHRoaXMgcHJvcGVydHksIHVuc2V0dGluZyBpdCBkb2Vzbid0IHJlbW92ZSB0aGUgbG9nIHJldGVudGlvbiBwb2xpY3kuIFRvXG4gICAqIHJlbW92ZSB0aGUgcmV0ZW50aW9uIHBvbGljeSwgc2V0IHRoZSB2YWx1ZSB0byBgSW5maW5pdHlgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGxvZ3MgbmV2ZXIgZXhwaXJlXG4gICAqL1xuICByZWFkb25seSBjbG91ZHdhdGNoTG9nc1JldGVudGlvbj86IGxvZ3MuUmV0ZW50aW9uRGF5cztcblxuICAvKipcbiAgICogVGhlIElBTSByb2xlIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uIGFzc29jaWF0ZWQgd2l0aCB0aGUgY3VzdG9tIHJlc291cmNlXG4gICAqIHRoYXQgc2V0cyB0aGUgcmV0ZW50aW9uIHBvbGljeS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIG5ldyByb2xlIGlzIGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBjbG91ZHdhdGNoTG9nc1JldGVudGlvblJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB0aGF0IG1pbm9yIGVuZ2luZSB1cGdyYWRlcyBhcmUgYXBwbGllZCBhdXRvbWF0aWNhbGx5IHRvIHRoZVxuICAgKiBEQiBpbnN0YW5jZSBkdXJpbmcgdGhlIG1haW50ZW5hbmNlIHdpbmRvdy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b01pbm9yVmVyc2lvblVwZ3JhZGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgd2Vla2x5IHRpbWUgcmFuZ2UgKGluIFVUQykgZHVyaW5nIHdoaWNoIHN5c3RlbSBtYWludGVuYW5jZSBjYW4gb2NjdXIuXG4gICAqXG4gICAqIEZvcm1hdDogYGRkZDpoaDI0Om1pLWRkZDpoaDI0Om1pYFxuICAgKiBDb25zdHJhaW50OiBNaW5pbXVtIDMwLW1pbnV0ZSB3aW5kb3dcbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIDMwLW1pbnV0ZSB3aW5kb3cgc2VsZWN0ZWQgYXQgcmFuZG9tIGZyb20gYW4gOC1ob3VyIGJsb2NrIG9mXG4gICAqIHRpbWUgZm9yIGVhY2ggQVdTIFJlZ2lvbiwgb2NjdXJyaW5nIG9uIGEgcmFuZG9tIGRheSBvZiB0aGUgd2Vlay4gVG8gc2VlXG4gICAqIHRoZSB0aW1lIGJsb2NrcyBhdmFpbGFibGUsIHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvVVNFUl9VcGdyYWRlREJJbnN0YW5jZS5NYWludGVuYW5jZS5odG1sI0NvbmNlcHRzLkRCTWFpbnRlbmFuY2VcbiAgICovXG4gIHJlYWRvbmx5IHByZWZlcnJlZE1haW50ZW5hbmNlV2luZG93Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgREIgaW5zdGFuY2Ugc2hvdWxkIGhhdmUgZGVsZXRpb24gcHJvdGVjdGlvbiBlbmFibGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgYGByZW1vdmFsUG9saWN5YGAgaXMgUkVUQUlOLCBmYWxzZSBvdGhlcndpc2VcbiAgICovXG4gIHJlYWRvbmx5IGRlbGV0aW9uUHJvdGVjdGlvbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBDbG91ZEZvcm1hdGlvbiBwb2xpY3kgdG8gYXBwbHkgd2hlbiB0aGUgaW5zdGFuY2UgaXMgcmVtb3ZlZCBmcm9tIHRoZVxuICAgKiBzdGFjayBvciByZXBsYWNlZCBkdXJpbmcgYW4gdXBkYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJlbW92YWxQb2xpY3kuU05BUFNIT1QgKHJlbW92ZSB0aGUgcmVzb3VyY2UsIGJ1dCByZXRhaW4gYSBzbmFwc2hvdCBvZiB0aGUgZGF0YSlcbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xuXG4gIC8qKlxuICAgKiBVcHBlciBsaW1pdCB0byB3aGljaCBSRFMgY2FuIHNjYWxlIHRoZSBzdG9yYWdlIGluIEdpQihHaWJpYnl0ZSkuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL1VTRVJfUElPUFMuU3RvcmFnZVR5cGVzLmh0bWwjVVNFUl9QSU9QUy5BdXRvc2NhbGluZ1xuICAgKiBAZGVmYXVsdCAtIE5vIGF1dG9zY2FsaW5nIG9mIFJEUyBpbnN0YW5jZVxuICAgKi9cbiAgcmVhZG9ubHkgbWF4QWxsb2NhdGVkU3RvcmFnZT86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEFjdGl2ZSBEaXJlY3RvcnkgZGlyZWN0b3J5IElEIHRvIGNyZWF0ZSB0aGUgREIgaW5zdGFuY2UgaW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRG8gbm90IGpvaW4gZG9tYWluXG4gICAqL1xuICByZWFkb25seSBkb21haW4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSB0byBiZSB1c2VkIHdoZW4gbWFraW5nIEFQSSBjYWxscyB0byB0aGUgRGlyZWN0b3J5IFNlcnZpY2UuIFRoZSByb2xlIG5lZWRzIHRoZSBBV1MtbWFuYWdlZCBwb2xpY3lcbiAgICogQW1hem9uUkRTRGlyZWN0b3J5U2VydmljZUFjY2VzcyBvciBlcXVpdmFsZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRoZSByb2xlIHdpbGwgYmUgY3JlYXRlZCBmb3IgeW91IGlmIGBEYXRhYmFzZUluc3RhbmNlTmV3UHJvcHMjZG9tYWluYCBpcyBzcGVjaWZpZWRcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpblJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEV4aXN0aW5nIHN1Ym5ldCBncm91cCBmb3IgdGhlIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHN1Ym5ldCBncm91cCB3aWxsIGJlIGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBzdWJuZXRHcm91cD86IElTdWJuZXRHcm91cDtcblxuICAvKipcbiAgICogUm9sZSB0aGF0IHdpbGwgYmUgYXNzb2NpYXRlZCB3aXRoIHRoaXMgREIgaW5zdGFuY2UgdG8gZW5hYmxlIFMzIGltcG9ydC5cbiAgICogVGhpcyBmZWF0dXJlIGlzIG9ubHkgc3VwcG9ydGVkIGJ5IHRoZSBNaWNyb3NvZnQgU1FMIFNlcnZlciwgT3JhY2xlLCBhbmQgUG9zdGdyZVNRTCBlbmdpbmVzLlxuICAgKlxuICAgKiBUaGlzIHByb3BlcnR5IG11c3Qgbm90IGJlIHVzZWQgaWYgYHMzSW1wb3J0QnVja2V0c2AgaXMgdXNlZC5cbiAgICpcbiAgICogRm9yIE1pY3Jvc29mdCBTUUwgU2VydmVyOlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9TUUxTZXJ2ZXIuUHJvY2VkdXJhbC5JbXBvcnRpbmcuaHRtbFxuICAgKiBGb3IgT3JhY2xlOlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9vcmFjbGUtczMtaW50ZWdyYXRpb24uaHRtbFxuICAgKiBGb3IgUG9zdGdyZVNRTDpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvUG9zdGdyZVNRTC5Qcm9jZWR1cmFsLkltcG9ydGluZy5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTmV3IHJvbGUgaXMgY3JlYXRlZCBpZiBgczNJbXBvcnRCdWNrZXRzYCBpcyBzZXQsIG5vIHJvbGUgaXMgZGVmaW5lZCBvdGhlcndpc2VcbiAgICovXG4gIHJlYWRvbmx5IHMzSW1wb3J0Um9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogUzMgYnVja2V0cyB0aGF0IHlvdSB3YW50IHRvIGxvYWQgZGF0YSBmcm9tLlxuICAgKiBUaGlzIGZlYXR1cmUgaXMgb25seSBzdXBwb3J0ZWQgYnkgdGhlIE1pY3Jvc29mdCBTUUwgU2VydmVyLCBPcmFjbGUsIGFuZCBQb3N0Z3JlU1FMIGVuZ2luZXMuXG4gICAqXG4gICAqIFRoaXMgcHJvcGVydHkgbXVzdCBub3QgYmUgdXNlZCBpZiBgczNJbXBvcnRSb2xlYCBpcyB1c2VkLlxuICAgKlxuICAgKiBGb3IgTWljcm9zb2Z0IFNRTCBTZXJ2ZXI6XG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL1NRTFNlcnZlci5Qcm9jZWR1cmFsLkltcG9ydGluZy5odG1sXG4gICAqIEZvciBPcmFjbGU6XG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL29yYWNsZS1zMy1pbnRlZ3JhdGlvbi5odG1sXG4gICAqIEZvciBQb3N0Z3JlU1FMOlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9Qb3N0Z3JlU1FMLlByb2NlZHVyYWwuSW1wb3J0aW5nLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBzM0ltcG9ydEJ1Y2tldHM/OiBzMy5JQnVja2V0W107XG5cbiAgLyoqXG4gICAqIFJvbGUgdGhhdCB3aWxsIGJlIGFzc29jaWF0ZWQgd2l0aCB0aGlzIERCIGluc3RhbmNlIHRvIGVuYWJsZSBTMyBleHBvcnQuXG4gICAqXG4gICAqIFRoaXMgcHJvcGVydHkgbXVzdCBub3QgYmUgdXNlZCBpZiBgczNFeHBvcnRCdWNrZXRzYCBpcyB1c2VkLlxuICAgKlxuICAgKiBGb3IgTWljcm9zb2Z0IFNRTCBTZXJ2ZXI6XG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL1NRTFNlcnZlci5Qcm9jZWR1cmFsLkltcG9ydGluZy5odG1sXG4gICAqIEZvciBPcmFjbGU6XG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblJEUy9sYXRlc3QvVXNlckd1aWRlL29yYWNsZS1zMy1pbnRlZ3JhdGlvbi5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTmV3IHJvbGUgaXMgY3JlYXRlZCBpZiBgczNFeHBvcnRCdWNrZXRzYCBpcyBzZXQsIG5vIHJvbGUgaXMgZGVmaW5lZCBvdGhlcndpc2VcbiAgICovXG4gIHJlYWRvbmx5IHMzRXhwb3J0Um9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogUzMgYnVja2V0cyB0aGF0IHlvdSB3YW50IHRvIGxvYWQgZGF0YSBpbnRvLlxuICAgKlxuICAgKiBUaGlzIHByb3BlcnR5IG11c3Qgbm90IGJlIHVzZWQgaWYgYHMzRXhwb3J0Um9sZWAgaXMgdXNlZC5cbiAgICpcbiAgICogRm9yIE1pY3Jvc29mdCBTUUwgU2VydmVyOlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9TUUxTZXJ2ZXIuUHJvY2VkdXJhbC5JbXBvcnRpbmcuaHRtbFxuICAgKiBGb3IgT3JhY2xlOlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L1VzZXJHdWlkZS9vcmFjbGUtczMtaW50ZWdyYXRpb24uaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHMzRXhwb3J0QnVja2V0cz86IHMzLklCdWNrZXRbXTtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIERCIGluc3RhbmNlIGlzIGFuIGludGVybmV0LWZhY2luZyBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgdHJ1ZWAgaWYgYHZwY1N1Ym5ldHNgIGlzIGBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQ2AsIGBmYWxzZWAgb3RoZXJ3aXNlXG4gICAqL1xuICByZWFkb25seSBwdWJsaWNseUFjY2Vzc2libGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbmV0d29yayB0eXBlIG9mIHRoZSBEQiBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJUFY0XG4gICAqL1xuICByZWFkb25seSBuZXR3b3JrVHlwZT86IE5ldHdvcmtUeXBlO1xufVxuXG4vKipcbiAqIEEgbmV3IGRhdGFiYXNlIGluc3RhbmNlLlxuICovXG5hYnN0cmFjdCBjbGFzcyBEYXRhYmFzZUluc3RhbmNlTmV3IGV4dGVuZHMgRGF0YWJhc2VJbnN0YW5jZUJhc2UgaW1wbGVtZW50cyBJRGF0YWJhc2VJbnN0YW5jZSB7XG4gIC8qKlxuICAgKiBUaGUgVlBDIHdoZXJlIHRoaXMgZGF0YWJhc2UgaW5zdGFuY2UgaXMgZGVwbG95ZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucztcblxuICBwcm90ZWN0ZWQgYWJzdHJhY3QgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBlYzIuSW5zdGFuY2VUeXBlO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSB2cGNQbGFjZW1lbnQ/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgbmV3Q2ZuUHJvcHM6IENmbkRCSW5zdGFuY2VQcm9wcztcblxuICBwcml2YXRlIHJlYWRvbmx5IGNsb3Vkd2F0Y2hMb2dzRXhwb3J0cz86IHN0cmluZ1tdO1xuICBwcml2YXRlIHJlYWRvbmx5IGNsb3Vkd2F0Y2hMb2dzUmV0ZW50aW9uPzogbG9ncy5SZXRlbnRpb25EYXlzO1xuICBwcml2YXRlIHJlYWRvbmx5IGNsb3Vkd2F0Y2hMb2dzUmV0ZW50aW9uUm9sZT86IGlhbS5JUm9sZTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGRvbWFpbklkPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGRvbWFpblJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgcHJvdGVjdGVkIGVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uPzogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VJbnN0YW5jZU5ld1Byb3BzKSB7XG4gICAgLy8gUkRTIGFsd2F5cyBsb3dlci1jYXNlcyB0aGUgSUQgb2YgdGhlIGRhdGFiYXNlLCBzbyB1c2UgdGhhdCBmb3IgdGhlIHBoeXNpY2FsIG5hbWVcbiAgICAvLyAod2hpY2ggaXMgdGhlIG5hbWUgdXNlZCBmb3IgY3Jvc3MtZW52aXJvbm1lbnQgYWNjZXNzLCBzbyBpdCBuZWVkcyB0byBiZSBjb3JyZWN0LFxuICAgIC8vIHJlZ2FyZGxlc3Mgb2YgdGhlIGZlYXR1cmUgZmxhZyB0aGF0IGNoYW5nZXMgaXQgaW4gdGhlIHRlbXBsYXRlIGZvciB0aGUgTDEpXG4gICAgY29uc3QgaW5zdGFuY2VQaHlzaWNhbE5hbWUgPSBUb2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuaW5zdGFuY2VJZGVudGlmaWVyKVxuICAgICAgPyBwcm9wcy5pbnN0YW5jZUlkZW50aWZpZXJcbiAgICAgIDogcHJvcHMuaW5zdGFuY2VJZGVudGlmaWVyPy50b0xvd2VyQ2FzZSgpO1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBpbnN0YW5jZVBoeXNpY2FsTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMudnBjID0gcHJvcHMudnBjO1xuICAgIGlmIChwcm9wcy52cGNTdWJuZXRzICYmIHByb3BzLnZwY1BsYWNlbWVudCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmx5IG9uZSBvZiBgdnBjU3VibmV0c2Agb3IgYHZwY1BsYWNlbWVudGAgY2FuIGJlIHNwZWNpZmllZCcpO1xuICAgIH1cbiAgICB0aGlzLnZwY1BsYWNlbWVudCA9IHByb3BzLnZwY1N1Ym5ldHMgPz8gcHJvcHMudnBjUGxhY2VtZW50O1xuXG4gICAgaWYgKHByb3BzLm11bHRpQXogPT09IHRydWUgJiYgcHJvcHMuYXZhaWxhYmlsaXR5Wm9uZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdSZXF1ZXN0aW5nIGEgc3BlY2lmaWMgYXZhaWxhYmlsaXR5IHpvbmUgaXMgbm90IHZhbGlkIGZvciBNdWx0aS1BWiBpbnN0YW5jZXMnKTtcbiAgICB9XG5cbiAgICBjb25zdCBzdWJuZXRHcm91cCA9IHByb3BzLnN1Ym5ldEdyb3VwID8/IG5ldyBTdWJuZXRHcm91cCh0aGlzLCAnU3VibmV0R3JvdXAnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogYFN1Ym5ldCBncm91cCBmb3IgJHt0aGlzLm5vZGUuaWR9IGRhdGFiYXNlYCxcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICB2cGNTdWJuZXRzOiB0aGlzLnZwY1BsYWNlbWVudCxcbiAgICAgIHJlbW92YWxQb2xpY3k6IHJlbmRlclVubGVzcyhoZWxwZXJSZW1vdmFsUG9saWN5KHByb3BzLnJlbW92YWxQb2xpY3kpLCBSZW1vdmFsUG9saWN5LkRFU1RST1kpLFxuICAgIH0pO1xuXG4gICAgY29uc3Qgc2VjdXJpdHlHcm91cHMgPSBwcm9wcy5zZWN1cml0eUdyb3VwcyB8fCBbbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdTZWN1cml0eUdyb3VwJywge1xuICAgICAgZGVzY3JpcHRpb246IGBTZWN1cml0eSBncm91cCBmb3IgJHt0aGlzLm5vZGUuaWR9IGRhdGFiYXNlYCxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgIH0pXTtcblxuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgZWMyLkNvbm5lY3Rpb25zKHtcbiAgICAgIHNlY3VyaXR5R3JvdXBzLFxuICAgICAgZGVmYXVsdFBvcnQ6IGVjMi5Qb3J0LnRjcChMYXp5Lm51bWJlcih7IHByb2R1Y2U6ICgpID0+IHRoaXMuaW5zdGFuY2VFbmRwb2ludC5wb3J0IH0pKSxcbiAgICB9KTtcblxuICAgIGxldCBtb25pdG9yaW5nUm9sZTtcbiAgICBpZiAocHJvcHMubW9uaXRvcmluZ0ludGVydmFsICYmIHByb3BzLm1vbml0b3JpbmdJbnRlcnZhbC50b1NlY29uZHMoKSkge1xuICAgICAgbW9uaXRvcmluZ1JvbGUgPSBwcm9wcy5tb25pdG9yaW5nUm9sZSB8fCBuZXcgaWFtLlJvbGUodGhpcywgJ01vbml0b3JpbmdSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbW9uaXRvcmluZy5yZHMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BbWF6b25SRFNFbmhhbmNlZE1vbml0b3JpbmdSb2xlJyldLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3Qgc3RvcmFnZVR5cGUgPSBwcm9wcy5zdG9yYWdlVHlwZSA/PyBTdG9yYWdlVHlwZS5HUDI7XG4gICAgY29uc3QgaW9wcyA9IGRlZmF1bHRJb3BzKHN0b3JhZ2VUeXBlLCBwcm9wcy5pb3BzKTtcbiAgICBpZiAocHJvcHMuc3RvcmFnZVRocm91Z2hwdXQgJiYgc3RvcmFnZVR5cGUgIT09IFN0b3JhZ2VUeXBlLkdQMykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgc3RvcmFnZSB0aHJvdWdocHV0IGNhbiBvbmx5IGJlIHNwZWNpZmllZCB3aXRoIEdQMyBzdG9yYWdlIHR5cGUuIEdvdCAke3N0b3JhZ2VUeXBlfS5gKTtcbiAgICB9XG4gICAgaWYgKHN0b3JhZ2VUeXBlID09PSBTdG9yYWdlVHlwZS5HUDMgJiYgcHJvcHMuc3RvcmFnZVRocm91Z2hwdXQgJiYgaW9wc1xuICAgICAgICAmJiAhVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLnN0b3JhZ2VUaHJvdWdocHV0KSAmJiAhVG9rZW4uaXNVbnJlc29sdmVkKGlvcHMpXG4gICAgICAgICYmIHByb3BzLnN0b3JhZ2VUaHJvdWdocHV0L2lvcHMgPiAwLjI1KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBtYXhpbXVtIHJhdGlvIG9mIHN0b3JhZ2UgdGhyb3VnaHB1dCB0byBJT1BTIGlzIDAuMjUuIEdvdCAke3Byb3BzLnN0b3JhZ2VUaHJvdWdocHV0L2lvcHN9LmApO1xuICAgIH1cblxuICAgIHRoaXMuY2xvdWR3YXRjaExvZ3NFeHBvcnRzID0gcHJvcHMuY2xvdWR3YXRjaExvZ3NFeHBvcnRzO1xuICAgIHRoaXMuY2xvdWR3YXRjaExvZ3NSZXRlbnRpb24gPSBwcm9wcy5jbG91ZHdhdGNoTG9nc1JldGVudGlvbjtcbiAgICB0aGlzLmNsb3Vkd2F0Y2hMb2dzUmV0ZW50aW9uUm9sZSA9IHByb3BzLmNsb3Vkd2F0Y2hMb2dzUmV0ZW50aW9uUm9sZTtcbiAgICB0aGlzLmVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uID0gcHJvcHMuaWFtQXV0aGVudGljYXRpb247XG5cbiAgICBjb25zdCBlbmFibGVQZXJmb3JtYW5jZUluc2lnaHRzID0gcHJvcHMuZW5hYmxlUGVyZm9ybWFuY2VJbnNpZ2h0c1xuICAgICAgfHwgcHJvcHMucGVyZm9ybWFuY2VJbnNpZ2h0UmV0ZW50aW9uICE9PSB1bmRlZmluZWQgfHwgcHJvcHMucGVyZm9ybWFuY2VJbnNpZ2h0RW5jcnlwdGlvbktleSAhPT0gdW5kZWZpbmVkO1xuICAgIGlmIChlbmFibGVQZXJmb3JtYW5jZUluc2lnaHRzICYmIHByb3BzLmVuYWJsZVBlcmZvcm1hbmNlSW5zaWdodHMgPT09IGZhbHNlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2BlbmFibGVQZXJmb3JtYW5jZUluc2lnaHRzYCBkaXNhYmxlZCwgYnV0IGBwZXJmb3JtYW5jZUluc2lnaHRSZXRlbnRpb25gIG9yIGBwZXJmb3JtYW5jZUluc2lnaHRFbmNyeXB0aW9uS2V5YCB3YXMgc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmRvbWFpbikge1xuICAgICAgdGhpcy5kb21haW5JZCA9IHByb3BzLmRvbWFpbjtcbiAgICAgIHRoaXMuZG9tYWluUm9sZSA9IHByb3BzLmRvbWFpblJvbGUgfHwgbmV3IGlhbS5Sb2xlKHRoaXMsICdSRFNEaXJlY3RvcnlTZXJ2aWNlUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3Jkcy5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvblJEU0RpcmVjdG9yeVNlcnZpY2VBY2Nlc3MnKSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IG1heWJlTG93ZXJjYXNlZEluc3RhbmNlSWQgPSBGZWF0dXJlRmxhZ3Mub2YodGhpcykuaXNFbmFibGVkKGN4YXBpLlJEU19MT1dFUkNBU0VfREJfSURFTlRJRklFUilcbiAgICAmJiAhVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLmluc3RhbmNlSWRlbnRpZmllcilcbiAgICAgID8gcHJvcHMuaW5zdGFuY2VJZGVudGlmaWVyPy50b0xvd2VyQ2FzZSgpXG4gICAgICA6IHByb3BzLmluc3RhbmNlSWRlbnRpZmllcjtcblxuICAgIGNvbnN0IGluc3RhbmNlUGFyYW1ldGVyR3JvdXBDb25maWcgPSBwcm9wcy5wYXJhbWV0ZXJHcm91cD8uYmluZFRvSW5zdGFuY2Uoe30pO1xuICAgIHRoaXMubmV3Q2ZuUHJvcHMgPSB7XG4gICAgICBhdXRvTWlub3JWZXJzaW9uVXBncmFkZTogcHJvcHMuYXV0b01pbm9yVmVyc2lvblVwZ3JhZGUsXG4gICAgICBhdmFpbGFiaWxpdHlab25lOiBwcm9wcy5tdWx0aUF6ID8gdW5kZWZpbmVkIDogcHJvcHMuYXZhaWxhYmlsaXR5Wm9uZSxcbiAgICAgIGJhY2t1cFJldGVudGlvblBlcmlvZDogcHJvcHMuYmFja3VwUmV0ZW50aW9uPy50b0RheXMoKSxcbiAgICAgIGNvcHlUYWdzVG9TbmFwc2hvdDogcHJvcHMuY29weVRhZ3NUb1NuYXBzaG90ID8/IHRydWUsXG4gICAgICBkYkluc3RhbmNlQ2xhc3M6IExhenkuc3RyaW5nKHsgcHJvZHVjZTogKCkgPT4gYGRiLiR7dGhpcy5pbnN0YW5jZVR5cGV9YCB9KSxcbiAgICAgIGRiSW5zdGFuY2VJZGVudGlmaWVyOiBUb2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuaW5zdGFuY2VJZGVudGlmaWVyKVxuICAgICAgICAvLyBpZiB0aGUgcGFzc2VkIGlkZW50aWZpZXIgaXMgYSBUb2tlbixcbiAgICAgICAgLy8gd2UgbmVlZCB0byB1c2UgdGhlIHBoeXNpY2FsTmFtZSBvZiB0aGUgZGF0YWJhc2VcbiAgICAgICAgLy8gKHdlIGNhbm5vdCBjaGFuZ2UgaXRzIGNhc2UgYW55d2F5KSxcbiAgICAgICAgLy8gYXMgaXQgbWlnaHQgYmUgdXNlZCBpbiBhIGNyb3NzLWVudmlyb25tZW50IGZhc2hpb25cbiAgICAgICAgPyB0aGlzLnBoeXNpY2FsTmFtZVxuICAgICAgICA6IG1heWJlTG93ZXJjYXNlZEluc3RhbmNlSWQsXG4gICAgICBkYlN1Ym5ldEdyb3VwTmFtZTogc3VibmV0R3JvdXAuc3VibmV0R3JvdXBOYW1lLFxuICAgICAgZGVsZXRlQXV0b21hdGVkQmFja3VwczogcHJvcHMuZGVsZXRlQXV0b21hdGVkQmFja3VwcyxcbiAgICAgIGRlbGV0aW9uUHJvdGVjdGlvbjogZGVmYXVsdERlbGV0aW9uUHJvdGVjdGlvbihwcm9wcy5kZWxldGlvblByb3RlY3Rpb24sIHByb3BzLnJlbW92YWxQb2xpY3kpLFxuICAgICAgZW5hYmxlQ2xvdWR3YXRjaExvZ3NFeHBvcnRzOiB0aGlzLmNsb3Vkd2F0Y2hMb2dzRXhwb3J0cyxcbiAgICAgIGVuYWJsZUlhbURhdGFiYXNlQXV0aGVudGljYXRpb246IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5lbmFibGVJYW1BdXRoZW50aWNhdGlvbiB9KSxcbiAgICAgIGVuYWJsZVBlcmZvcm1hbmNlSW5zaWdodHM6IGVuYWJsZVBlcmZvcm1hbmNlSW5zaWdodHMgfHwgcHJvcHMuZW5hYmxlUGVyZm9ybWFuY2VJbnNpZ2h0cywgLy8gZmFsbCBiYWNrIHRvIHVuZGVmaW5lZCBpZiBub3Qgc2V0LFxuICAgICAgaW9wcyxcbiAgICAgIG1vbml0b3JpbmdJbnRlcnZhbDogcHJvcHMubW9uaXRvcmluZ0ludGVydmFsPy50b1NlY29uZHMoKSxcbiAgICAgIG1vbml0b3JpbmdSb2xlQXJuOiBtb25pdG9yaW5nUm9sZT8ucm9sZUFybixcbiAgICAgIG11bHRpQXo6IHByb3BzLm11bHRpQXosXG4gICAgICBkYlBhcmFtZXRlckdyb3VwTmFtZTogaW5zdGFuY2VQYXJhbWV0ZXJHcm91cENvbmZpZz8ucGFyYW1ldGVyR3JvdXBOYW1lLFxuICAgICAgb3B0aW9uR3JvdXBOYW1lOiBwcm9wcy5vcHRpb25Hcm91cD8ub3B0aW9uR3JvdXBOYW1lLFxuICAgICAgcGVyZm9ybWFuY2VJbnNpZ2h0c0ttc0tleUlkOiBwcm9wcy5wZXJmb3JtYW5jZUluc2lnaHRFbmNyeXB0aW9uS2V5Py5rZXlBcm4sXG4gICAgICBwZXJmb3JtYW5jZUluc2lnaHRzUmV0ZW50aW9uUGVyaW9kOiBlbmFibGVQZXJmb3JtYW5jZUluc2lnaHRzXG4gICAgICAgID8gKHByb3BzLnBlcmZvcm1hbmNlSW5zaWdodFJldGVudGlvbiB8fCBQZXJmb3JtYW5jZUluc2lnaHRSZXRlbnRpb24uREVGQVVMVClcbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBwb3J0OiBwcm9wcy5wb3J0ICE9PSB1bmRlZmluZWQgPyBUb2tlbml6YXRpb24uc3RyaW5naWZ5TnVtYmVyKHByb3BzLnBvcnQpIDogdW5kZWZpbmVkLFxuICAgICAgcHJlZmVycmVkQmFja3VwV2luZG93OiBwcm9wcy5wcmVmZXJyZWRCYWNrdXBXaW5kb3csXG4gICAgICBwcmVmZXJyZWRNYWludGVuYW5jZVdpbmRvdzogcHJvcHMucHJlZmVycmVkTWFpbnRlbmFuY2VXaW5kb3csXG4gICAgICBwcm9jZXNzb3JGZWF0dXJlczogcHJvcHMucHJvY2Vzc29yRmVhdHVyZXMgJiYgcmVuZGVyUHJvY2Vzc29yRmVhdHVyZXMocHJvcHMucHJvY2Vzc29yRmVhdHVyZXMpLFxuICAgICAgcHVibGljbHlBY2Nlc3NpYmxlOiBwcm9wcy5wdWJsaWNseUFjY2Vzc2libGUgPz8gKHRoaXMudnBjUGxhY2VtZW50ICYmIHRoaXMudnBjUGxhY2VtZW50LnN1Ym5ldFR5cGUgPT09IGVjMi5TdWJuZXRUeXBlLlBVQkxJQyksXG4gICAgICBzdG9yYWdlVHlwZSxcbiAgICAgIHN0b3JhZ2VUaHJvdWdocHV0OiBwcm9wcy5zdG9yYWdlVGhyb3VnaHB1dCxcbiAgICAgIHZwY1NlY3VyaXR5R3JvdXBzOiBzZWN1cml0eUdyb3Vwcy5tYXAocyA9PiBzLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICBtYXhBbGxvY2F0ZWRTdG9yYWdlOiBwcm9wcy5tYXhBbGxvY2F0ZWRTdG9yYWdlLFxuICAgICAgZG9tYWluOiB0aGlzLmRvbWFpbklkLFxuICAgICAgZG9tYWluSWFtUm9sZU5hbWU6IHRoaXMuZG9tYWluUm9sZT8ucm9sZU5hbWUsXG4gICAgICBuZXR3b3JrVHlwZTogcHJvcHMubmV0d29ya1R5cGUsXG4gICAgfTtcbiAgfVxuXG4gIHByb3RlY3RlZCBzZXRMb2dSZXRlbnRpb24oKSB7XG4gICAgaWYgKHRoaXMuY2xvdWR3YXRjaExvZ3NFeHBvcnRzICYmIHRoaXMuY2xvdWR3YXRjaExvZ3NSZXRlbnRpb24pIHtcbiAgICAgIGZvciAoY29uc3QgbG9nIG9mIHRoaXMuY2xvdWR3YXRjaExvZ3NFeHBvcnRzKSB7XG4gICAgICAgIG5ldyBsb2dzLkxvZ1JldGVudGlvbih0aGlzLCBgTG9nUmV0ZW50aW9uJHtsb2d9YCwge1xuICAgICAgICAgIGxvZ0dyb3VwTmFtZTogYC9hd3MvcmRzL2luc3RhbmNlLyR7dGhpcy5pbnN0YW5jZUlkZW50aWZpZXJ9LyR7bG9nfWAsXG4gICAgICAgICAgcmV0ZW50aW9uOiB0aGlzLmNsb3Vkd2F0Y2hMb2dzUmV0ZW50aW9uLFxuICAgICAgICAgIHJvbGU6IHRoaXMuY2xvdWR3YXRjaExvZ3NSZXRlbnRpb25Sb2xlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBEYXRhYmFzZUluc3RhbmNlU291cmNlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VJbnN0YW5jZVNvdXJjZVByb3BzIGV4dGVuZHMgRGF0YWJhc2VJbnN0YW5jZU5ld1Byb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBkYXRhYmFzZSBlbmdpbmUuXG4gICAqL1xuICByZWFkb25seSBlbmdpbmU6IElJbnN0YW5jZUVuZ2luZTtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGNvbXB1dGUgYW5kIG1lbW9yeSBjYXBhY2l0eSBmb3IgdGhlIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG01LmxhcmdlIChvciwgbW9yZSBzcGVjaWZpY2FsbHksIGRiLm01LmxhcmdlKVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogVGhlIGxpY2Vuc2UgbW9kZWwuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gUkRTIGRlZmF1bHQgbGljZW5zZSBtb2RlbFxuICAgKi9cbiAgcmVhZG9ubHkgbGljZW5zZU1vZGVsPzogTGljZW5zZU1vZGVsO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGFsbG93IG1ham9yIHZlcnNpb24gdXBncmFkZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBhbGxvd01ham9yVmVyc2lvblVwZ3JhZGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgdGltZSB6b25lIG9mIHRoZSBpbnN0YW5jZS4gVGhpcyBpcyBjdXJyZW50bHkgc3VwcG9ydGVkIG9ubHkgYnkgTWljcm9zb2Z0IFNxbCBTZXJ2ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gUkRTIGRlZmF1bHQgdGltZXpvbmVcbiAgICovXG4gIHJlYWRvbmx5IHRpbWV6b25lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgYWxsb2NhdGVkIHN0b3JhZ2Ugc2l6ZSwgc3BlY2lmaWVkIGluIGdpYmlieXRlcyAoR2lCKS5cbiAgICpcbiAgICogQGRlZmF1bHQgMTAwXG4gICAqL1xuICByZWFkb25seSBhbGxvY2F0ZWRTdG9yYWdlPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZGF0YWJhc2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YWJhc2VOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFyYW1ldGVycyBpbiB0aGUgREJQYXJhbWV0ZXJHcm91cCB0byBjcmVhdGUgYXV0b21hdGljYWxseVxuICAgKlxuICAgKiBZb3UgY2FuIG9ubHkgc3BlY2lmeSBwYXJhbWV0ZXJHcm91cCBvciBwYXJhbWV0ZXJzIGJ1dCBub3QgYm90aC5cbiAgICogWW91IG5lZWQgdG8gdXNlIGEgdmVyc2lvbmVkIGVuZ2luZSB0byBhdXRvLWdlbmVyYXRlIGEgREJQYXJhbWV0ZXJHcm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBwYXJhbWV0ZXJzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbn1cblxuLyoqXG4gKiBBIG5ldyBzb3VyY2UgZGF0YWJhc2UgaW5zdGFuY2UgKG5vdCBhIHJlYWQgcmVwbGljYSlcbiAqL1xuYWJzdHJhY3QgY2xhc3MgRGF0YWJhc2VJbnN0YW5jZVNvdXJjZSBleHRlbmRzIERhdGFiYXNlSW5zdGFuY2VOZXcgaW1wbGVtZW50cyBJRGF0YWJhc2VJbnN0YW5jZSB7XG4gIHB1YmxpYyByZWFkb25seSBlbmdpbmU/OiBJSW5zdGFuY2VFbmdpbmU7XG4gIC8qKlxuICAgKiBUaGUgQVdTIFNlY3JldHMgTWFuYWdlciBzZWNyZXQgYXR0YWNoZWQgdG8gdGhlIGluc3RhbmNlLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHNlY3JldD86IHNlY3JldHNtYW5hZ2VyLklTZWNyZXQ7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHNvdXJjZUNmblByb3BzOiBDZm5EQkluc3RhbmNlUHJvcHM7XG4gIHByb3RlY3RlZCByZWFkb25seSBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGU7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBzaW5nbGVVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbjtcbiAgcHJpdmF0ZSByZWFkb25seSBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEYXRhYmFzZUluc3RhbmNlU291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIHRoaXMuc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb24gPSBwcm9wcy5lbmdpbmUuc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb247XG4gICAgdGhpcy5tdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uID0gcHJvcHMuZW5naW5lLm11bHRpVXNlclJvdGF0aW9uQXBwbGljYXRpb247XG4gICAgdGhpcy5lbmdpbmUgPSBwcm9wcy5lbmdpbmU7XG5cbiAgICBjb25zdCBlbmdpbmVUeXBlID0gcHJvcHMuZW5naW5lLmVuZ2luZVR5cGU7XG4gICAgLy8gb25seSBPcmFjbGUgYW5kIFNRTCBTZXJ2ZXIgcmVxdWlyZSB0aGUgaW1wb3J0IGFuZCBleHBvcnQgUm9sZXMgdG8gYmUgdGhlIHNhbWVcbiAgICBjb25zdCBjb21iaW5lUm9sZXMgPSBlbmdpbmVUeXBlLnN0YXJ0c1dpdGgoJ29yYWNsZS0nKSB8fCBlbmdpbmVUeXBlLnN0YXJ0c1dpdGgoJ3NxbHNlcnZlci0nKTtcbiAgICBsZXQgeyBzM0ltcG9ydFJvbGUsIHMzRXhwb3J0Um9sZSB9ID0gc2V0dXBTM0ltcG9ydEV4cG9ydCh0aGlzLCBwcm9wcywgY29tYmluZVJvbGVzKTtcbiAgICBjb25zdCBlbmdpbmVDb25maWcgPSBwcm9wcy5lbmdpbmUuYmluZFRvSW5zdGFuY2UodGhpcywge1xuICAgICAgLi4ucHJvcHMsXG4gICAgICBzM0ltcG9ydFJvbGUsXG4gICAgICBzM0V4cG9ydFJvbGUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBpbnN0YW5jZUFzc29jaWF0ZWRSb2xlczogQ2ZuREJJbnN0YW5jZS5EQkluc3RhbmNlUm9sZVByb3BlcnR5W10gPSBbXTtcbiAgICBjb25zdCBlbmdpbmVGZWF0dXJlcyA9IGVuZ2luZUNvbmZpZy5mZWF0dXJlcztcbiAgICBpZiAoczNJbXBvcnRSb2xlKSB7XG4gICAgICBpZiAoIWVuZ2luZUZlYXR1cmVzPy5zM0ltcG9ydCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEVuZ2luZSAnJHtlbmdpbmVEZXNjcmlwdGlvbihwcm9wcy5lbmdpbmUpfScgZG9lcyBub3Qgc3VwcG9ydCBTMyBpbXBvcnRgKTtcbiAgICAgIH1cbiAgICAgIGluc3RhbmNlQXNzb2NpYXRlZFJvbGVzLnB1c2goeyByb2xlQXJuOiBzM0ltcG9ydFJvbGUucm9sZUFybiwgZmVhdHVyZU5hbWU6IGVuZ2luZUZlYXR1cmVzPy5zM0ltcG9ydCB9KTtcbiAgICB9XG4gICAgaWYgKHMzRXhwb3J0Um9sZSkge1xuICAgICAgaWYgKCFlbmdpbmVGZWF0dXJlcz8uczNFeHBvcnQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBFbmdpbmUgJyR7ZW5naW5lRGVzY3JpcHRpb24ocHJvcHMuZW5naW5lKX0nIGRvZXMgbm90IHN1cHBvcnQgUzMgZXhwb3J0YCk7XG4gICAgICB9XG4gICAgICAvLyBvbmx5IGFkZCB0aGUgZXhwb3J0IGZlYXR1cmUgaWYgaXQncyBkaWZmZXJlbnQgZnJvbSB0aGUgaW1wb3J0IGZlYXR1cmVcbiAgICAgIGlmIChlbmdpbmVGZWF0dXJlcy5zM0ltcG9ydCAhPT0gZW5naW5lRmVhdHVyZXM/LnMzRXhwb3J0KSB7XG4gICAgICAgIGluc3RhbmNlQXNzb2NpYXRlZFJvbGVzLnB1c2goeyByb2xlQXJuOiBzM0V4cG9ydFJvbGUucm9sZUFybiwgZmVhdHVyZU5hbWU6IGVuZ2luZUZlYXR1cmVzPy5zM0V4cG9ydCB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLmluc3RhbmNlVHlwZSA9IHByb3BzLmluc3RhbmNlVHlwZSA/PyBlYzIuSW5zdGFuY2VUeXBlLm9mKGVjMi5JbnN0YW5jZUNsYXNzLk01LCBlYzIuSW5zdGFuY2VTaXplLkxBUkdFKTtcblxuICAgIGlmIChwcm9wcy5wYXJhbWV0ZXJHcm91cCAmJiBwcm9wcy5wYXJhbWV0ZXJzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5ub3Qgc3BlY2lmeSBib3RoIHBhcmFtZXRlckdyb3VwIGFuZCBwYXJhbWV0ZXJzJyk7XG4gICAgfVxuXG4gICAgY29uc3QgZGJQYXJhbWV0ZXJHcm91cE5hbWUgPSBwcm9wcy5wYXJhbWV0ZXJzXG4gICAgICA/IG5ldyBQYXJhbWV0ZXJHcm91cCh0aGlzLCAnUGFyYW1ldGVyR3JvdXAnLCB7XG4gICAgICAgIGVuZ2luZTogcHJvcHMuZW5naW5lLFxuICAgICAgICBwYXJhbWV0ZXJzOiBwcm9wcy5wYXJhbWV0ZXJzLFxuICAgICAgfSkuYmluZFRvSW5zdGFuY2Uoe30pLnBhcmFtZXRlckdyb3VwTmFtZVxuICAgICAgOiB0aGlzLm5ld0NmblByb3BzLmRiUGFyYW1ldGVyR3JvdXBOYW1lO1xuXG4gICAgdGhpcy5zb3VyY2VDZm5Qcm9wcyA9IHtcbiAgICAgIC4uLnRoaXMubmV3Q2ZuUHJvcHMsXG4gICAgICBhc3NvY2lhdGVkUm9sZXM6IGluc3RhbmNlQXNzb2NpYXRlZFJvbGVzLmxlbmd0aCA+IDAgPyBpbnN0YW5jZUFzc29jaWF0ZWRSb2xlcyA6IHVuZGVmaW5lZCxcbiAgICAgIG9wdGlvbkdyb3VwTmFtZTogZW5naW5lQ29uZmlnLm9wdGlvbkdyb3VwPy5vcHRpb25Hcm91cE5hbWUsXG4gICAgICBhbGxvY2F0ZWRTdG9yYWdlOiBwcm9wcy5hbGxvY2F0ZWRTdG9yYWdlPy50b1N0cmluZygpID8/ICcxMDAnLFxuICAgICAgYWxsb3dNYWpvclZlcnNpb25VcGdyYWRlOiBwcm9wcy5hbGxvd01ham9yVmVyc2lvblVwZ3JhZGUsXG4gICAgICBkYk5hbWU6IHByb3BzLmRhdGFiYXNlTmFtZSxcbiAgICAgIGVuZ2luZTogZW5naW5lVHlwZSxcbiAgICAgIGVuZ2luZVZlcnNpb246IHByb3BzLmVuZ2luZS5lbmdpbmVWZXJzaW9uPy5mdWxsVmVyc2lvbixcbiAgICAgIGxpY2Vuc2VNb2RlbDogcHJvcHMubGljZW5zZU1vZGVsLFxuICAgICAgdGltZXpvbmU6IHByb3BzLnRpbWV6b25lLFxuICAgICAgZGJQYXJhbWV0ZXJHcm91cE5hbWUsXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIHRoZSBzaW5nbGUgdXNlciByb3RhdGlvbiBvZiB0aGUgbWFzdGVyIHBhc3N3b3JkIHRvIHRoaXMgaW5zdGFuY2UuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIHRoZSBvcHRpb25zIGZvciB0aGUgcm90YXRpb24sXG4gICAqICAgICAgICAgICAgICAgIGlmIHlvdSB3YW50IHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0c1xuICAgKi9cbiAgcHVibGljIGFkZFJvdGF0aW9uU2luZ2xlVXNlcihvcHRpb25zOiBSb3RhdGlvblNpbmdsZVVzZXJPcHRpb25zID0ge30pOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbiB7XG4gICAgaWYgKCF0aGlzLnNlY3JldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYWRkIHNpbmdsZSB1c2VyIHJvdGF0aW9uIGZvciBhbiBpbnN0YW5jZSB3aXRob3V0IHNlY3JldC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBpZCA9ICdSb3RhdGlvblNpbmdsZVVzZXInO1xuICAgIGNvbnN0IGV4aXN0aW5nID0gdGhpcy5ub2RlLnRyeUZpbmRDaGlsZChpZCk7XG4gICAgaWYgKGV4aXN0aW5nKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Egc2luZ2xlIHVzZXIgcm90YXRpb24gd2FzIGFscmVhZHkgYWRkZWQgdG8gdGhpcyBpbnN0YW5jZS4nKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uKHRoaXMsIGlkLCB7XG4gICAgICAuLi5hcHBseURlZmF1bHRSb3RhdGlvbk9wdGlvbnMob3B0aW9ucywgdGhpcy52cGNQbGFjZW1lbnQpLFxuICAgICAgc2VjcmV0OiB0aGlzLnNlY3JldCxcbiAgICAgIGFwcGxpY2F0aW9uOiB0aGlzLnNpbmdsZVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uLFxuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIHRhcmdldDogdGhpcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIHRoZSBtdWx0aSB1c2VyIHJvdGF0aW9uIHRvIHRoaXMgaW5zdGFuY2UuXG4gICAqL1xuICBwdWJsaWMgYWRkUm90YXRpb25NdWx0aVVzZXIoaWQ6IHN0cmluZywgb3B0aW9uczogUm90YXRpb25NdWx0aVVzZXJPcHRpb25zKTogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb24ge1xuICAgIGlmICghdGhpcy5zZWNyZXQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGFkZCBtdWx0aSB1c2VyIHJvdGF0aW9uIGZvciBhbiBpbnN0YW5jZSB3aXRob3V0IHNlY3JldC4nKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uKHRoaXMsIGlkLCB7XG4gICAgICAuLi5hcHBseURlZmF1bHRSb3RhdGlvbk9wdGlvbnMob3B0aW9ucywgdGhpcy52cGNQbGFjZW1lbnQpLFxuICAgICAgc2VjcmV0OiBvcHRpb25zLnNlY3JldCxcbiAgICAgIG1hc3RlclNlY3JldDogdGhpcy5zZWNyZXQsXG4gICAgICBhcHBsaWNhdGlvbjogdGhpcy5tdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uLFxuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIHRhcmdldDogdGhpcyxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciBhIERhdGFiYXNlSW5zdGFuY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VJbnN0YW5jZVByb3BzIGV4dGVuZHMgRGF0YWJhc2VJbnN0YW5jZVNvdXJjZVByb3BzIHtcbiAgLyoqXG4gICAqIENyZWRlbnRpYWxzIGZvciB0aGUgYWRtaW5pc3RyYXRpdmUgdXNlclxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgdXNlcm5hbWUgb2YgJ2FkbWluJyAob3IgJ3Bvc3RncmVzJyBmb3IgUG9zdGdyZVNRTCkgYW5kIFNlY3JldHNNYW5hZ2VyLWdlbmVyYXRlZCBwYXNzd29yZFxuICAgKi9cbiAgcmVhZG9ubHkgY3JlZGVudGlhbHM/OiBDcmVkZW50aWFscztcblxuICAvKipcbiAgICogRm9yIHN1cHBvcnRlZCBlbmdpbmVzLCBzcGVjaWZpZXMgdGhlIGNoYXJhY3RlciBzZXQgdG8gYXNzb2NpYXRlIHdpdGggdGhlXG4gICAqIERCIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJEUyBkZWZhdWx0IGNoYXJhY3RlciBzZXQgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgY2hhcmFjdGVyU2V0TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIERCIGluc3RhbmNlIGlzIGVuY3J5cHRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIHN0b3JhZ2VFbmNyeXB0aW9uS2V5IGhhcyBiZWVuIHByb3ZpZGVkLCBmYWxzZSBvdGhlcndpc2VcbiAgICovXG4gIHJlYWRvbmx5IHN0b3JhZ2VFbmNyeXB0ZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgS01TIGtleSB0aGF0J3MgdXNlZCB0byBlbmNyeXB0IHRoZSBEQiBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG1hc3RlciBrZXkgaWYgc3RvcmFnZUVuY3J5cHRlZCBpcyB0cnVlLCBubyBrZXkgb3RoZXJ3aXNlXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlRW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xufVxuXG4vKipcbiAqIEEgZGF0YWJhc2UgaW5zdGFuY2VcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpSRFM6OkRCSW5zdGFuY2VcbiAqL1xuZXhwb3J0IGNsYXNzIERhdGFiYXNlSW5zdGFuY2UgZXh0ZW5kcyBEYXRhYmFzZUluc3RhbmNlU291cmNlIGltcGxlbWVudHMgSURhdGFiYXNlSW5zdGFuY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VJZGVudGlmaWVyOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBkYkluc3RhbmNlRW5kcG9pbnRBZGRyZXNzOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBkYkluc3RhbmNlRW5kcG9pbnRQb3J0OiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBpbnN0YW5jZUVuZHBvaW50OiBFbmRwb2ludDtcbiAgcHVibGljIHJlYWRvbmx5IHNlY3JldD86IHNlY3JldHNtYW5hZ2VyLklTZWNyZXQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERhdGFiYXNlSW5zdGFuY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgY29uc3QgY3JlZGVudGlhbHMgPSByZW5kZXJDcmVkZW50aWFscyh0aGlzLCBwcm9wcy5lbmdpbmUsIHByb3BzLmNyZWRlbnRpYWxzKTtcbiAgICBjb25zdCBzZWNyZXQgPSBjcmVkZW50aWFscy5zZWNyZXQ7XG5cbiAgICBjb25zdCBpbnN0YW5jZSA9IG5ldyBDZm5EQkluc3RhbmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIC4uLnRoaXMuc291cmNlQ2ZuUHJvcHMsXG4gICAgICBjaGFyYWN0ZXJTZXROYW1lOiBwcm9wcy5jaGFyYWN0ZXJTZXROYW1lLFxuICAgICAga21zS2V5SWQ6IHByb3BzLnN0b3JhZ2VFbmNyeXB0aW9uS2V5ICYmIHByb3BzLnN0b3JhZ2VFbmNyeXB0aW9uS2V5LmtleUFybixcbiAgICAgIG1hc3RlclVzZXJuYW1lOiBjcmVkZW50aWFscy51c2VybmFtZSxcbiAgICAgIG1hc3RlclVzZXJQYXNzd29yZDogY3JlZGVudGlhbHMucGFzc3dvcmQ/LnVuc2FmZVVud3JhcCgpLFxuICAgICAgc3RvcmFnZUVuY3J5cHRlZDogcHJvcHMuc3RvcmFnZUVuY3J5cHRpb25LZXkgPyB0cnVlIDogcHJvcHMuc3RvcmFnZUVuY3J5cHRlZCxcbiAgICB9KTtcblxuICAgIHRoaXMuaW5zdGFuY2VJZGVudGlmaWVyID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUoaW5zdGFuY2UucmVmKTtcbiAgICB0aGlzLmRiSW5zdGFuY2VFbmRwb2ludEFkZHJlc3MgPSBpbnN0YW5jZS5hdHRyRW5kcG9pbnRBZGRyZXNzO1xuICAgIHRoaXMuZGJJbnN0YW5jZUVuZHBvaW50UG9ydCA9IGluc3RhbmNlLmF0dHJFbmRwb2ludFBvcnQ7XG5cbiAgICAvLyBjcmVhdGUgYSBudW1iZXIgdG9rZW4gdGhhdCByZXByZXNlbnRzIHRoZSBwb3J0IG9mIHRoZSBpbnN0YW5jZVxuICAgIGNvbnN0IHBvcnRBdHRyaWJ1dGUgPSBUb2tlbi5hc051bWJlcihpbnN0YW5jZS5hdHRyRW5kcG9pbnRQb3J0KTtcbiAgICB0aGlzLmluc3RhbmNlRW5kcG9pbnQgPSBuZXcgRW5kcG9pbnQoaW5zdGFuY2UuYXR0ckVuZHBvaW50QWRkcmVzcywgcG9ydEF0dHJpYnV0ZSk7XG5cbiAgICBpbnN0YW5jZS5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSA/PyBSZW1vdmFsUG9saWN5LlNOQVBTSE9UKTtcblxuICAgIGlmIChzZWNyZXQpIHtcbiAgICAgIHRoaXMuc2VjcmV0ID0gc2VjcmV0LmF0dGFjaCh0aGlzKTtcbiAgICB9XG5cbiAgICB0aGlzLnNldExvZ1JldGVudGlvbigpO1xuICB9XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgRGF0YWJhc2VJbnN0YW5jZUZyb21TbmFwc2hvdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZUluc3RhbmNlRnJvbVNuYXBzaG90UHJvcHMgZXh0ZW5kcyBEYXRhYmFzZUluc3RhbmNlU291cmNlUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb3IgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgb2YgdGhlIERCIHNuYXBzaG90IHRoYXQncyB1c2VkIHRvXG4gICAqIHJlc3RvcmUgdGhlIERCIGluc3RhbmNlLiBJZiB5b3UncmUgcmVzdG9yaW5nIGZyb20gYSBzaGFyZWQgbWFudWFsIERCXG4gICAqIHNuYXBzaG90LCB5b3UgbXVzdCBzcGVjaWZ5IHRoZSBBUk4gb2YgdGhlIHNuYXBzaG90LlxuICAgKi9cbiAgcmVhZG9ubHkgc25hcHNob3RJZGVudGlmaWVyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE1hc3RlciB1c2VyIGNyZWRlbnRpYWxzLlxuICAgKlxuICAgKiBOb3RlIC0gSXQgaXMgbm90IHBvc3NpYmxlIHRvIGNoYW5nZSB0aGUgbWFzdGVyIHVzZXJuYW1lIGZvciBhIHNuYXBzaG90O1xuICAgKiBob3dldmVyLCBpdCBpcyBwb3NzaWJsZSB0byBwcm92aWRlIChvciBnZW5lcmF0ZSkgYSBuZXcgcGFzc3dvcmQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIGV4aXN0aW5nIHVzZXJuYW1lIGFuZCBwYXNzd29yZCBmcm9tIHRoZSBzbmFwc2hvdCB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBjcmVkZW50aWFscz86IFNuYXBzaG90Q3JlZGVudGlhbHM7XG59XG5cbi8qKlxuICogQSBkYXRhYmFzZSBpbnN0YW5jZSByZXN0b3JlZCBmcm9tIGEgc25hcHNob3QuXG4gKlxuICogQHJlc291cmNlIEFXUzo6UkRTOjpEQkluc3RhbmNlXG4gKi9cbmV4cG9ydCBjbGFzcyBEYXRhYmFzZUluc3RhbmNlRnJvbVNuYXBzaG90IGV4dGVuZHMgRGF0YWJhc2VJbnN0YW5jZVNvdXJjZSBpbXBsZW1lbnRzIElEYXRhYmFzZUluc3RhbmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlSWRlbnRpZmllcjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50QWRkcmVzczogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50UG9ydDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VFbmRwb2ludDogRW5kcG9pbnQ7XG4gIHB1YmxpYyByZWFkb25seSBzZWNyZXQ/OiBzZWNyZXRzbWFuYWdlci5JU2VjcmV0O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEYXRhYmFzZUluc3RhbmNlRnJvbVNuYXBzaG90UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIGxldCBjcmVkZW50aWFscyA9IHByb3BzLmNyZWRlbnRpYWxzO1xuICAgIGxldCBzZWNyZXQgPSBjcmVkZW50aWFscz8uc2VjcmV0O1xuICAgIGlmICghc2VjcmV0ICYmIGNyZWRlbnRpYWxzPy5nZW5lcmF0ZVBhc3N3b3JkKSB7XG4gICAgICBpZiAoIWNyZWRlbnRpYWxzLnVzZXJuYW1lKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYGNyZWRlbnRpYWxzYCBgdXNlcm5hbWVgIG11c3QgYmUgc3BlY2lmaWVkIHdoZW4gYGdlbmVyYXRlUGFzc3dvcmRgIGlzIHNldCB0byB0cnVlJyk7XG4gICAgICB9XG5cbiAgICAgIHNlY3JldCA9IG5ldyBEYXRhYmFzZVNlY3JldCh0aGlzLCAnU2VjcmV0Jywge1xuICAgICAgICB1c2VybmFtZTogY3JlZGVudGlhbHMudXNlcm5hbWUsXG4gICAgICAgIGVuY3J5cHRpb25LZXk6IGNyZWRlbnRpYWxzLmVuY3J5cHRpb25LZXksXG4gICAgICAgIGV4Y2x1ZGVDaGFyYWN0ZXJzOiBjcmVkZW50aWFscy5leGNsdWRlQ2hhcmFjdGVycyxcbiAgICAgICAgcmVwbGFjZU9uUGFzc3dvcmRDcml0ZXJpYUNoYW5nZXM6IGNyZWRlbnRpYWxzLnJlcGxhY2VPblBhc3N3b3JkQ3JpdGVyaWFDaGFuZ2VzLFxuICAgICAgICByZXBsaWNhUmVnaW9uczogY3JlZGVudGlhbHMucmVwbGljYVJlZ2lvbnMsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBpbnN0YW5jZSA9IG5ldyBDZm5EQkluc3RhbmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIC4uLnRoaXMuc291cmNlQ2ZuUHJvcHMsXG4gICAgICBkYlNuYXBzaG90SWRlbnRpZmllcjogcHJvcHMuc25hcHNob3RJZGVudGlmaWVyLFxuICAgICAgbWFzdGVyVXNlclBhc3N3b3JkOiBzZWNyZXQ/LnNlY3JldFZhbHVlRnJvbUpzb24oJ3Bhc3N3b3JkJyk/LnVuc2FmZVVud3JhcCgpID8/IGNyZWRlbnRpYWxzPy5wYXNzd29yZD8udW5zYWZlVW53cmFwKCksIC8vIFNhZmUgdXNhZ2VcbiAgICB9KTtcblxuICAgIHRoaXMuaW5zdGFuY2VJZGVudGlmaWVyID0gaW5zdGFuY2UucmVmO1xuICAgIHRoaXMuZGJJbnN0YW5jZUVuZHBvaW50QWRkcmVzcyA9IGluc3RhbmNlLmF0dHJFbmRwb2ludEFkZHJlc3M7XG4gICAgdGhpcy5kYkluc3RhbmNlRW5kcG9pbnRQb3J0ID0gaW5zdGFuY2UuYXR0ckVuZHBvaW50UG9ydDtcblxuICAgIC8vIGNyZWF0ZSBhIG51bWJlciB0b2tlbiB0aGF0IHJlcHJlc2VudHMgdGhlIHBvcnQgb2YgdGhlIGluc3RhbmNlXG4gICAgY29uc3QgcG9ydEF0dHJpYnV0ZSA9IFRva2VuLmFzTnVtYmVyKGluc3RhbmNlLmF0dHJFbmRwb2ludFBvcnQpO1xuICAgIHRoaXMuaW5zdGFuY2VFbmRwb2ludCA9IG5ldyBFbmRwb2ludChpbnN0YW5jZS5hdHRyRW5kcG9pbnRBZGRyZXNzLCBwb3J0QXR0cmlidXRlKTtcblxuICAgIGluc3RhbmNlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5ID8/IFJlbW92YWxQb2xpY3kuU05BUFNIT1QpO1xuXG4gICAgaWYgKHNlY3JldCkge1xuICAgICAgdGhpcy5zZWNyZXQgPSBzZWNyZXQuYXR0YWNoKHRoaXMpO1xuICAgIH1cblxuICAgIHRoaXMuc2V0TG9nUmV0ZW50aW9uKCk7XG4gIH1cbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBEYXRhYmFzZUluc3RhbmNlUmVhZFJlcGxpY2EuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VJbnN0YW5jZVJlYWRSZXBsaWNhUHJvcHMgZXh0ZW5kcyBEYXRhYmFzZUluc3RhbmNlTmV3UHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGNvbXB1dGUgYW5kIG1lbW9yeSBjYXBhY2l0eSBjbGFzc2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBlYzIuSW5zdGFuY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgc291cmNlIGRhdGFiYXNlIGluc3RhbmNlLlxuICAgKlxuICAgKiBFYWNoIERCIGluc3RhbmNlIGNhbiBoYXZlIGEgbGltaXRlZCBudW1iZXIgb2YgcmVhZCByZXBsaWNhcy4gRm9yIG1vcmVcbiAgICogaW5mb3JtYXRpb24sIHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9EZXZlbG9wZXJHdWlkZS9VU0VSX1JlYWRSZXBsLmh0bWwuXG4gICAqXG4gICAqL1xuICByZWFkb25seSBzb3VyY2VEYXRhYmFzZUluc3RhbmNlOiBJRGF0YWJhc2VJbnN0YW5jZTtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIERCIGluc3RhbmNlIGlzIGVuY3J5cHRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIHN0b3JhZ2VFbmNyeXB0aW9uS2V5IGhhcyBiZWVuIHByb3ZpZGVkLCBmYWxzZSBvdGhlcndpc2VcbiAgICovXG4gIHJlYWRvbmx5IHN0b3JhZ2VFbmNyeXB0ZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgS01TIGtleSB0aGF0J3MgdXNlZCB0byBlbmNyeXB0IHRoZSBEQiBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IG1hc3RlciBrZXkgaWYgc3RvcmFnZUVuY3J5cHRlZCBpcyB0cnVlLCBubyBrZXkgb3RoZXJ3aXNlXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlRW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xufVxuXG4vKipcbiAqIEEgcmVhZCByZXBsaWNhIGRhdGFiYXNlIGluc3RhbmNlLlxuICpcbiAqIEByZXNvdXJjZSBBV1M6OlJEUzo6REJJbnN0YW5jZVxuICovXG5leHBvcnQgY2xhc3MgRGF0YWJhc2VJbnN0YW5jZVJlYWRSZXBsaWNhIGV4dGVuZHMgRGF0YWJhc2VJbnN0YW5jZU5ldyBpbXBsZW1lbnRzIElEYXRhYmFzZUluc3RhbmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlSWRlbnRpZmllcjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50QWRkcmVzczogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZGJJbnN0YW5jZUVuZHBvaW50UG9ydDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VFbmRwb2ludDogRW5kcG9pbnQ7XG4gIHB1YmxpYyByZWFkb25seSBlbmdpbmU/OiBJSW5zdGFuY2VFbmdpbmUgPSB1bmRlZmluZWQ7XG4gIHByb3RlY3RlZCByZWFkb25seSBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERhdGFiYXNlSW5zdGFuY2VSZWFkUmVwbGljYVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICBpZiAocHJvcHMuc291cmNlRGF0YWJhc2VJbnN0YW5jZS5lbmdpbmVcbiAgICAgICAgJiYgIXByb3BzLnNvdXJjZURhdGFiYXNlSW5zdGFuY2UuZW5naW5lLnN1cHBvcnRzUmVhZFJlcGxpY2FCYWNrdXBzXG4gICAgICAgICYmIHByb3BzLmJhY2t1cFJldGVudGlvbikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3Qgc2V0ICdiYWNrdXBSZXRlbnRpb24nLCBhcyBlbmdpbmUgJyR7ZW5naW5lRGVzY3JpcHRpb24ocHJvcHMuc291cmNlRGF0YWJhc2VJbnN0YW5jZS5lbmdpbmUpfScgZG9lcyBub3Qgc3VwcG9ydCBhdXRvbWF0aWMgYmFja3VwcyBmb3IgcmVhZCByZXBsaWNhc2ApO1xuICAgIH1cblxuICAgIC8vIFRoZSByZWFkIHJlcGxpY2EgaW5zdGFuY2UgYWx3YXlzIHVzZXMgdGhlIHNhbWUgZW5naW5lIGFzIHRoZSBzb3VyY2UgaW5zdGFuY2VcbiAgICAvLyBidXQgc29tZSBDRiB2YWxpZGF0aW9ucyByZXF1aXJlIHRoZSBlbmdpbmUgdG8gYmUgZXhwbGljaXRlbHkgcGFzc2VkIHdoZW4gc29tZVxuICAgIC8vIHByb3BlcnRpZXMgYXJlIHNwZWNpZmllZC5cbiAgICBjb25zdCBzaG91bGRQYXNzRW5naW5lID0gcHJvcHMuZG9tYWluICE9IG51bGw7XG5cbiAgICBjb25zdCBpbnN0YW5jZSA9IG5ldyBDZm5EQkluc3RhbmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIC4uLnRoaXMubmV3Q2ZuUHJvcHMsXG4gICAgICAvLyB0aGlzIG11c3QgYmUgQVJOLCBub3QgSUQsIGJlY2F1c2Ugb2YgaHR0cHM6Ly9naXRodWIuY29tL3RlcnJhZm9ybS1wcm92aWRlcnMvdGVycmFmb3JtLXByb3ZpZGVyLWF3cy9pc3N1ZXMvNTI4I2lzc3VlY29tbWVudC0zOTExNjkwMTJcbiAgICAgIHNvdXJjZURiSW5zdGFuY2VJZGVudGlmaWVyOiBwcm9wcy5zb3VyY2VEYXRhYmFzZUluc3RhbmNlLmluc3RhbmNlQXJuLFxuICAgICAga21zS2V5SWQ6IHByb3BzLnN0b3JhZ2VFbmNyeXB0aW9uS2V5Py5rZXlBcm4sXG4gICAgICBzdG9yYWdlRW5jcnlwdGVkOiBwcm9wcy5zdG9yYWdlRW5jcnlwdGlvbktleSA/IHRydWUgOiBwcm9wcy5zdG9yYWdlRW5jcnlwdGVkLFxuICAgICAgZW5naW5lOiBzaG91bGRQYXNzRW5naW5lID8gcHJvcHMuc291cmNlRGF0YWJhc2VJbnN0YW5jZS5lbmdpbmU/LmVuZ2luZVR5cGUgOiB1bmRlZmluZWQsXG4gICAgfSk7XG5cbiAgICB0aGlzLmluc3RhbmNlVHlwZSA9IHByb3BzLmluc3RhbmNlVHlwZTtcbiAgICB0aGlzLmluc3RhbmNlSWRlbnRpZmllciA9IGluc3RhbmNlLnJlZjtcbiAgICB0aGlzLmRiSW5zdGFuY2VFbmRwb2ludEFkZHJlc3MgPSBpbnN0YW5jZS5hdHRyRW5kcG9pbnRBZGRyZXNzO1xuICAgIHRoaXMuZGJJbnN0YW5jZUVuZHBvaW50UG9ydCA9IGluc3RhbmNlLmF0dHJFbmRwb2ludFBvcnQ7XG5cbiAgICAvLyBjcmVhdGUgYSBudW1iZXIgdG9rZW4gdGhhdCByZXByZXNlbnRzIHRoZSBwb3J0IG9mIHRoZSBpbnN0YW5jZVxuICAgIGNvbnN0IHBvcnRBdHRyaWJ1dGUgPSBUb2tlbi5hc051bWJlcihpbnN0YW5jZS5hdHRyRW5kcG9pbnRQb3J0KTtcbiAgICB0aGlzLmluc3RhbmNlRW5kcG9pbnQgPSBuZXcgRW5kcG9pbnQoaW5zdGFuY2UuYXR0ckVuZHBvaW50QWRkcmVzcywgcG9ydEF0dHJpYnV0ZSk7XG5cbiAgICBpbnN0YW5jZS5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSA/PyBSZW1vdmFsUG9saWN5LlNOQVBTSE9UKTtcblxuICAgIHRoaXMuc2V0TG9nUmV0ZW50aW9uKCk7XG4gIH1cbn1cblxuLyoqXG4gKiBSZW5kZXJzIHRoZSBwcm9jZXNzb3IgZmVhdHVyZXMgc3BlY2lmaWNhdGlvbnNcbiAqXG4gKiBAcGFyYW0gZmVhdHVyZXMgdGhlIHByb2Nlc3NvciBmZWF0dXJlc1xuICovXG5mdW5jdGlvbiByZW5kZXJQcm9jZXNzb3JGZWF0dXJlcyhmZWF0dXJlczogUHJvY2Vzc29yRmVhdHVyZXMpOiBDZm5EQkluc3RhbmNlLlByb2Nlc3NvckZlYXR1cmVQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgZmVhdHVyZXNMaXN0ID0gT2JqZWN0LmVudHJpZXMoZmVhdHVyZXMpLm1hcCgoW25hbWUsIHZhbHVlXSkgPT4gKHsgbmFtZSwgdmFsdWU6IHZhbHVlLnRvU3RyaW5nKCkgfSkpO1xuXG4gIHJldHVybiBmZWF0dXJlc0xpc3QubGVuZ3RoID09PSAwID8gdW5kZWZpbmVkIDogZmVhdHVyZXNMaXN0O1xufVxuXG5mdW5jdGlvbiBkZWZhdWx0SW9wcyhzdG9yYWdlVHlwZTogU3RvcmFnZVR5cGUsIGlvcHM/OiBudW1iZXIpOiBudW1iZXIgfCB1bmRlZmluZWQge1xuICBzd2l0Y2ggKHN0b3JhZ2VUeXBlKSB7XG4gICAgY2FzZSBTdG9yYWdlVHlwZS5TVEFOREFSRDpcbiAgICBjYXNlIFN0b3JhZ2VUeXBlLkdQMjpcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgY2FzZSBTdG9yYWdlVHlwZS5HUDM6XG4gICAgICByZXR1cm4gaW9wcztcbiAgICBjYXNlIFN0b3JhZ2VUeXBlLklPMTpcbiAgICAgIHJldHVybiBpb3BzID8/IDEwMDA7XG4gIH1cbn1cbiJdfQ==