"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyDefaultRotationOptions = exports.renderUnless = exports.helperRemovalPolicy = exports.renderCredentials = exports.defaultDeletionProtection = exports.engineDescription = exports.setupS3ImportExport = exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = void 0;
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const database_secret_1 = require("../database-secret");
const props_1 = require("../props");
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 *
 * This constant is private to the RDS module.
 */
exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    return engine.engineType + (engine.engineVersion?.fullVersion ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection ?? (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
/**
 * Renders the credentials for an instance or cluster
 */
function renderCredentials(scope, engine, credentials) {
    let renderedCredentials = credentials ?? props_1.Credentials.fromUsername(engine.defaultUsername ?? 'admin'); // For backwards compatibilty
    if (!renderedCredentials.secret && !renderedCredentials.password) {
        renderedCredentials = props_1.Credentials.fromSecret(new database_secret_1.DatabaseSecret(scope, 'Secret', {
            username: renderedCredentials.username,
            secretName: renderedCredentials.secretName,
            encryptionKey: renderedCredentials.encryptionKey,
            excludeCharacters: renderedCredentials.excludeCharacters,
            // if username must be referenced as a string we can safely replace the
            // secret when customization options are changed without risking a replacement
            replaceOnPasswordCriteriaChanges: credentials?.usernameAsString,
            replicaRegions: renderedCredentials.replicaRegions,
        }), 
        // pass username if it must be referenced as a string
        credentials?.usernameAsString ? renderedCredentials.username : undefined);
    }
    return renderedCredentials;
}
exports.renderCredentials = renderCredentials;
/**
 * The RemovalPolicy that should be applied to a "helper" resource, if the base resource has the given removal policy
 *
 * - For Clusters, this determines the RemovalPolicy for Instances/SubnetGroups.
 * - For Instances, this determines the RemovalPolicy for SubnetGroups.
 *
 * If the basePolicy is:
 *
 *  DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
 *  RETAIN              -> RETAIN  (anything else will lose data or fail to deploy)
 *  (undefined)         -> DESTROY (base policy is assumed to be SNAPSHOT)
 */
function helperRemovalPolicy(basePolicy) {
    return basePolicy === core_1.RemovalPolicy.RETAIN
        ? core_1.RemovalPolicy.RETAIN
        : core_1.RemovalPolicy.DESTROY;
}
exports.helperRemovalPolicy = helperRemovalPolicy;
/**
 * Return a given value unless it's the same as another value
 */
function renderUnless(value, suppressValue) {
    return value === suppressValue ? undefined : value;
}
exports.renderUnless = renderUnless;
/**
 * Applies defaults for rotation options
 */
function applyDefaultRotationOptions(options, defaultvpcSubnets) {
    return {
        excludeCharacters: exports.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        vpcSubnets: defaultvpcSubnets,
        ...options,
    };
}
exports.applyDefaultRotationOptions = applyDefaultRotationOptions;
//# sourceMappingURL=data:application/json;base64,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