"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const secretsmanager = require("../../aws-secretsmanager");
const cdk = require("../../core");
const cxapi = require("../../cx-api");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SessionPinningFilter[_a] = { fqn: "aws-cdk-lib.aws_rds.SessionPinningFilter", version: "2.74.0" };
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_IDatabaseInstance(instance);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromInstance);
            }
            throw error;
        }
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_IDatabaseCluster(cluster);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromCluster);
            }
            throw error;
        }
        return new ProxyTarget(undefined, cluster);
    }
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(proxy) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseProxy(proxy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const engine = this.dbInstance?.engine ?? this.dbCluster?.engine;
        if (!engine) {
            const errorResource = this.dbCluster ?? this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource?.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${(0, util_1.engineDescription)(engine)}' does not support proxies`);
        }
        // allow connecting to the Cluster/Instance from the Proxy
        this.dbCluster?.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Cluster from the Proxy');
        this.dbInstance?.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Instance from the Proxy');
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
_b = JSII_RTTI_SYMBOL_1;
ProxyTarget[_b] = { fqn: "aws-cdk-lib.aws_rds.ProxyTarget", version: "2.74.0" };
exports.ProxyTarget = ProxyTarget;
/**
 * Represents an RDS Database Proxy.
 *
 */
class DatabaseProxyBase extends cdk.Resource {
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            throw new Error('For imported Database Proxies, the dbUser is required in grantConnect()');
        }
        const scopeStack = cdk.Stack.of(this);
        const proxyGeneratedId = scopeStack.splitArn(this.dbProxyArn, cdk.ArnFormat.COLON_RESOURCE_NAME).resourceName;
        const userArn = scopeStack.formatArn({
            service: 'rds-db',
            resource: 'dbuser',
            resourceName: `${proxyGeneratedId}/${dbUser}`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [userArn],
        });
    }
}
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends DatabaseProxyBase {
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseProxyAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseProxyAttributes);
            }
            throw error;
        }
        class Import extends DatabaseProxyBase {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_rds_DatabaseProxyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseProxy);
            }
            throw error;
        }
        const physicalName = props.dbProxyName || (cdk.FeatureFlags.of(this).isEnabled(cxapi.DATABASE_PROXY_UNIQUE_RESOURCE_NAME) ?
            cdk.Names.uniqueResourceName(this, { maxLength: 60 }) : id);
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        const securityGroups = props.securityGroups ?? [
            new ec2.SecurityGroup(this, 'ProxySecurityGroup', {
                description: 'SecurityGroup for Database Proxy',
                vpc: props.vpc,
            }),
        ];
        this.connections = new ec2.Connections({ securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.secrets = props.secrets;
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: props.idleClientTimeout?.toSeconds(),
            requireTls: props.requireTLS ?? true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(_ => _.securityGroupId) }),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        bindResult.dbClusters?.forEach((c) => proxyTargetGroup.node.addDependency(c));
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
    grantConnect(grantee, dbUser) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantConnect);
            }
            throw error;
        }
        if (!dbUser) {
            if (this.secrets.length > 1) {
                throw new Error('When the Proxy contains multiple Secrets, you must pass a dbUser explicitly to grantConnect()');
            }
            // 'username' is the field RDS uses here,
            // see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy-secrets-arns
            dbUser = this.secrets[0].secretValueFromJson('username').unsafeUnwrap();
        }
        return super.grantConnect(grantee, dbUser);
    }
}
_c = JSII_RTTI_SYMBOL_1;
DatabaseProxy[_c] = { fqn: "aws-cdk-lib.aws_rds.DatabaseProxy", version: "2.74.0" };
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    return {
        connectionBorrowTimeout: props.borrowTimeout?.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: props.sessionPinningFilters?.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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