"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpsRedirect = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_certificatemanager_1 = require("../../aws-certificatemanager");
const aws_cloudfront_1 = require("../../aws-cloudfront");
const aws_route53_1 = require("../../aws-route53");
const aws_route53_targets_1 = require("../../aws-route53-targets");
const aws_s3_1 = require("../../aws-s3");
const core_1 = require("../../core");
const helpers_internal_1 = require("../../core/lib/helpers-internal");
const cx_api_1 = require("../../cx-api");
const constructs_1 = require("constructs");
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_route53_patterns_HttpsRedirectProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, HttpsRedirect);
            }
            throw error;
        }
        const domainNames = props.recordNames ?? [props.zone.zoneName];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).splitArn(props.certificate.certificateArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`The certificate must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const redirectCert = props.certificate ?? this.createCertificate(domainNames, props.zone);
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            viewerCertificate: aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(redirectCert, {
                aliases: domainNames,
            }),
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = (0, helpers_internal_1.md5hash)(domainName).slice(0, 6);
            const aliasProps = {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            };
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, aliasProps);
            new aws_route53_1.AaaaRecord(this, `RedirectAliasRecordSix${hash}`, aliasProps);
        });
    }
    /**
     * Gets the stack to use for creating the Certificate
     * If the current stack is not in `us-east-1` then this
     * will create a new `us-east-1` stack.
     *
     * CloudFront is a global resource which you can create (via CloudFormation) from
     * _any_ region. So I could create a CloudFront distribution in `us-east-2` if I wanted
     * to (maybe the rest of my application lives there). The problem is that some supporting resources
     * that CloudFront uses (i.e. ACM Certificates) are required to exist in `us-east-1`. This means
     * that if I want to create a CloudFront distribution in `us-east-2` I still need to create a ACM certificate in
     * `us-east-1`.
     *
     * In order to do this correctly we need to know which region the CloudFront distribution is being created in.
     * We have two options, either require the user to specify the region or make an assumption if they do not.
     * This implementation requires the user specify the region.
     */
    certificateScope() {
        const stack = core_1.Stack.of(this);
        const parent = stack.node.scope;
        if (!parent) {
            throw new Error(`Stack ${stack.stackId} must be created in the scope of an App or Stage`);
        }
        if (core_1.Token.isUnresolved(stack.region)) {
            throw new Error(`When ${cx_api_1.ROUTE53_PATTERNS_USE_CERTIFICATE} is enabled, a region must be defined on the Stack`);
        }
        if (stack.region !== 'us-east-1') {
            const stackId = `certificate-redirect-stack-${stack.node.addr}`;
            const certStack = parent.node.tryFindChild(stackId);
            return certStack ?? new core_1.Stack(parent, stackId, {
                env: { region: 'us-east-1', account: stack.account },
            });
        }
        return this;
    }
    /**
     * Creates a certificate.
     *
     * If the `ROUTE53_PATTERNS_USE_CERTIFICATE` feature flag is set then
     * this will use the `Certificate` class otherwise it will use the
     * `DnsValidatedCertificate` class
     *
     * This is also safe to upgrade since the new certificate will be created and updated
     * on the CloudFront distribution before the old one is deleted.
     */
    createCertificate(domainNames, zone) {
        const useCertificate = core_1.FeatureFlags.of(this).isEnabled(cx_api_1.ROUTE53_PATTERNS_USE_CERTIFICATE);
        if (useCertificate) {
            // this preserves backwards compatibility. Previously the certificate was always created in `this` scope
            // so we need to keep the name the same
            const id = (this.certificateScope() === this) ? 'RedirectCertificate' : 'RedirectCertificate' + this.node.addr;
            return new aws_certificatemanager_1.Certificate(this.certificateScope(), id, {
                domainName: domainNames[0],
                subjectAlternativeNames: domainNames,
                validation: aws_certificatemanager_1.CertificateValidation.fromDns(zone),
            });
        }
        else {
            return new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
                domainName: domainNames[0],
                subjectAlternativeNames: domainNames,
                hostedZone: zone,
                region: 'us-east-1',
            });
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
HttpsRedirect[_a] = { fqn: "aws-cdk-lib.aws_route53_patterns.HttpsRedirect", version: "2.74.0" };
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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