"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
async function handler(event) {
    const resourceProps = event.ResourceProperties;
    switch (event.RequestType) {
        case 'Create':
        case 'Update':
            return cfnEventHandler(resourceProps, false);
        case 'Delete':
            return cfnEventHandler(resourceProps, true);
    }
}
exports.handler = handler;
async function cfnEventHandler(props, isDeleteEvent) {
    const { AssumeRoleArn, ParentZoneId, ParentZoneName, DelegatedZoneName, DelegatedZoneNameServers, TTL, UseRegionalStsEndpoint } = props;
    if (!ParentZoneId && !ParentZoneName) {
        throw Error('One of ParentZoneId or ParentZoneName must be specified');
    }
    const credentials = await getCrossAccountCredentials(AssumeRoleArn, !!UseRegionalStsEndpoint);
    const route53 = new aws_sdk_1.Route53({ credentials });
    const parentZoneId = ParentZoneId ?? await getHostedZoneIdByName(ParentZoneName, route53);
    await route53.changeResourceRecordSets({
        HostedZoneId: parentZoneId,
        ChangeBatch: {
            Changes: [{
                    Action: isDeleteEvent ? 'DELETE' : 'UPSERT',
                    ResourceRecordSet: {
                        Name: DelegatedZoneName,
                        Type: 'NS',
                        TTL,
                        ResourceRecords: DelegatedZoneNameServers.map(ns => ({ Value: ns })),
                    },
                }],
        },
    }).promise();
}
async function getCrossAccountCredentials(roleArn, regionalEndpoint) {
    const sts = new aws_sdk_1.STS(regionalEndpoint ? { stsRegionalEndpoints: 'regional' } : {});
    const timestamp = (new Date()).getTime();
    const { Credentials: assumedCredentials } = await sts
        .assumeRole({
        RoleArn: roleArn,
        RoleSessionName: `cross-account-zone-delegation-${timestamp}`,
    })
        .promise();
    if (!assumedCredentials) {
        throw Error('Error getting assume role credentials');
    }
    return new aws_sdk_1.Credentials({
        accessKeyId: assumedCredentials.AccessKeyId,
        secretAccessKey: assumedCredentials.SecretAccessKey,
        sessionToken: assumedCredentials.SessionToken,
    });
}
async function getHostedZoneIdByName(name, route53) {
    const zones = await route53.listHostedZonesByName({ DNSName: name }).promise();
    const matchedZones = zones.HostedZones.filter(zone => zone.Name === `${name}.`);
    if (matchedZones.length !== 1) {
        throw Error(`Expected one hosted zone to match the given name but found ${matchedZones.length}`);
    }
    return matchedZones[0].Id;
}
//# sourceMappingURL=data:application/json;base64,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