"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcEndpointServiceDomainName = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const helpers_internal_1 = require("../../core/lib/helpers-internal");
const custom_resources_1 = require("../../custom-resources");
const constructs_1 = require("constructs");
const lib_1 = require("../lib");
/**
 * A Private DNS configuration for a VPC endpoint service.
 */
class VpcEndpointServiceDomainName extends constructs_1.Construct {
    // The way this class works is by using three custom resources and a TxtRecord in conjunction
    // The first custom resource tells the VPC endpoint service to use the given DNS name
    // The VPC endpoint service will then say:
    // "ok, create a TXT record using these two values to prove you own the domain"
    // The second custom resource retrieves these two values from the service
    // The TxtRecord is created from these two values
    // The third custom resource tells the VPC Endpoint Service to verify the domain ownership
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_route53_VpcEndpointServiceDomainNameProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, VpcEndpointServiceDomainName);
            }
            throw error;
        }
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        const serviceId = props.endpointService.vpcEndpointServiceId;
        this.domainName = props.domainName;
        // Make sure a user doesn't accidentally add multiple domains
        this.validateProps(props);
        VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId] = this.domainName;
        VpcEndpointServiceDomainName.endpointServices.push(props.endpointService);
        // Enable Private DNS on the endpoint service and retrieve the AWS-generated configuration
        const privateDnsConfiguration = this.getPrivateDnsConfiguration(serviceUniqueId, serviceId, this.domainName);
        // Tell AWS to verify that this account owns the domain attached to the service
        this.verifyPrivateDnsConfiguration(privateDnsConfiguration, props.publicHostedZone);
        // Finally, don't do any of the above before the endpoint service is created
        this.node.addDependency(props.endpointService);
    }
    validateProps(props) {
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        if (serviceUniqueId in VpcEndpointServiceDomainName.endpointServicesMap) {
            const endpoint = VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId];
            throw new Error(`Cannot create a VpcEndpointServiceDomainName for service ${serviceUniqueId}, another VpcEndpointServiceDomainName (${endpoint}) is already associated with it`);
        }
    }
    /**
     * Sets up Custom Resources to make AWS calls to set up Private DNS on an endpoint service,
     * returning the values to use in a TxtRecord, which AWS uses to verify domain ownership.
     */
    getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName) {
        // The custom resource which tells AWS to enable Private DNS on the given service, using the given domain name
        // AWS will generate a name/value pair for use in a TxtRecord, which is used to verify domain ownership.
        const enablePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                PrivateDnsName: privateDnsName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(serviceUniqueId),
        };
        const removePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                RemovePrivateDnsName: true,
            },
        };
        const enable = new custom_resources_1.AwsCustomResource(this, 'EnableDns', {
            onCreate: enablePrivateDnsAction,
            onUpdate: enablePrivateDnsAction,
            onDelete: removePrivateDnsAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            serviceId,
                        ]),
                    ]),
                ],
            }),
            // APIs are available in 2.1055.0
            installLatestAwsSdk: false,
        });
        // Look up the name/value pair if the domain changes, or the service changes,
        // which would cause the values to be different. If the unique ID changes,
        // the resource may be entirely recreated, so we will need to look it up again.
        const lookup = hashcode(core_1.Names.uniqueId(this) + serviceUniqueId + privateDnsName);
        // Create the custom resource to look up the name/value pair generated by AWS
        // after the previous API call
        const retrieveNameValuePairAction = {
            service: 'EC2',
            action: 'describeVpcEndpointServiceConfigurations',
            parameters: {
                ServiceIds: [serviceId],
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(lookup),
        };
        const getNames = new custom_resources_1.AwsCustomResource(this, 'GetNames', {
            onCreate: retrieveNameValuePairAction,
            onUpdate: retrieveNameValuePairAction,
            // describeVpcEndpointServiceConfigurations can't take an ARN for granular permissions
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            installLatestAwsSdk: false,
        });
        // We only want to call and get the name/value pair after we've told AWS to enable Private DNS
        // If we call before then, we'll get an empty pair of values.
        getNames.node.addDependency(enable);
        // Get the references to the name/value pair associated with the endpoint service
        const name = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Name');
        const value = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Value');
        return { name, value, serviceId };
    }
    /**
     * Creates a Route53 entry and a Custom Resource which explicitly tells AWS to verify ownership
     * of the domain name attached to an endpoint service.
     */
    verifyPrivateDnsConfiguration(config, publicHostedZone) {
        // Create the TXT record in the provided hosted zone
        const verificationRecord = new lib_1.TxtRecord(this, 'DnsVerificationRecord', {
            recordName: config.name,
            values: [config.value],
            zone: publicHostedZone,
        });
        // Tell the endpoint service to verify the domain ownership
        const startVerificationAction = {
            service: 'EC2',
            action: 'startVpcEndpointServicePrivateDnsVerification',
            parameters: {
                ServiceId: config.serviceId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(core_1.Fn.join(':', [config.name, config.value])),
        };
        const startVerification = new custom_resources_1.AwsCustomResource(this, 'StartVerification', {
            onCreate: startVerificationAction,
            onUpdate: startVerificationAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            config.serviceId,
                        ]),
                    ]),
                ],
            }),
            installLatestAwsSdk: false,
        });
        // Only verify after the record has been created
        startVerification.node.addDependency(verificationRecord);
    }
}
_a = JSII_RTTI_SYMBOL_1;
VpcEndpointServiceDomainName[_a] = { fqn: "aws-cdk-lib.aws_route53.VpcEndpointServiceDomainName", version: "2.74.0" };
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServices = [];
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServicesMap = {};
exports.VpcEndpointServiceDomainName = VpcEndpointServiceDomainName;
/**
 * Hash a string
 */
function hashcode(s) {
    return (0, helpers_internal_1.md5hash)(s);
}
;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBjLWVuZHBvaW50LXNlcnZpY2UtZG9tYWluLW5hbWUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ2cGMtZW5kcG9pbnQtc2VydmljZS1kb21haW4tbmFtZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFDQSxxQ0FBOEM7QUFDOUMsc0VBQTBEO0FBQzFELDZEQUF3RztBQUN4RywyQ0FBdUM7QUFDdkMsZ0NBQXNEO0FBNEJ0RDs7R0FFRztBQUNILE1BQWEsNEJBQTZCLFNBQVEsc0JBQVM7SUFhekQsNkZBQTZGO0lBQzdGLHFGQUFxRjtJQUNyRiwwQ0FBMEM7SUFDMUMsK0VBQStFO0lBQy9FLHlFQUF5RTtJQUN6RSxpREFBaUQ7SUFDakQsMEZBQTBGO0lBQzFGLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0M7UUFDaEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7Ozs7OytDQXJCUiw0QkFBNEI7Ozs7UUF1QnJDLE1BQU0sZUFBZSxHQUFHLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RSxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDLG9CQUFvQixDQUFDO1FBQzdELElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUVuQyw2REFBNkQ7UUFDN0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQiw0QkFBNEIsQ0FBQyxtQkFBbUIsQ0FBQyxlQUFlLENBQUMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDO1FBQ3BGLDRCQUE0QixDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFMUUsMEZBQTBGO1FBQzFGLE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxDQUFDLDBCQUEwQixDQUFDLGVBQWUsRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRTdHLCtFQUErRTtRQUMvRSxJQUFJLENBQUMsNkJBQTZCLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFFcEYsNEVBQTRFO1FBQzVFLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQztLQUNoRDtJQUVPLGFBQWEsQ0FBQyxLQUF3QztRQUM1RCxNQUFNLGVBQWUsR0FBRyxZQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkUsSUFBSSxlQUFlLElBQUksNEJBQTRCLENBQUMsbUJBQW1CLEVBQUU7WUFDdkUsTUFBTSxRQUFRLEdBQUcsNEJBQTRCLENBQUMsbUJBQW1CLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDbkYsTUFBTSxJQUFJLEtBQUssQ0FDYiw0REFBNEQsZUFBZSwyQ0FBMkMsUUFBUSxpQ0FBaUMsQ0FBQyxDQUFDO1NBQ3BLO0tBQ0Y7SUFFRDs7O09BR0c7SUFDSywwQkFBMEIsQ0FBQyxlQUF1QixFQUFFLFNBQWlCLEVBQUUsY0FBc0I7UUFFbkcsOEdBQThHO1FBQzlHLHdHQUF3RztRQUN4RyxNQUFNLHNCQUFzQixHQUFHO1lBQzdCLE9BQU8sRUFBRSxLQUFLO1lBQ2QsTUFBTSxFQUFFLHVDQUF1QztZQUMvQyxVQUFVLEVBQUU7Z0JBQ1YsU0FBUyxFQUFFLFNBQVM7Z0JBQ3BCLGNBQWMsRUFBRSxjQUFjO2FBQy9CO1lBQ0Qsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQztTQUMzRCxDQUFDO1FBQ0YsTUFBTSxzQkFBc0IsR0FBRztZQUM3QixPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSx1Q0FBdUM7WUFDL0MsVUFBVSxFQUFFO2dCQUNWLFNBQVMsRUFBRSxTQUFTO2dCQUNwQixvQkFBb0IsRUFBRSxJQUFJO2FBQzNCO1NBQ0YsQ0FBQztRQUNGLE1BQU0sTUFBTSxHQUFHLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUN0RCxRQUFRLEVBQUUsc0JBQXNCO1lBQ2hDLFFBQVEsRUFBRSxzQkFBc0I7WUFDaEMsUUFBUSxFQUFFLHNCQUFzQjtZQUNoQyxNQUFNLEVBQUUsMENBQXVCLENBQUMsWUFBWSxDQUFDO2dCQUMzQyxTQUFTLEVBQUU7b0JBQ1QsU0FBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7d0JBQ1gsS0FBSzt3QkFDTCxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVM7d0JBQ3hCLEtBQUs7d0JBQ0wsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNO3dCQUNyQixZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU87d0JBQ3RCLFNBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFOzRCQUNYLHNCQUFzQjs0QkFDdEIsU0FBUzt5QkFDVixDQUFDO3FCQUNILENBQUM7aUJBQ0g7YUFDRixDQUFDO1lBQ0YsaUNBQWlDO1lBQ2pDLG1CQUFtQixFQUFFLEtBQUs7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsNkVBQTZFO1FBQzdFLDBFQUEwRTtRQUMxRSwrRUFBK0U7UUFDL0UsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsZUFBZSxHQUFHLGNBQWMsQ0FBQyxDQUFDO1FBRWpGLDZFQUE2RTtRQUM3RSw4QkFBOEI7UUFDOUIsTUFBTSwyQkFBMkIsR0FBRztZQUNsQyxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSwwQ0FBMEM7WUFDbEQsVUFBVSxFQUFFO2dCQUNWLFVBQVUsRUFBRSxDQUFDLFNBQVMsQ0FBQzthQUN4QjtZQUNELGtCQUFrQixFQUFFLHFDQUFrQixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUM7U0FDbEQsQ0FBQztRQUNGLE1BQU0sUUFBUSxHQUFHLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN2RCxRQUFRLEVBQUUsMkJBQTJCO1lBQ3JDLFFBQVEsRUFBRSwyQkFBMkI7WUFDckMsc0ZBQXNGO1lBQ3RGLE1BQU0sRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZLENBQUM7Z0JBQzNDLFNBQVMsRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZO2FBQ2hELENBQUM7WUFDRixtQkFBbUIsRUFBRSxLQUFLO1NBQzNCLENBQUMsQ0FBQztRQUVILDhGQUE4RjtRQUM5Riw2REFBNkQ7UUFDN0QsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFcEMsaUZBQWlGO1FBQ2pGLE1BQU0sSUFBSSxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQywwREFBMEQsQ0FBQyxDQUFDO1FBQ25HLE1BQU0sS0FBSyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1FBRXJHLE9BQU8sRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDO0tBQ25DO0lBRUQ7OztPQUdHO0lBQ0ssNkJBQTZCLENBQUMsTUFBK0IsRUFBRSxnQkFBbUM7UUFDeEcsb0RBQW9EO1FBQ3BELE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO1lBQ3RFLFVBQVUsRUFBRSxNQUFNLENBQUMsSUFBSTtZQUN2QixNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDO1lBQ3RCLElBQUksRUFBRSxnQkFBZ0I7U0FDdkIsQ0FBQyxDQUFDO1FBRUgsMkRBQTJEO1FBQzNELE1BQU0sdUJBQXVCLEdBQUc7WUFDOUIsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNLEVBQUUsK0NBQStDO1lBQ3ZELFVBQVUsRUFBRTtnQkFDVixTQUFTLEVBQUUsTUFBTSxDQUFDLFNBQVM7YUFDNUI7WUFDRCxrQkFBa0IsRUFBRSxxQ0FBa0IsQ0FBQyxFQUFFLENBQUMsU0FBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ3JGLENBQUM7UUFDRixNQUFNLGlCQUFpQixHQUFHLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ3pFLFFBQVEsRUFBRSx1QkFBdUI7WUFDakMsUUFBUSxFQUFFLHVCQUF1QjtZQUNqQyxNQUFNLEVBQUUsMENBQXVCLENBQUMsWUFBWSxDQUFDO2dCQUMzQyxTQUFTLEVBQUU7b0JBQ1QsU0FBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7d0JBQ1gsS0FBSzt3QkFDTCxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVM7d0JBQ3hCLEtBQUs7d0JBQ0wsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNO3dCQUNyQixZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU87d0JBQ3RCLFNBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFOzRCQUNYLHNCQUFzQjs0QkFDdEIsTUFBTSxDQUFDLFNBQVM7eUJBQ2pCLENBQUM7cUJBQ0gsQ0FBQztpQkFDSDthQUNGLENBQUM7WUFDRixtQkFBbUIsRUFBRSxLQUFLO1NBQzNCLENBQUMsQ0FBQztRQUNILGdEQUFnRDtRQUNoRCxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGtCQUFrQixDQUFDLENBQUM7S0FDMUQ7Ozs7QUFqTEQsOEdBQThHO0FBQ3RGLDZDQUFnQixHQUEwQixFQUFFLENBQUM7QUFFckUsOEdBQThHO0FBQ3RGLGdEQUFtQixHQUF5QyxFQUFFLENBQUM7QUFONUUsb0VBQTRCO0FBK0x6Qzs7R0FFRztBQUNILFNBQVMsUUFBUSxDQUFDLENBQVM7SUFDekIsT0FBTyxJQUFBLDBCQUFPLEVBQUMsQ0FBQyxDQUFDLENBQUM7QUFDcEIsQ0FBQztBQUFBLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJVnBjRW5kcG9pbnRTZXJ2aWNlIH0gZnJvbSAnLi4vLi4vYXdzLWVjMic7XG5pbXBvcnQgeyBGbiwgTmFtZXMsIFN0YWNrIH0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBtZDVoYXNoIH0gZnJvbSAnLi4vLi4vY29yZS9saWIvaGVscGVycy1pbnRlcm5hbCc7XG5pbXBvcnQgeyBBd3NDdXN0b21SZXNvdXJjZSwgQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3ksIFBoeXNpY2FsUmVzb3VyY2VJZCB9IGZyb20gJy4uLy4uL2N1c3RvbS1yZXNvdXJjZXMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJUHVibGljSG9zdGVkWm9uZSwgVHh0UmVjb3JkIH0gZnJvbSAnLi4vbGliJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIGNvbmZpZ3VyZSBhIFZQQyBFbmRwb2ludCBTZXJ2aWNlIGRvbWFpbiBuYW1lXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVnBjRW5kcG9pbnRTZXJ2aWNlRG9tYWluTmFtZVByb3BzIHtcblxuICAvKipcbiAgICogVGhlIFZQQyBFbmRwb2ludCBTZXJ2aWNlIHRvIGNvbmZpZ3VyZSBQcml2YXRlIEROUyBmb3JcbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50U2VydmljZTogSVZwY0VuZHBvaW50U2VydmljZTtcblxuICAvKipcbiAgICogVGhlIGRvbWFpbiBuYW1lIHRvIHVzZS5cbiAgICpcbiAgICogVGhpcyBkb21haW4gbmFtZSBtdXN0IGJlIG93bmVkIGJ5IHRoaXMgYWNjb3VudCAocmVnaXN0ZXJlZCB0aHJvdWdoIFJvdXRlNTMpLFxuICAgKiBvciBkZWxlZ2F0ZWQgdG8gdGhpcyBhY2NvdW50LiBEb21haW4gb3duZXJzaGlwIHdpbGwgYmUgdmVyaWZpZWQgYnkgQVdTIGJlZm9yZVxuICAgKiBwcml2YXRlIEROUyBjYW4gYmUgdXNlZC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdnBjL2xhdGVzdC91c2VyZ3VpZGUvZW5kcG9pbnQtc2VydmljZXMtZG5zLXZhbGlkYXRpb24uaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcHVibGljIGhvc3RlZCB6b25lIHRvIHVzZSBmb3IgdGhlIGRvbWFpbi5cbiAgICovXG4gIHJlYWRvbmx5IHB1YmxpY0hvc3RlZFpvbmU6IElQdWJsaWNIb3N0ZWRab25lO1xufVxuXG4vKipcbiAqIEEgUHJpdmF0ZSBETlMgY29uZmlndXJhdGlvbiBmb3IgYSBWUEMgZW5kcG9pbnQgc2VydmljZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFZwY0VuZHBvaW50U2VydmljZURvbWFpbk5hbWUgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIC8vIFRyYWNrIGFsbCBkb21haW4gbmFtZXMgY3JlYXRlZCwgc28gc29tZW9uZSBkb2Vzbid0IGFjY2lkZW50YWxseSBhc3NvY2lhdGUgdHdvIGRvbWFpbnMgd2l0aCBhIHNpbmdsZSBzZXJ2aWNlXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IGVuZHBvaW50U2VydmljZXM6IElWcGNFbmRwb2ludFNlcnZpY2VbXSA9IFtdO1xuXG4gIC8vIFRyYWNrIGFsbCBkb21haW4gbmFtZXMgY3JlYXRlZCwgc28gc29tZW9uZSBkb2Vzbid0IGFjY2lkZW50YWxseSBhc3NvY2lhdGUgdHdvIGRvbWFpbnMgd2l0aCBhIHNpbmdsZSBzZXJ2aWNlXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IGVuZHBvaW50U2VydmljZXNNYXA6IHsgW2VuZHBvaW50U2VydmljZTogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuXG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIG5hbWUgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcml2YXRlIEROUyBjb25maWd1cmF0aW9uXG4gICAqL1xuICBwdWJsaWMgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIC8vIFRoZSB3YXkgdGhpcyBjbGFzcyB3b3JrcyBpcyBieSB1c2luZyB0aHJlZSBjdXN0b20gcmVzb3VyY2VzIGFuZCBhIFR4dFJlY29yZCBpbiBjb25qdW5jdGlvblxuICAvLyBUaGUgZmlyc3QgY3VzdG9tIHJlc291cmNlIHRlbGxzIHRoZSBWUEMgZW5kcG9pbnQgc2VydmljZSB0byB1c2UgdGhlIGdpdmVuIEROUyBuYW1lXG4gIC8vIFRoZSBWUEMgZW5kcG9pbnQgc2VydmljZSB3aWxsIHRoZW4gc2F5OlxuICAvLyBcIm9rLCBjcmVhdGUgYSBUWFQgcmVjb3JkIHVzaW5nIHRoZXNlIHR3byB2YWx1ZXMgdG8gcHJvdmUgeW91IG93biB0aGUgZG9tYWluXCJcbiAgLy8gVGhlIHNlY29uZCBjdXN0b20gcmVzb3VyY2UgcmV0cmlldmVzIHRoZXNlIHR3byB2YWx1ZXMgZnJvbSB0aGUgc2VydmljZVxuICAvLyBUaGUgVHh0UmVjb3JkIGlzIGNyZWF0ZWQgZnJvbSB0aGVzZSB0d28gdmFsdWVzXG4gIC8vIFRoZSB0aGlyZCBjdXN0b20gcmVzb3VyY2UgdGVsbHMgdGhlIFZQQyBFbmRwb2ludCBTZXJ2aWNlIHRvIHZlcmlmeSB0aGUgZG9tYWluIG93bmVyc2hpcFxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVnBjRW5kcG9pbnRTZXJ2aWNlRG9tYWluTmFtZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHNlcnZpY2VVbmlxdWVJZCA9IE5hbWVzLm5vZGVVbmlxdWVJZChwcm9wcy5lbmRwb2ludFNlcnZpY2Uubm9kZSk7XG4gICAgY29uc3Qgc2VydmljZUlkID0gcHJvcHMuZW5kcG9pbnRTZXJ2aWNlLnZwY0VuZHBvaW50U2VydmljZUlkO1xuICAgIHRoaXMuZG9tYWluTmFtZSA9IHByb3BzLmRvbWFpbk5hbWU7XG5cbiAgICAvLyBNYWtlIHN1cmUgYSB1c2VyIGRvZXNuJ3QgYWNjaWRlbnRhbGx5IGFkZCBtdWx0aXBsZSBkb21haW5zXG4gICAgdGhpcy52YWxpZGF0ZVByb3BzKHByb3BzKTtcblxuICAgIFZwY0VuZHBvaW50U2VydmljZURvbWFpbk5hbWUuZW5kcG9pbnRTZXJ2aWNlc01hcFtzZXJ2aWNlVW5pcXVlSWRdID0gdGhpcy5kb21haW5OYW1lO1xuICAgIFZwY0VuZHBvaW50U2VydmljZURvbWFpbk5hbWUuZW5kcG9pbnRTZXJ2aWNlcy5wdXNoKHByb3BzLmVuZHBvaW50U2VydmljZSk7XG5cbiAgICAvLyBFbmFibGUgUHJpdmF0ZSBETlMgb24gdGhlIGVuZHBvaW50IHNlcnZpY2UgYW5kIHJldHJpZXZlIHRoZSBBV1MtZ2VuZXJhdGVkIGNvbmZpZ3VyYXRpb25cbiAgICBjb25zdCBwcml2YXRlRG5zQ29uZmlndXJhdGlvbiA9IHRoaXMuZ2V0UHJpdmF0ZURuc0NvbmZpZ3VyYXRpb24oc2VydmljZVVuaXF1ZUlkLCBzZXJ2aWNlSWQsIHRoaXMuZG9tYWluTmFtZSk7XG5cbiAgICAvLyBUZWxsIEFXUyB0byB2ZXJpZnkgdGhhdCB0aGlzIGFjY291bnQgb3ducyB0aGUgZG9tYWluIGF0dGFjaGVkIHRvIHRoZSBzZXJ2aWNlXG4gICAgdGhpcy52ZXJpZnlQcml2YXRlRG5zQ29uZmlndXJhdGlvbihwcml2YXRlRG5zQ29uZmlndXJhdGlvbiwgcHJvcHMucHVibGljSG9zdGVkWm9uZSk7XG5cbiAgICAvLyBGaW5hbGx5LCBkb24ndCBkbyBhbnkgb2YgdGhlIGFib3ZlIGJlZm9yZSB0aGUgZW5kcG9pbnQgc2VydmljZSBpcyBjcmVhdGVkXG4gICAgdGhpcy5ub2RlLmFkZERlcGVuZGVuY3kocHJvcHMuZW5kcG9pbnRTZXJ2aWNlKTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyhwcm9wczogVnBjRW5kcG9pbnRTZXJ2aWNlRG9tYWluTmFtZVByb3BzKTogdm9pZCB7XG4gICAgY29uc3Qgc2VydmljZVVuaXF1ZUlkID0gTmFtZXMubm9kZVVuaXF1ZUlkKHByb3BzLmVuZHBvaW50U2VydmljZS5ub2RlKTtcbiAgICBpZiAoc2VydmljZVVuaXF1ZUlkIGluIFZwY0VuZHBvaW50U2VydmljZURvbWFpbk5hbWUuZW5kcG9pbnRTZXJ2aWNlc01hcCkge1xuICAgICAgY29uc3QgZW5kcG9pbnQgPSBWcGNFbmRwb2ludFNlcnZpY2VEb21haW5OYW1lLmVuZHBvaW50U2VydmljZXNNYXBbc2VydmljZVVuaXF1ZUlkXTtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgYENhbm5vdCBjcmVhdGUgYSBWcGNFbmRwb2ludFNlcnZpY2VEb21haW5OYW1lIGZvciBzZXJ2aWNlICR7c2VydmljZVVuaXF1ZUlkfSwgYW5vdGhlciBWcGNFbmRwb2ludFNlcnZpY2VEb21haW5OYW1lICgke2VuZHBvaW50fSkgaXMgYWxyZWFkeSBhc3NvY2lhdGVkIHdpdGggaXRgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogU2V0cyB1cCBDdXN0b20gUmVzb3VyY2VzIHRvIG1ha2UgQVdTIGNhbGxzIHRvIHNldCB1cCBQcml2YXRlIEROUyBvbiBhbiBlbmRwb2ludCBzZXJ2aWNlLFxuICAgKiByZXR1cm5pbmcgdGhlIHZhbHVlcyB0byB1c2UgaW4gYSBUeHRSZWNvcmQsIHdoaWNoIEFXUyB1c2VzIHRvIHZlcmlmeSBkb21haW4gb3duZXJzaGlwLlxuICAgKi9cbiAgcHJpdmF0ZSBnZXRQcml2YXRlRG5zQ29uZmlndXJhdGlvbihzZXJ2aWNlVW5pcXVlSWQ6IHN0cmluZywgc2VydmljZUlkOiBzdHJpbmcsIHByaXZhdGVEbnNOYW1lOiBzdHJpbmcpOiBQcml2YXRlRG5zQ29uZmlndXJhdGlvbiB7XG5cbiAgICAvLyBUaGUgY3VzdG9tIHJlc291cmNlIHdoaWNoIHRlbGxzIEFXUyB0byBlbmFibGUgUHJpdmF0ZSBETlMgb24gdGhlIGdpdmVuIHNlcnZpY2UsIHVzaW5nIHRoZSBnaXZlbiBkb21haW4gbmFtZVxuICAgIC8vIEFXUyB3aWxsIGdlbmVyYXRlIGEgbmFtZS92YWx1ZSBwYWlyIGZvciB1c2UgaW4gYSBUeHRSZWNvcmQsIHdoaWNoIGlzIHVzZWQgdG8gdmVyaWZ5IGRvbWFpbiBvd25lcnNoaXAuXG4gICAgY29uc3QgZW5hYmxlUHJpdmF0ZURuc0FjdGlvbiA9IHtcbiAgICAgIHNlcnZpY2U6ICdFQzInLFxuICAgICAgYWN0aW9uOiAnbW9kaWZ5VnBjRW5kcG9pbnRTZXJ2aWNlQ29uZmlndXJhdGlvbicsXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIFNlcnZpY2VJZDogc2VydmljZUlkLFxuICAgICAgICBQcml2YXRlRG5zTmFtZTogcHJpdmF0ZURuc05hbWUsXG4gICAgICB9LFxuICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQub2Yoc2VydmljZVVuaXF1ZUlkKSxcbiAgICB9O1xuICAgIGNvbnN0IHJlbW92ZVByaXZhdGVEbnNBY3Rpb24gPSB7XG4gICAgICBzZXJ2aWNlOiAnRUMyJyxcbiAgICAgIGFjdGlvbjogJ21vZGlmeVZwY0VuZHBvaW50U2VydmljZUNvbmZpZ3VyYXRpb24nLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBTZXJ2aWNlSWQ6IHNlcnZpY2VJZCxcbiAgICAgICAgUmVtb3ZlUHJpdmF0ZURuc05hbWU6IHRydWUsXG4gICAgICB9LFxuICAgIH07XG4gICAgY29uc3QgZW5hYmxlID0gbmV3IEF3c0N1c3RvbVJlc291cmNlKHRoaXMsICdFbmFibGVEbnMnLCB7XG4gICAgICBvbkNyZWF0ZTogZW5hYmxlUHJpdmF0ZURuc0FjdGlvbixcbiAgICAgIG9uVXBkYXRlOiBlbmFibGVQcml2YXRlRG5zQWN0aW9uLFxuICAgICAgb25EZWxldGU6IHJlbW92ZVByaXZhdGVEbnNBY3Rpb24sXG4gICAgICBwb2xpY3k6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LmZyb21TZGtDYWxscyh7XG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIEZuLmpvaW4oJzonLCBbXG4gICAgICAgICAgICAnYXJuJyxcbiAgICAgICAgICAgIFN0YWNrLm9mKHRoaXMpLnBhcnRpdGlvbixcbiAgICAgICAgICAgICdlYzInLFxuICAgICAgICAgICAgU3RhY2sub2YodGhpcykucmVnaW9uLFxuICAgICAgICAgICAgU3RhY2sub2YodGhpcykuYWNjb3VudCxcbiAgICAgICAgICAgIEZuLmpvaW4oJy8nLCBbXG4gICAgICAgICAgICAgICd2cGMtZW5kcG9pbnQtc2VydmljZScsXG4gICAgICAgICAgICAgIHNlcnZpY2VJZCxcbiAgICAgICAgICAgIF0pLFxuICAgICAgICAgIF0pLFxuICAgICAgICBdLFxuICAgICAgfSksXG4gICAgICAvLyBBUElzIGFyZSBhdmFpbGFibGUgaW4gMi4xMDU1LjBcbiAgICAgIGluc3RhbGxMYXRlc3RBd3NTZGs6IGZhbHNlLFxuICAgIH0pO1xuXG4gICAgLy8gTG9vayB1cCB0aGUgbmFtZS92YWx1ZSBwYWlyIGlmIHRoZSBkb21haW4gY2hhbmdlcywgb3IgdGhlIHNlcnZpY2UgY2hhbmdlcyxcbiAgICAvLyB3aGljaCB3b3VsZCBjYXVzZSB0aGUgdmFsdWVzIHRvIGJlIGRpZmZlcmVudC4gSWYgdGhlIHVuaXF1ZSBJRCBjaGFuZ2VzLFxuICAgIC8vIHRoZSByZXNvdXJjZSBtYXkgYmUgZW50aXJlbHkgcmVjcmVhdGVkLCBzbyB3ZSB3aWxsIG5lZWQgdG8gbG9vayBpdCB1cCBhZ2Fpbi5cbiAgICBjb25zdCBsb29rdXAgPSBoYXNoY29kZShOYW1lcy51bmlxdWVJZCh0aGlzKSArIHNlcnZpY2VVbmlxdWVJZCArIHByaXZhdGVEbnNOYW1lKTtcblxuICAgIC8vIENyZWF0ZSB0aGUgY3VzdG9tIHJlc291cmNlIHRvIGxvb2sgdXAgdGhlIG5hbWUvdmFsdWUgcGFpciBnZW5lcmF0ZWQgYnkgQVdTXG4gICAgLy8gYWZ0ZXIgdGhlIHByZXZpb3VzIEFQSSBjYWxsXG4gICAgY29uc3QgcmV0cmlldmVOYW1lVmFsdWVQYWlyQWN0aW9uID0ge1xuICAgICAgc2VydmljZTogJ0VDMicsXG4gICAgICBhY3Rpb246ICdkZXNjcmliZVZwY0VuZHBvaW50U2VydmljZUNvbmZpZ3VyYXRpb25zJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgU2VydmljZUlkczogW3NlcnZpY2VJZF0sXG4gICAgICB9LFxuICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQub2YobG9va3VwKSxcbiAgICB9O1xuICAgIGNvbnN0IGdldE5hbWVzID0gbmV3IEF3c0N1c3RvbVJlc291cmNlKHRoaXMsICdHZXROYW1lcycsIHtcbiAgICAgIG9uQ3JlYXRlOiByZXRyaWV2ZU5hbWVWYWx1ZVBhaXJBY3Rpb24sXG4gICAgICBvblVwZGF0ZTogcmV0cmlldmVOYW1lVmFsdWVQYWlyQWN0aW9uLFxuICAgICAgLy8gZGVzY3JpYmVWcGNFbmRwb2ludFNlcnZpY2VDb25maWd1cmF0aW9ucyBjYW4ndCB0YWtlIGFuIEFSTiBmb3IgZ3JhbnVsYXIgcGVybWlzc2lvbnNcbiAgICAgIHBvbGljeTogQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3kuZnJvbVNka0NhbGxzKHtcbiAgICAgICAgcmVzb3VyY2VzOiBBd3NDdXN0b21SZXNvdXJjZVBvbGljeS5BTllfUkVTT1VSQ0UsXG4gICAgICB9KSxcbiAgICAgIGluc3RhbGxMYXRlc3RBd3NTZGs6IGZhbHNlLFxuICAgIH0pO1xuXG4gICAgLy8gV2Ugb25seSB3YW50IHRvIGNhbGwgYW5kIGdldCB0aGUgbmFtZS92YWx1ZSBwYWlyIGFmdGVyIHdlJ3ZlIHRvbGQgQVdTIHRvIGVuYWJsZSBQcml2YXRlIEROU1xuICAgIC8vIElmIHdlIGNhbGwgYmVmb3JlIHRoZW4sIHdlJ2xsIGdldCBhbiBlbXB0eSBwYWlyIG9mIHZhbHVlcy5cbiAgICBnZXROYW1lcy5ub2RlLmFkZERlcGVuZGVuY3koZW5hYmxlKTtcblxuICAgIC8vIEdldCB0aGUgcmVmZXJlbmNlcyB0byB0aGUgbmFtZS92YWx1ZSBwYWlyIGFzc29jaWF0ZWQgd2l0aCB0aGUgZW5kcG9pbnQgc2VydmljZVxuICAgIGNvbnN0IG5hbWUgPSBnZXROYW1lcy5nZXRSZXNwb25zZUZpZWxkKCdTZXJ2aWNlQ29uZmlndXJhdGlvbnMuMC5Qcml2YXRlRG5zTmFtZUNvbmZpZ3VyYXRpb24uTmFtZScpO1xuICAgIGNvbnN0IHZhbHVlID0gZ2V0TmFtZXMuZ2V0UmVzcG9uc2VGaWVsZCgnU2VydmljZUNvbmZpZ3VyYXRpb25zLjAuUHJpdmF0ZURuc05hbWVDb25maWd1cmF0aW9uLlZhbHVlJyk7XG5cbiAgICByZXR1cm4geyBuYW1lLCB2YWx1ZSwgc2VydmljZUlkIH07XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFJvdXRlNTMgZW50cnkgYW5kIGEgQ3VzdG9tIFJlc291cmNlIHdoaWNoIGV4cGxpY2l0bHkgdGVsbHMgQVdTIHRvIHZlcmlmeSBvd25lcnNoaXBcbiAgICogb2YgdGhlIGRvbWFpbiBuYW1lIGF0dGFjaGVkIHRvIGFuIGVuZHBvaW50IHNlcnZpY2UuXG4gICAqL1xuICBwcml2YXRlIHZlcmlmeVByaXZhdGVEbnNDb25maWd1cmF0aW9uKGNvbmZpZzogUHJpdmF0ZURuc0NvbmZpZ3VyYXRpb24sIHB1YmxpY0hvc3RlZFpvbmU6IElQdWJsaWNIb3N0ZWRab25lKSB7XG4gICAgLy8gQ3JlYXRlIHRoZSBUWFQgcmVjb3JkIGluIHRoZSBwcm92aWRlZCBob3N0ZWQgem9uZVxuICAgIGNvbnN0IHZlcmlmaWNhdGlvblJlY29yZCA9IG5ldyBUeHRSZWNvcmQodGhpcywgJ0Ruc1ZlcmlmaWNhdGlvblJlY29yZCcsIHtcbiAgICAgIHJlY29yZE5hbWU6IGNvbmZpZy5uYW1lLFxuICAgICAgdmFsdWVzOiBbY29uZmlnLnZhbHVlXSxcbiAgICAgIHpvbmU6IHB1YmxpY0hvc3RlZFpvbmUsXG4gICAgfSk7XG5cbiAgICAvLyBUZWxsIHRoZSBlbmRwb2ludCBzZXJ2aWNlIHRvIHZlcmlmeSB0aGUgZG9tYWluIG93bmVyc2hpcFxuICAgIGNvbnN0IHN0YXJ0VmVyaWZpY2F0aW9uQWN0aW9uID0ge1xuICAgICAgc2VydmljZTogJ0VDMicsXG4gICAgICBhY3Rpb246ICdzdGFydFZwY0VuZHBvaW50U2VydmljZVByaXZhdGVEbnNWZXJpZmljYXRpb24nLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBTZXJ2aWNlSWQ6IGNvbmZpZy5zZXJ2aWNlSWQsXG4gICAgICB9LFxuICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQub2YoRm4uam9pbignOicsIFtjb25maWcubmFtZSwgY29uZmlnLnZhbHVlXSkpLFxuICAgIH07XG4gICAgY29uc3Qgc3RhcnRWZXJpZmljYXRpb24gPSBuZXcgQXdzQ3VzdG9tUmVzb3VyY2UodGhpcywgJ1N0YXJ0VmVyaWZpY2F0aW9uJywge1xuICAgICAgb25DcmVhdGU6IHN0YXJ0VmVyaWZpY2F0aW9uQWN0aW9uLFxuICAgICAgb25VcGRhdGU6IHN0YXJ0VmVyaWZpY2F0aW9uQWN0aW9uLFxuICAgICAgcG9saWN5OiBBd3NDdXN0b21SZXNvdXJjZVBvbGljeS5mcm9tU2RrQ2FsbHMoe1xuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICBGbi5qb2luKCc6JywgW1xuICAgICAgICAgICAgJ2FybicsXG4gICAgICAgICAgICBTdGFjay5vZih0aGlzKS5wYXJ0aXRpb24sXG4gICAgICAgICAgICAnZWMyJyxcbiAgICAgICAgICAgIFN0YWNrLm9mKHRoaXMpLnJlZ2lvbixcbiAgICAgICAgICAgIFN0YWNrLm9mKHRoaXMpLmFjY291bnQsXG4gICAgICAgICAgICBGbi5qb2luKCcvJywgW1xuICAgICAgICAgICAgICAndnBjLWVuZHBvaW50LXNlcnZpY2UnLFxuICAgICAgICAgICAgICBjb25maWcuc2VydmljZUlkLFxuICAgICAgICAgICAgXSksXG4gICAgICAgICAgXSksXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICAgIGluc3RhbGxMYXRlc3RBd3NTZGs6IGZhbHNlLFxuICAgIH0pO1xuICAgIC8vIE9ubHkgdmVyaWZ5IGFmdGVyIHRoZSByZWNvcmQgaGFzIGJlZW4gY3JlYXRlZFxuICAgIHN0YXJ0VmVyaWZpY2F0aW9uLm5vZGUuYWRkRGVwZW5kZW5jeSh2ZXJpZmljYXRpb25SZWNvcmQpO1xuICB9XG59XG5cbi8qKlxuICogUmVwcmVzZW50IHRoZSBuYW1lL3ZhbHVlIHBhaXIgYXNzb2NpYXRlZCB3aXRoIGEgUHJpdmF0ZSBETlMgZW5hYmxlZCBlbmRwb2ludCBzZXJ2aWNlXG4gKi9cbmludGVyZmFjZSBQcml2YXRlRG5zQ29uZmlndXJhdGlvbiB7XG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbiAgcmVhZG9ubHkgc2VydmljZUlkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogSGFzaCBhIHN0cmluZ1xuICovXG5mdW5jdGlvbiBoYXNoY29kZShzOiBzdHJpbmcpOiBzdHJpbmcge1xuICByZXR1cm4gbWQ1aGFzaChzKTtcbn07Il19