"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Asset = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const kms = require("../../aws-kms");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
const compat_1 = require("./compat");
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_assets_AssetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Asset);
            }
            throw error;
        }
        this.isBundled = props.bundling != null;
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: props.followSymlinks ?? (0, compat_1.toSymlinkFollow)(props.follow),
            assetHash: props.assetHash ?? props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        const stack = cdk.Stack.of(this);
        this.assetPath = staging.relativeStagedPath(stack);
        this.isFile = staging.packaging === cdk.FileAssetPackaging.FILE;
        this.isZipArchive = staging.isArchive;
        const location = stack.synthesizer.addFileAsset({
            packaging: staging.packaging,
            sourceHash: this.sourceHash,
            fileName: this.assetPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        const kmsKey = location.kmsKeyArn ? kms.Key.fromKeyArn(this, 'Key', location.kmsKeyArn) : undefined;
        this.bucket = s3.Bucket.fromBucketAttributes(this, 'AssetBucket', {
            bucketName: this.s3BucketName,
            encryptionKey: kmsKey,
        });
        for (const reader of (props.readers ?? [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_CfnResource(resource);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addResourceMetadata);
            }
            throw error;
        }
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_IS_BUNDLED_KEY] = this.isBundled;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the assets bucket.
     */
    grantRead(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantRead);
            }
            throw error;
        }
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Asset[_a] = { fqn: "aws-cdk-lib.aws_s3_assets.Asset", version: "2.74.0" };
exports.Asset = Asset;
//# sourceMappingURL=data:application/json;base64,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