"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("../../aws-efs");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const lambda_layer_awscli_1 = require("../../lambda-layer-awscli");
const case_1 = require("case");
const constructs_1 = require("constructs");
// tag key has a limit of 128 characters
const CUSTOM_RESOURCE_OWNER_TAG = 'aws-cdk:cr-owned';
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from
 * other S3 buckets or from local disk
 */
class BucketDeployment extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.requestDestinationArn = false;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_deployment_BucketDeploymentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BucketDeployment);
            }
            throw error;
        }
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        this.destinationBucket = props.destinationBucket;
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.ephemeralStorageSize, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_9,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            ephemeralStorageSize: props.ephemeralStorageSize,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
            logRetention: props.logRetention,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        this.handlerRole = handlerRole;
        this.sources = props.sources.map((source) => source.bind(this, { handlerRole: this.handlerRole }));
        this.destinationBucket.grantReadWrite(handler);
        if (props.accessControl) {
            this.destinationBucket.grantPutAcl(handler);
        }
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        // Markers are not replaced if zip sources are not extracted, so throw an error
        // if extraction is not wanted and sources have markers.
        const _this = this;
        this.node.addValidation({
            validate() {
                if (_this.sources.some(source => source.markers) && props.extract == false) {
                    return ['Some sources are incompatible with extract=false; sources with deploy-time values (such as \'snsTopic.topicArn\') must be extracted.'];
                }
                return [];
            },
        });
        const crUniqueId = `CustomResource${this.renderUniqueId(props.memoryLimit, props.ephemeralStorageSize, props.vpc)}`;
        this.cr = new cdk.CustomResource(this, crUniqueId, {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: cdk.Lazy.list({ produce: () => this.sources.map(source => source.bucket.bucketName) }),
                SourceObjectKeys: cdk.Lazy.list({ produce: () => this.sources.map(source => source.zipObjectKey) }),
                SourceMarkers: cdk.Lazy.any({
                    produce: () => {
                        return this.sources.reduce((acc, source) => {
                            if (source.markers) {
                                acc.push(source.markers);
                                // if there are more than 1 source, then all sources
                                // require markers (custom resource will throw an error otherwise)
                            }
                            else if (this.sources.length > 1) {
                                acc.push({});
                            }
                            return acc;
                        }, []);
                    },
                }, { omitEmptyArray: true }),
                DestinationBucketName: this.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Extract: props.extract,
                Prune: props.prune ?? true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution?.distributionId,
                DistributionPaths: props.distributionPaths,
                // Passing through the ARN sequences dependency on the deployment
                DestinationBucketArn: cdk.Lazy.string({ produce: () => this.requestDestinationArn ? this.destinationBucket.bucketArn : undefined }),
            },
        });
        let prefix = props.destinationKeyPrefix ?
            `:${props.destinationKeyPrefix}` :
            '';
        prefix += `:${this.cr.node.addr.slice(-8)}`;
        const tagKey = CUSTOM_RESOURCE_OWNER_TAG + prefix;
        // destinationKeyPrefix can be 104 characters before we hit
        // the tag key limit of 128
        // '/this/is/a/random/key/prefix/that/is/a/lot/of/characters/do/we/think/that/it/will/ever/be/this/long?????'
        // better to throw an error here than wait for CloudFormation to fail
        if (!cdk.Token.isUnresolved(tagKey) && tagKey.length > 128) {
            throw new Error('The BucketDeployment construct requires that the "destinationKeyPrefix" be <=104 characters.');
        }
        /*
         * This will add a tag to the deployment bucket in the format of
         * `aws-cdk:cr-owned:{keyPrefix}:{uniqueHash}`
         *
         * For example:
         * {
         *   Key: 'aws-cdk:cr-owned:deploy/here/:240D17B3',
         *   Value: 'true',
         * }
         *
         * This will allow for scenarios where there is a single S3 Bucket that has multiple
         * BucketDeployment resources deploying to it. Each bucket + keyPrefix can be "owned" by
         * 1 or more BucketDeployment resources. Since there are some scenarios where multiple BucketDeployment
         * resources can deploy to the same bucket and key prefix (e.g. using include/exclude) we
         * also append part of the id to make the key unique.
         *
         * As long as a bucket + keyPrefix is "owned" by a BucketDeployment resource, another CR
         * cannot delete data. There are a couple of scenarios where this comes into play.
         *
         * 1. If the LogicalResourceId of the CustomResource changes (e.g. the crUniqueId changes)
         * CloudFormation will first issue a 'Create' to create the new CustomResource and will
         * update the Tag on the bucket. CloudFormation will then issue a 'Delete' on the old CustomResource
         * and since the new CR "owns" the Bucket+keyPrefix it will not delete the contents of the bucket
         *
         * 2. If the BucketDeployment resource is deleted _and_ it is the only CR for that bucket+keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the
         * CR. Since there are no tags indicating that this bucket+keyPrefix is "owned" then it will delete
         * the contents.
         *
         * 3. If the BucketDeployment resource is deleted _and_ it is *not* the only CR for that bucket:keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the CR.
         * Since there are other CRs that also "own" that bucket+keyPrefix there will still be a tag on the bucket
         * and the contents will not be removed.
         *
         * 4. If the BucketDeployment resource _and_ the S3 Bucket are both removed, then CloudFormation will first
         * issue a "Delete" to the CR and since there is a tag on the bucket the contents will not be removed. If you
         * want the contents of the bucket to be removed on bucket deletion, then `autoDeleteObjects` property should
         * be set to true on the Bucket.
         */
        cdk.Tags.of(this.destinationBucket).add(tagKey, 'true');
    }
    /**
     * The bucket after the deployment
     *
     * If you want to reference the destination bucket in another construct and make sure the
     * bucket deployment has happened before the next operation is started, pass the other construct
     * a reference to `deployment.deployedBucket`.
     *
     * Note that this only returns an immutable reference to the destination bucket.
     * If sequenced access to the original destination bucket is required, you may add a dependency
     * on the bucket deployment instead: `otherResource.node.addDependency(deployment)`
     */
    get deployedBucket() {
        this.requestDestinationArn = true;
        this._deployedBucket = this._deployedBucket ?? s3.Bucket.fromBucketAttributes(this, 'DestinationBucket', {
            bucketArn: cdk.Token.asString(this.cr.getAtt('DestinationBucketArn')),
            region: this.destinationBucket.env.region,
            account: this.destinationBucket.env.account,
            isWebsite: this.destinationBucket.isWebsite,
        });
        return this._deployedBucket;
    }
    /**
     * The object keys for the sources deployed to the S3 bucket.
     *
     * This returns a list of tokenized object keys for source files that are deployed to the bucket.
     *
     * This can be useful when using `BucketDeployment` with `extract` set to `false` and you need to reference
     * the object key that resides in the bucket for that zip source file somewhere else in your CDK
     * application, such as in a CFN output.
     *
     * For example, use `Fn.select(0, myBucketDeployment.objectKeys)` to reference the object key of the
     * first source file in your bucket deployment.
     */
    get objectKeys() {
        const objectKeys = cdk.Token.asList(this.cr.getAtt('SourceObjectKeys'));
        return objectKeys;
    }
    /**
     * Add an additional source to the bucket deployment
     *
     * @example
     * declare const websiteBucket: s3.IBucket;
     * const deployment = new s3deploy.BucketDeployment(this, 'Deployment', {
     *   sources: [s3deploy.Source.asset('./website-dist')],
     *   destinationBucket: websiteBucket,
     * });
     *
     * deployment.addSource(s3deploy.Source.asset('./another-asset'));
     */
    addSource(source) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_deployment_ISource(source);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSource);
            }
            throw error;
        }
        this.sources.push(source.bind(this, { handlerRole: this.handlerRole }));
    }
    renderUniqueId(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '';
        // if the user specifes a custom memory limit, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error("Can't use tokens when specifying 'memoryLimit' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if the user specifies a custom ephemeral storage size, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (ephemeralStorageSize) {
            if (ephemeralStorageSize.isUnresolved()) {
                throw new Error("Can't use tokens when specifying 'ephemeralStorageSize' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${ephemeralStorageSize.toMebibytes().toString()}MiB`;
        }
        // if the user specifies a VPC, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    renderSingletonUuid(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        uuid += this.renderUniqueId(memoryLimit, ephemeralStorageSize, vpc);
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return stack.node.tryFindChild(uuid) ?? new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "aws-cdk-lib.aws_s3_deployment.BucketDeployment", version: "2.74.0" };
exports.BucketDeployment = BucketDeployment;
/**
 * Metadata.
 *
 * The `x-amz-meta-` prefix will automatically be added to keys.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#UserMetadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = (0, case_1.kebab)(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    /**
     * Sets 'must-revalidate'.
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     */
    static maxAge(t) { try {
        jsiiDeprecationWarnings.aws_cdk_lib_Duration(t);
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.maxAge);
        }
        throw error;
    } return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     */
    static sMaxAge(t) { try {
        jsiiDeprecationWarnings.aws_cdk_lib_Duration(t);
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.sMaxAge);
        }
        throw error;
    } return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     */
    static fromString(s) { return new CacheControl(s); }
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
}
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "aws-cdk-lib.aws_s3_deployment.CacheControl", version: "2.74.0" };
exports.CacheControl = CacheControl;
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    /**
     * 'AES256'
     */
    ServerSideEncryption["AES_256"] = "AES256";
    /**
     * 'aws:kms'
     */
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    /**
     * 'STANDARD'
     */
    StorageClass["STANDARD"] = "STANDARD";
    /**
     * 'REDUCED_REDUNDANCY'
     */
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    /**
     * 'STANDARD_IA'
     */
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    /**
     * 'ONEZONE_IA'
     */
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    /**
     * 'INTELLIGENT_TIERING'
     */
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    /**
     * 'GLACIER'
     */
    StorageClass["GLACIER"] = "GLACIER";
    /**
     * 'DEEP_ARCHIVE'
     */
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Create an expiration date from a raw date string.
     */
    static fromString(s) { return new Expires(s); }
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
}
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "aws-cdk-lib.aws_s3_deployment.Expires", version: "2.74.0" };
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,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