"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Source = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path_1 = require("path");
const s3_assets = require("../../aws-s3-assets");
const core_1 = require("../../core");
const render_data_1 = require("./render-data");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *     Source.data('hello/world/file.txt', 'Hello, world!')
 *     Source.dataJson('config.json', { baz: topic.topicArn })
 *     Source.dataYaml('config.yaml', { baz: topic.topicArn })
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     *
     * Make sure you trust the producer of the archive.
     *
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucket(bucket);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bucket);
            }
            throw error;
        }
        return {
            bind: (_, context) => {
                if (!context) {
                    throw new Error('To use a Source.bucket(), context must be provided');
                }
                bucket.grantRead(context.handlerRole);
                return { bucket, zipObjectKey };
            },
        };
    }
    /**
     * Uses a local asset as the deployment source.
     *
     * If the local asset is a .zip archive, make sure you trust the
     * producer of the archive.
     *
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_assets_AssetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.asset);
            }
            throw error;
        }
        return {
            bind(scope, context) {
                if (!context) {
                    throw new Error('To use a Source.asset(), context must be provided');
                }
                let id = 1;
                while (scope.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(scope, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                asset.grantRead(context.handlerRole);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    /**
     * Deploys an object with the specified string contents into the bucket. The
     * content can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * To store a JSON object use `Source.jsonData()`.
     * To store YAML content use `Source.yamlData()`.
     *
     * @param objectKey The destination S3 object key (relative to the root of the
     * S3 deployment).
     * @param data The data to be stored in the object.
     */
    static data(objectKey, data) {
        return {
            bind: (scope, context) => {
                const workdir = core_1.FileSystem.mkdtemp('s3-deployment');
                const outputPath = (0, path_1.join)(workdir, objectKey);
                const rendered = (0, render_data_1.renderData)(scope, data);
                fs.mkdirSync((0, path_1.dirname)(outputPath), { recursive: true });
                fs.writeFileSync(outputPath, rendered.text);
                const asset = this.asset(workdir).bind(scope, context);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.zipObjectKey,
                    markers: rendered.markers,
                };
            },
        };
    }
    /**
     * Deploys an object with the specified JSON object into the bucket. The
     * object can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * @param objectKey The destination S3 object key (relative to the root of the
     * S3 deployment).
     * @param obj A JSON object.
     */
    static jsonData(objectKey, obj) {
        return {
            bind: (scope, context) => {
                return Source.data(objectKey, core_1.Stack.of(scope).toJsonString(obj)).bind(scope, context);
            },
        };
    }
    /**
     * Deploys an object with the specified JSON object formatted as YAML into the bucket.
     * The object can include deploy-time values (such as `snsTopic.topicArn`) that
     * will get resolved only during deployment.
     *
     * @param objectKey The destination S3 object key (relative to the root of the
     * S3 deployment).
     * @param obj A JSON object.
     */
    static yamlData(objectKey, obj) {
        return {
            bind: (scope, context) => {
                return Source.data(objectKey, core_1.Stack.of(scope).toYamlString(obj)).bind(scope, context);
            },
        };
    }
    constructor() { }
}
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "aws-cdk-lib.aws_s3_deployment.Source", version: "2.74.0" };
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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