"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsDestination = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const s3 = require("../../aws-s3");
const core_1 = require("../../core");
/**
 * Use an SQS queue as a bucket notification destination
 */
class SqsDestination {
    constructor(queue) {
        this.queue = queue;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sqs_IQueue(queue);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SqsDestination);
            }
            throw error;
        }
    }
    /**
     * Allows using SQS queues as destinations for bucket notifications.
     * Use `bucket.onEvent(event, queue)` to subscribe.
     */
    bind(_scope, bucket) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucket(bucket);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        this.queue.grantSendMessages(new iam.ServicePrincipal('s3.amazonaws.com', {
            conditions: {
                ArnLike: { 'aws:SourceArn': bucket.bucketArn },
            },
        }));
        // if this queue is encrypted, we need to allow S3 to read messages since that's how
        // it verifies that the notification destination configuration is valid.
        if (this.queue.encryptionMasterKey) {
            const statement = new iam.PolicyStatement({
                principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                actions: ['kms:GenerateDataKey*', 'kms:Decrypt'],
                resources: ['*'],
            });
            const addResult = this.queue.encryptionMasterKey.addToResourcePolicy(statement, /* allowNoOp */ true);
            if (!addResult.statementAdded) {
                core_1.Annotations.of(this.queue.encryptionMasterKey).addWarning(`Can not change key policy of imported kms key. Ensure that your key policy contains the following permissions: \n${JSON.stringify(statement.toJSON(), null, 2)}`);
            }
        }
        return {
            arn: this.queue.queueArn,
            type: s3.BucketNotificationDestinationType.QUEUE,
            dependencies: [this.queue],
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
SqsDestination[_a] = { fqn: "aws-cdk-lib.aws_s3_notifications.SqsDestination", version: "2.74.0" };
exports.SqsDestination = SqsDestination;
//# sourceMappingURL=data:application/json;base64,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