"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObjectLockRetention = exports.ObjectLockMode = exports.ReplaceKey = exports.BucketAccessControl = exports.EventType = exports.BucketEncryption = exports.Bucket = exports.ObjectOwnership = exports.InventoryObjectVersion = exports.InventoryFrequency = exports.InventoryFormat = exports.RedirectProtocol = exports.HttpMethods = exports.BlockPublicAccess = exports.BucketBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const path = require("path");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const cfn_reference_1 = require("../../core/lib/private/cfn-reference");
const cxapi = require("../../cx-api");
const regionInformation = require("../../region-info");
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
const AUTO_DELETE_OBJECTS_RESOURCE_TYPE = 'Custom::S3AutoDeleteObjects';
const AUTO_DELETE_OBJECTS_TAG = 'aws-cdk:auto-delete-objects';
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_ResourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BucketBase);
            }
            throw error;
        }
        this.node.addValidation({ validate: () => this.policy?.document.validateForResourcePolicy() ?? [] });
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_OnCloudTrailBucketEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onCloudTrailEvent);
            }
            throw error;
        }
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: options.paths?.map(p => this.arnForObjects(p)) ?? [this.bucketArn],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded
     * to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_OnCloudTrailBucketEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onCloudTrailPutObject);
            }
            throw error;
        }
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the
     * specified paths (keys) in this bucket are written to.  This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailWriteObject(id, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_OnCloudTrailBucketEventOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.onCloudTrailWriteObject);
            }
            throw error;
        }
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: [
                    'CompleteMultipartUpload',
                    'CopyObject',
                    'PutObject',
                ],
                requestParameters: {
                    bucketName: [this.bucketName],
                    key: options.paths,
                },
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or its
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     *
     * Note that the policy statement may or may not be added to the policy.
     * For example, when an `IBucket` is created from an existing bucket,
     * it's not possible to tell whether the bucket already has a policy
     * attached, let alone to re-use that policy to add more statements to it.
     * So it's safest to do nothing in these cases.
     *
     * @param permission the policy statement to be added to the bucket's
     * policy.
     * @returns metadata about the execution of this method. If the policy
     * was not added, the value of `statementAdded` will be `false`. You
     * should always check this value to make sure that the operation was
     * actually carried out. Otherwise, synthesis and deploy will terminate
     * silently, which may be confusing.
     */
    addToResourcePolicy(permission) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(permission);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToResourcePolicy);
            }
            throw error;
        }
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(permission);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * The https URL of an S3 object. Specify `regional: false` at the options
     * for non-regional URLs. For example:
     *
     * - `https://s3.us-west-1.amazonaws.com/onlybucket`
     * - `https://s3.us-west-1.amazonaws.com/bucket/key`
     * - `https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey`
     *
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key) {
        const stack = core_1.Stack.of(this);
        const prefix = `https://s3.${this.env.region}.${stack.urlSuffix}/`;
        if (typeof key !== 'string') {
            return this.urlJoin(prefix, this.bucketName);
        }
        return this.urlJoin(prefix, this.bucketName, key);
    }
    /**
     * The https Transfer Acceleration URL of an S3 object. Specify `dualStack: true` at the options
     * for dual-stack endpoint (connect to the bucket over IPv6). For example:
     *
     * - `https://bucket.s3-accelerate.amazonaws.com`
     * - `https://bucket.s3-accelerate.amazonaws.com/key`
     *
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @param options Options for generating URL.
     * @returns an TransferAccelerationUrl token
     */
    transferAccelerationUrlForObject(key, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_TransferAccelerationUrlOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.transferAccelerationUrlForObject);
            }
            throw error;
        }
        const dualStack = options?.dualStack ? '.dualstack' : '';
        const prefix = `https://${this.bucketName}.s3-accelerate${dualStack}.amazonaws.com/`;
        if (typeof key !== 'string') {
            return this.urlJoin(prefix);
        }
        return this.urlJoin(prefix, key);
    }
    /**
     * The virtual hosted-style URL of an S3 object. Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * - `https://only-bucket.s3.us-west-1.amazonaws.com`
     * - `https://bucket.s3.us-west-1.amazonaws.com/key`
     * - `https://bucket.s3.amazonaws.com/key`
     * - `https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey`
     *
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     */
    virtualHostedUrlForObject(key, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_VirtualHostedStyleUrlOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.virtualHostedUrlForObject);
            }
            throw error;
        }
        const domainName = options?.regional ?? true ? this.bucketRegionalDomainName : this.bucketDomainName;
        const prefix = `https://${domainName}`;
        if (typeof key !== 'string') {
            return prefix;
        }
        return this.urlJoin(prefix, key);
    }
    /**
     * The S3 URL of an S3 object. For example:
     *
     * - `s3://onlybucket`
     * - `s3://bucket/key`
     *
     * @param key The S3 key of the object. If not specified, the S3 URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    s3UrlForObject(key) {
        const prefix = 's3://';
        if (typeof key !== 'string') {
            return this.urlJoin(prefix, this.bucketName);
        }
        return this.urlJoin(prefix, this.bucketName, key);
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     *
     * If you need to specify a keyPattern with multiple components, concatenate them into a single string, e.g.:
     *
     *   arnForObjects(`home/${team}/${user}/*`)
     *
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity, objectsKeyPattern = '*') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantRead);
            }
            throw error;
        }
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    grantWrite(identity, objectsKeyPattern = '*', allowedActionPatterns = []) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantWrite);
            }
            throw error;
        }
        const grantedWriteActions = allowedActionPatterns.length > 0 ? allowedActionPatterns : this.writeActions;
        return this.grant(identity, grantedWriteActions, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity, objectsKeyPattern = '*') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPut);
            }
            throw error;
        }
        return this.grant(identity, this.putActions, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    grantPutAcl(identity, objectsKeyPattern = '*') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPutAcl);
            }
            throw error;
        }
        return this.grant(identity, perms.BUCKET_PUT_ACL_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM principal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantDelete);
            }
            throw error;
        }
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    grantReadWrite(identity, objectsKeyPattern = '*') {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(identity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantReadWrite);
            }
            throw error;
        }
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(this.writeActions);
        // we need unique permissions because some permissions are common between read and write key actions
        const keyActions = [...new Set([...perms.KEY_READ_ACTIONS, ...perms.KEY_WRITE_ACTIONS])];
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * Note that if this `IBucket` refers to an existing bucket, possibly not
     * managed by CloudFormation, this method will have no effect, since it's
     * impossible to modify the policy of an existing bucket.
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.AnyPrincipal(),
            resource: this,
        });
    }
    /**
     * Adds a bucket notification event destination.
     * @param event The event to trigger the notification
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue)
     *
     * @param filters S3 object key filter rules to determine which objects
     * trigger this event. Each filter must include a `prefix` and/or `suffix`
     * that will be matched against the s3 object key. Refer to the S3 Developer Guide
     * for details about allowed filter rules.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering
     *
     * @example
     *
     *    declare const myLambda: lambda.Function;
     *    const bucket = new s3.Bucket(this, 'MyBucket');
     *    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(myLambda), {prefix: 'home/myusername/*'});
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     */
    addEventNotification(event, dest, ...filters) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_EventType(event);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucketNotificationDestination(dest);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_NotificationKeyFilter(filters);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventNotification);
            }
            throw error;
        }
        this.withNotifications(notifications => notifications.addNotification(event, dest, ...filters));
    }
    withNotifications(cb) {
        if (!this.notifications) {
            this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', {
                bucket: this,
                handlerRole: this.notificationsHandlerRole,
            });
        }
        cb(this.notifications);
    }
    /**
     * Subscribes a destination to receive notifications when an object is
     * created in the bucket. This is identical to calling
     * `onEvent(EventType.OBJECT_CREATED)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectCreatedNotification(dest, ...filters) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucketNotificationDestination(dest);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_NotificationKeyFilter(filters);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addObjectCreatedNotification);
            }
            throw error;
        }
        return this.addEventNotification(EventType.OBJECT_CREATED, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notifications when an object is
     * removed from the bucket. This is identical to calling
     * `onEvent(EventType.OBJECT_REMOVED)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectRemovedNotification(dest, ...filters) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucketNotificationDestination(dest);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_NotificationKeyFilter(filters);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addObjectRemovedNotification);
            }
            throw error;
        }
        return this.addEventNotification(EventType.OBJECT_REMOVED, dest, ...filters);
    }
    /**
     * Enables event bridge notification, causing all events below to be sent to EventBridge:
     *
     * - Object Deleted (DeleteObject)
     * - Object Deleted (Lifecycle expiration)
     * - Object Restore Initiated
     * - Object Restore Completed
     * - Object Restore Expired
     * - Object Storage Class Changed
     * - Object Access Tier Changed
     * - Object ACL Updated
     * - Object Tags Added
     * - Object Tags Deleted
     */
    enableEventBridgeNotification() {
        this.withNotifications(notifications => notifications.enableEventBridgeNotification());
    }
    get writeActions() {
        return [
            ...perms.BUCKET_DELETE_ACTIONS,
            ...this.putActions,
        ];
    }
    get putActions() {
        return core_1.FeatureFlags.of(this).isEnabled(cxapi.S3_GRANT_WRITE_WITHOUT_ACL)
            ? perms.BUCKET_PUT_ACTIONS
            : perms.LEGACY_BUCKET_PUT_ACTIONS;
    }
    urlJoin(...components) {
        return components.reduce((result, component) => {
            if (result.endsWith('/')) {
                result = result.slice(0, -1);
            }
            if (component.startsWith('/')) {
                component = component.slice(1);
            }
            return `${result}/${component}`;
        });
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const ret = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: bucketActions,
            resourceArns: resources,
            resource: this,
        });
        if (this.encryptionKey && keyActions && keyActions.length !== 0) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
}
_a = JSII_RTTI_SYMBOL_1;
BucketBase[_a] = { fqn: "aws-cdk-lib.aws_s3.BucketBase", version: "2.74.0" };
exports.BucketBase = BucketBase;
class BlockPublicAccess {
    constructor(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_BlockPublicAccessOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BlockPublicAccess);
            }
            throw error;
        }
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
_b = JSII_RTTI_SYMBOL_1;
BlockPublicAccess[_b] = { fqn: "aws-cdk-lib.aws_s3.BlockPublicAccess", version: "2.74.0" };
BlockPublicAccess.BLOCK_ALL = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true,
});
BlockPublicAccess.BLOCK_ACLS = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true,
});
exports.BlockPublicAccess = BlockPublicAccess;
/**
 * All http request methods
 */
var HttpMethods;
(function (HttpMethods) {
    /**
     * The GET method requests a representation of the specified resource.
     */
    HttpMethods["GET"] = "GET";
    /**
     * The PUT method replaces all current representations of the target resource with the request payload.
     */
    HttpMethods["PUT"] = "PUT";
    /**
     * The HEAD method asks for a response identical to that of a GET request, but without the response body.
     */
    HttpMethods["HEAD"] = "HEAD";
    /**
     * The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     */
    HttpMethods["POST"] = "POST";
    /**
     * The DELETE method deletes the specified resource.
     */
    HttpMethods["DELETE"] = "DELETE";
})(HttpMethods = exports.HttpMethods || (exports.HttpMethods = {}));
/**
 * All http request methods
 */
var RedirectProtocol;
(function (RedirectProtocol) {
    RedirectProtocol["HTTP"] = "http";
    RedirectProtocol["HTTPS"] = "https";
})(RedirectProtocol = exports.RedirectProtocol || (exports.RedirectProtocol = {}));
/**
 * All supported inventory list formats.
 */
var InventoryFormat;
(function (InventoryFormat) {
    /**
     * Generate the inventory list as CSV.
     */
    InventoryFormat["CSV"] = "CSV";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["PARQUET"] = "Parquet";
    /**
     * Generate the inventory list as ORC.
     */
    InventoryFormat["ORC"] = "ORC";
})(InventoryFormat = exports.InventoryFormat || (exports.InventoryFormat = {}));
/**
 * All supported inventory frequencies.
 */
var InventoryFrequency;
(function (InventoryFrequency) {
    /**
     * A report is generated every day.
     */
    InventoryFrequency["DAILY"] = "Daily";
    /**
     * A report is generated every Sunday (UTC timezone) after the initial report.
     */
    InventoryFrequency["WEEKLY"] = "Weekly";
})(InventoryFrequency = exports.InventoryFrequency || (exports.InventoryFrequency = {}));
/**
 * Inventory version support.
 */
var InventoryObjectVersion;
(function (InventoryObjectVersion) {
    /**
     * Includes all versions of each object in the report.
     */
    InventoryObjectVersion["ALL"] = "All";
    /**
     * Includes only the current version of each object in the report.
     */
    InventoryObjectVersion["CURRENT"] = "Current";
})(InventoryObjectVersion = exports.InventoryObjectVersion || (exports.InventoryObjectVersion = {}));
/**
   * The ObjectOwnership of the bucket.
   *
   * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html
   *
   */
var ObjectOwnership;
(function (ObjectOwnership) {
    /**
     * ACLs are disabled, and the bucket owner automatically owns
     * and has full control over every object in the bucket.
     * ACLs no longer affect permissions to data in the S3 bucket.
     * The bucket uses policies to define access control.
     */
    ObjectOwnership["BUCKET_OWNER_ENFORCED"] = "BucketOwnerEnforced";
    /**
     * Objects uploaded to the bucket change ownership to the bucket owner .
     */
    ObjectOwnership["BUCKET_OWNER_PREFERRED"] = "BucketOwnerPreferred";
    /**
     * The uploading account will own the object.
     */
    ObjectOwnership["OBJECT_WRITER"] = "ObjectWriter";
})(ObjectOwnership = exports.ObjectOwnership || (exports.ObjectOwnership = {}));
/**
 * An S3 bucket with associated policy objects
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 *
 * @example
 *
 * new Bucket(scope, 'Bucket', {
 *   blockPublicAccess: BlockPublicAccess.BLOCK_ALL,
 *   encryption: BucketEncryption.S3_MANAGED,
 *   enforceSSL: true,
 *   versioned: true,
 *   removalPolicy: RemovalPolicy.RETAIN,
 * });
 *
 */
class Bucket extends BucketBase {
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object. Can be obtained from a call to
     * `bucket.export()` or manually created.
     */
    static fromBucketAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_BucketAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromBucketAttributes);
            }
            throw error;
        }
        const stack = core_1.Stack.of(scope);
        const region = attrs.region ?? stack.region;
        const regionInfo = regionInformation.RegionInfo.get(region);
        const urlSuffix = regionInfo.domainSuffix ?? stack.urlSuffix;
        const bucketName = (0, util_1.parseBucketName)(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        Bucket.validateBucketName(bucketName);
        const oldEndpoint = `s3-website-${region}.${urlSuffix}`;
        const newEndpoint = `s3-website.${region}.${urlSuffix}`;
        let staticDomainEndpoint = regionInfo.s3StaticWebsiteEndpoint
            ?? core_1.Lazy.string({ produce: () => stack.regionalFact(regionInformation.FactName.S3_STATIC_WEBSITE_ENDPOINT, newEndpoint) });
        // Deprecated use of bucketWebsiteNewUrlFormat
        if (attrs.bucketWebsiteNewUrlFormat !== undefined) {
            staticDomainEndpoint = attrs.bucketWebsiteNewUrlFormat ? newEndpoint : oldEndpoint;
        }
        const websiteDomain = `${bucketName}.${staticDomainEndpoint}`;
        class Import extends BucketBase {
            constructor() {
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = (0, util_1.parseBucketArn)(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || `http://${websiteDomain}`;
                this.bucketWebsiteDomainName = attrs.bucketWebsiteUrl ? core_1.Fn.select(2, core_1.Fn.split('/', attrs.bucketWebsiteUrl)) : websiteDomain;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = attrs.bucketWebsiteNewUrlFormat ?? false;
                this.encryptionKey = attrs.encryptionKey;
                this.isWebsite = attrs.isWebsite ?? false;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
                this.notificationsHandlerRole = attrs.notificationsHandlerRole;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id, {
            account: attrs.account,
            region: attrs.region,
        });
    }
    /**
     * Create a mutable `IBucket` based on a low-level `CfnBucket`.
     */
    static fromCfnBucket(cfnBucket) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_CfnBucket(cfnBucket);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromCfnBucket);
            }
            throw error;
        }
        // use a "weird" id that has a higher chance of being unique
        const id = '@FromCfnBucket';
        // if fromCfnBucket() was already called on this cfnBucket,
        // return the same L2
        // (as different L2s would conflict, because of the mutation of the policy property of the L1 below)
        const existing = cfnBucket.node.tryFindChild(id);
        if (existing) {
            return existing;
        }
        // handle the KMS Key if the Bucket references one
        let encryptionKey;
        if (cfnBucket.bucketEncryption) {
            const serverSideEncryptionConfiguration = cfnBucket.bucketEncryption.serverSideEncryptionConfiguration;
            if (Array.isArray(serverSideEncryptionConfiguration) && serverSideEncryptionConfiguration.length === 1) {
                const serverSideEncryptionRuleProperty = serverSideEncryptionConfiguration[0];
                const serverSideEncryptionByDefault = serverSideEncryptionRuleProperty.serverSideEncryptionByDefault;
                if (serverSideEncryptionByDefault && core_1.Token.isUnresolved(serverSideEncryptionByDefault.kmsMasterKeyId)) {
                    const kmsIResolvable = core_1.Tokenization.reverse(serverSideEncryptionByDefault.kmsMasterKeyId);
                    if (kmsIResolvable instanceof cfn_reference_1.CfnReference) {
                        const cfnElement = kmsIResolvable.target;
                        if (cfnElement instanceof kms.CfnKey) {
                            encryptionKey = kms.Key.fromCfnKey(cfnElement);
                        }
                    }
                }
            }
        }
        return new class extends BucketBase {
            constructor() {
                super(cfnBucket, id);
                this.bucketArn = cfnBucket.attrArn;
                this.bucketName = cfnBucket.ref;
                this.bucketDomainName = cfnBucket.attrDomainName;
                this.bucketDualStackDomainName = cfnBucket.attrDualStackDomainName;
                this.bucketRegionalDomainName = cfnBucket.attrRegionalDomainName;
                this.bucketWebsiteUrl = cfnBucket.attrWebsiteUrl;
                this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', cfnBucket.attrWebsiteUrl));
                this.encryptionKey = encryptionKey;
                this.isWebsite = cfnBucket.websiteConfiguration !== undefined;
                this.policy = undefined;
                this.autoCreatePolicy = true;
                this.disallowPublicAccess = cfnBucket.publicAccessBlockConfiguration &&
                    cfnBucket.publicAccessBlockConfiguration.blockPublicPolicy;
                this.node.defaultChild = cfnBucket;
            }
        }();
    }
    /**
     * Thrown an exception if the given bucket name is not valid.
     *
     * @param physicalName name of the bucket.
     */
    static validateBucketName(physicalName) {
        const bucketName = physicalName;
        if (!bucketName || core_1.Token.isUnresolved(bucketName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.bucketName,
        });
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        this.cors = [];
        this.inventories = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_BucketProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Bucket);
            }
            throw error;
        }
        this.notificationsHandlerRole = props.notificationsHandlerRole;
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        Bucket.validateBucketName(this.physicalName);
        const websiteConfiguration = this.renderWebsiteConfiguration(props);
        this.isWebsite = (websiteConfiguration !== undefined);
        const objectLockConfiguration = this.parseObjectLockConfig(props);
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: this.physicalName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: core_1.Lazy.any({ produce: () => this.parseLifecycleConfiguration() }),
            websiteConfiguration,
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: core_1.Lazy.any({ produce: () => this.parseMetricConfiguration() }),
            corsConfiguration: core_1.Lazy.any({ produce: () => this.parseCorsConfiguration() }),
            accessControl: core_1.Lazy.string({ produce: () => this.accessControl }),
            loggingConfiguration: this.parseServerAccessLogs(props),
            inventoryConfigurations: core_1.Lazy.any({ produce: () => this.parseInventoryConfiguration() }),
            ownershipControls: this.parseOwnershipControls(props),
            accelerateConfiguration: props.transferAcceleration ? { accelerationStatus: 'Enabled' } : undefined,
            intelligentTieringConfigurations: this.parseTieringConfig(props),
            objectLockEnabled: objectLockConfiguration ? true : props.objectLockEnabled,
            objectLockConfiguration: objectLockConfiguration,
        });
        this._resource = resource;
        resource.applyRemovalPolicy(props.removalPolicy);
        this.encryptionKey = encryptionKey;
        this.eventBridgeEnabled = props.eventBridgeEnabled;
        this.bucketName = this.getResourceNameAttribute(resource.ref);
        this.bucketArn = this.getResourceArnAttribute(resource.attrArn, {
            region: '',
            account: '',
            service: 's3',
            resource: this.physicalName,
        });
        this.bucketDomainName = resource.attrDomainName;
        this.bucketWebsiteUrl = resource.attrWebsiteUrl;
        this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', this.bucketWebsiteUrl));
        this.bucketDualStackDomainName = resource.attrDualStackDomainName;
        this.bucketRegionalDomainName = resource.attrRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        this.accessControl = props.accessControl;
        // Enforce AWS Foundational Security Best Practice
        if (props.enforceSSL) {
            this.enforceSSLStatement();
        }
        if (props.serverAccessLogsBucket instanceof Bucket) {
            props.serverAccessLogsBucket.allowLogDelivery(this, props.serverAccessLogsPrefix);
            // It is possible that `serverAccessLogsBucket` was specified but is some other `IBucket`
            // that cannot have the ACLs or bucket policy applied. In that scenario, we should only
            // setup log delivery permissions to `this` if a bucket was not specified at all, as documented.
            // For example, we should not allow log delivery to `this` if given an imported bucket or
            // another situation that causes `instanceof` to fail
        }
        else if (!props.serverAccessLogsBucket && props.serverAccessLogsPrefix) {
            this.allowLogDelivery(this, props.serverAccessLogsPrefix);
        }
        else if (props.serverAccessLogsBucket) {
            // A `serverAccessLogsBucket` was provided but it is not a concrete `Bucket` and it
            // may not be possible to configure the ACLs or bucket policy as required.
            core_1.Annotations.of(this).addWarning(`Unable to add necessary logging permissions to imported target bucket: ${props.serverAccessLogsBucket}`);
        }
        for (const inventory of props.inventories ?? []) {
            this.addInventory(inventory);
        }
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all cors configuration rules
        (props.cors || []).forEach(this.addCorsRule.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
        if (props.autoDeleteObjects) {
            if (props.removalPolicy !== core_1.RemovalPolicy.DESTROY) {
                throw new Error('Cannot use \'autoDeleteObjects\' property on a bucket without setting removal policy to \'DESTROY\'.');
            }
            this.enableAutoDeleteObjects();
        }
        if (this.eventBridgeEnabled) {
            this.enableEventBridgeNotification();
        }
    }
    /**
     * Add a lifecycle rule to the bucket
     *
     * @param rule The rule to add
     */
    addLifecycleRule(rule) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_LifecycleRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLifecycleRule);
            }
            throw error;
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add
     */
    addMetric(metric) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_BucketMetrics(metric);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMetric);
            }
            throw error;
        }
        this.metrics.push(metric);
    }
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket
     *
     * @param rule The CORS configuration rule to add
     */
    addCorsRule(rule) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_CorsRule(rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addCorsRule);
            }
            throw error;
        }
        this.cors.push(rule);
    }
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add
     */
    addInventory(inventory) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_Inventory(inventory);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addInventory);
            }
            throw error;
        }
        this.inventories.push(inventory);
    }
    /**
     * Adds an iam statement to enforce SSL requests only.
     */
    enforceSSLStatement() {
        const statement = new iam.PolicyStatement({
            actions: ['s3:*'],
            conditions: {
                Bool: { 'aws:SecureTransport': 'false' },
            },
            effect: iam.Effect.DENY,
            resources: [
                this.bucketArn,
                this.arnForObjects('*'),
            ],
            principals: [new iam.AnyPrincipal()],
        });
        this.addToResourcePolicy(statement);
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration, according to the following table:
     *
     * | props.encryption | props.encryptionKey | props.bucketKeyEnabled | bucketEncryption (return value) | encryptionKey (return value) |
     * |------------------|---------------------|------------------------|---------------------------------|------------------------------|
     * | undefined        | undefined           | e                      | undefined                       | undefined                    |
     * | UNENCRYPTED      | undefined           | false                  | undefined                       | undefined                    |
     * | undefined        | k                   | e                      | SSE-KMS, bucketKeyEnabled = e   | k                            |
     * | KMS              | k                   | e                      | SSE-KMS, bucketKeyEnabled = e   | k                            |
     * | KMS              | undefined           | e                      | SSE-KMS, bucketKeyEnabled = e   | new key                      |
     * | KMS_MANAGED      | undefined           | e                      | SSE-KMS, bucketKeyEnabled = e   | undefined                    |
     * | S3_MANAGED       | undefined           | false                  | SSE-S3                          | undefined                    |
     * | UNENCRYPTED      | undefined           | true                   | ERROR!                          | ERROR!                       |
     * | UNENCRYPTED      | k                   | e                      | ERROR!                          | ERROR!                       |
     * | KMS_MANAGED      | k                   | e                      | ERROR!                          | ERROR!                       |
     * | S3_MANAGED       | undefined           | true                   | ERROR!                          | ERROR!                       |
     * | S3_MANAGED       | k                   | e                      | ERROR!                          | ERROR!                       |
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        let encryptionType = props.encryption;
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey ? BucketEncryption.KMS : BucketEncryption.UNENCRYPTED;
        }
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        // if bucketKeyEnabled is set, encryption must be set to KMS.
        if (props.bucketKeyEnabled && ![BucketEncryption.KMS, BucketEncryption.KMS_MANAGED].includes(encryptionType)) {
            throw new Error(`bucketKeyEnabled is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.UNENCRYPTED) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        bucketKeyEnabled: props.bucketKeyEnabled,
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn,
                        },
                    },
                ],
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } },
                ],
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KMS_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        bucketKeyEnabled: props.bucketKeyEnabled,
                        serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' },
                    },
                ],
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the bucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            const enabled = rule.enabled ?? true;
            const x = {
                // eslint-disable-next-line max-len
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfter !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfter.toDays() } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: rule.expiration?.toDays(),
                id: rule.id,
                noncurrentVersionExpiration: rule.noncurrentVersionExpiration && {
                    noncurrentDays: rule.noncurrentVersionExpiration.toDays(),
                    newerNoncurrentVersions: rule.noncurrentVersionsToRetain,
                },
                noncurrentVersionTransitions: mapOrUndefined(rule.noncurrentVersionTransitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionInDays: t.transitionAfter.toDays(),
                    newerNoncurrentVersions: t.noncurrentVersionsToRetain,
                })),
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: mapOrUndefined(rule.transitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionDate: t.transitionDate,
                    transitionInDays: t.transitionAfter && t.transitionAfter.toDays(),
                })),
                expiredObjectDeleteMarker: rule.expiredObjectDeleteMarker,
                tagFilters: self.parseTagFilters(rule.tagFilters),
                objectSizeLessThan: rule.objectSizeLessThan,
                objectSizeGreaterThan: rule.objectSizeGreaterThan,
            };
            return x;
        }
    }
    parseServerAccessLogs(props) {
        if (!props.serverAccessLogsBucket && !props.serverAccessLogsPrefix) {
            return undefined;
        }
        if (
        // KMS can't be used for logging since the logging service can't use the key - logs don't write
        // KMS_MANAGED can't be used for logging since the account can't access the logging service key - account can't read logs
        (!props.serverAccessLogsBucket && (props.encryptionKey ||
            props.encryption === BucketEncryption.KMS_MANAGED ||
            props.encryption === BucketEncryption.KMS)) ||
            // Another bucket is being used that is configured for default SSE-KMS
            props.serverAccessLogsBucket?.encryptionKey) {
            throw new Error('SSE-S3 is the only supported default bucket encryption for Server Access Logging target buckets');
        }
        return {
            destinationBucketName: props.serverAccessLogsBucket?.bucketName,
            logFilePrefix: props.serverAccessLogsPrefix,
        };
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters),
            };
        }
    }
    parseCorsConfiguration() {
        if (!this.cors || this.cors.length === 0) {
            return undefined;
        }
        return { corsRules: this.cors.map(parseCors) };
        function parseCors(rule) {
            return {
                id: rule.id,
                maxAge: rule.maxAge,
                allowedHeaders: rule.allowedHeaders,
                allowedMethods: rule.allowedMethods,
                allowedOrigins: rule.allowedOrigins,
                exposedHeaders: rule.exposedHeaders,
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag],
        }));
    }
    parseOwnershipControls({ objectOwnership }) {
        if (!objectOwnership) {
            return undefined;
        }
        return {
            rules: [{
                    objectOwnership,
                }],
        };
    }
    parseTieringConfig({ intelligentTieringConfigurations }) {
        if (!intelligentTieringConfigurations) {
            return undefined;
        }
        return intelligentTieringConfigurations.map(config => {
            const tierings = [];
            if (config.archiveAccessTierTime) {
                tierings.push({
                    accessTier: 'ARCHIVE_ACCESS',
                    days: config.archiveAccessTierTime.toDays({ integral: true }),
                });
            }
            if (config.deepArchiveAccessTierTime) {
                tierings.push({
                    accessTier: 'DEEP_ARCHIVE_ACCESS',
                    days: config.deepArchiveAccessTierTime.toDays({ integral: true }),
                });
            }
            return {
                id: config.name,
                prefix: config.prefix,
                status: 'Enabled',
                tagFilters: config.tags,
                tierings: tierings,
            };
        });
    }
    parseObjectLockConfig(props) {
        const { objectLockEnabled, objectLockDefaultRetention } = props;
        if (!objectLockDefaultRetention) {
            return undefined;
        }
        if (objectLockEnabled === false && objectLockDefaultRetention) {
            throw new Error('Object Lock must be enabled to configure default retention settings');
        }
        return {
            objectLockEnabled: 'Enabled',
            rule: {
                defaultRetention: {
                    days: objectLockDefaultRetention.duration.toDays(),
                    mode: objectLockDefaultRetention.mode,
                },
            },
        };
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument && !props.websiteRedirect && !props.websiteRoutingRules) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error('"websiteIndexDocument" is required if "websiteErrorDocument" is set');
        }
        if (props.websiteRedirect && (props.websiteErrorDocument || props.websiteIndexDocument || props.websiteRoutingRules)) {
            throw new Error('"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used');
        }
        const routingRules = props.websiteRoutingRules ? props.websiteRoutingRules.map((rule) => {
            if (rule.condition && !rule.condition.httpErrorCodeReturnedEquals && !rule.condition.keyPrefixEquals) {
                throw new Error('The condition property cannot be an empty object');
            }
            return {
                redirectRule: {
                    hostName: rule.hostName,
                    httpRedirectCode: rule.httpRedirectCode,
                    protocol: rule.protocol,
                    replaceKeyWith: rule.replaceKey && rule.replaceKey.withKey,
                    replaceKeyPrefixWith: rule.replaceKey && rule.replaceKey.prefixWithKey,
                },
                routingRuleCondition: rule.condition,
            };
        }) : undefined;
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument,
            redirectAllRequestsTo: props.websiteRedirect,
            routingRules,
        };
    }
    /**
     * Allows Log Delivery to the S3 bucket, using a Bucket Policy if the relevant feature
     * flag is enabled, otherwise the canned ACL is used.
     *
     * If log delivery is to be allowed using the ACL and an ACL has already been set, this fails.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/enable-server-access-logging.html
     */
    allowLogDelivery(from, prefix) {
        if (core_1.FeatureFlags.of(this).isEnabled(cxapi.S3_SERVER_ACCESS_LOGS_USE_BUCKET_POLICY)) {
            let conditions = undefined;
            // The conditions for the bucket policy can be applied only when the buckets are in
            // the same stack and a concrete bucket instance (not imported). Otherwise, the
            // necessary imports may result in a cyclic dependency between the stacks.
            if (from instanceof Bucket && core_1.Stack.of(this) === core_1.Stack.of(from)) {
                conditions = {
                    ArnLike: {
                        'aws:SourceArn': from.bucketArn,
                    },
                    StringEquals: {
                        'aws:SourceAccount': from.env.account,
                    },
                };
            }
            this.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [new iam.ServicePrincipal('logging.s3.amazonaws.com')],
                actions: ['s3:PutObject'],
                resources: [this.arnForObjects(prefix ? `${prefix}*` : '*')],
                conditions: conditions,
            }));
        }
        else if (this.accessControl && this.accessControl !== BucketAccessControl.LOG_DELIVERY_WRITE) {
            throw new Error("Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed");
        }
        else {
            this.accessControl = BucketAccessControl.LOG_DELIVERY_WRITE;
        }
    }
    parseInventoryConfiguration() {
        if (!this.inventories || this.inventories.length === 0) {
            return undefined;
        }
        return this.inventories.map((inventory, index) => {
            const format = inventory.format ?? InventoryFormat.CSV;
            const frequency = inventory.frequency ?? InventoryFrequency.WEEKLY;
            const id = inventory.inventoryId ?? `${this.node.id}Inventory${index}`;
            if (inventory.destination.bucket instanceof Bucket) {
                inventory.destination.bucket.addToResourcePolicy(new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['s3:PutObject'],
                    resources: [
                        inventory.destination.bucket.bucketArn,
                        inventory.destination.bucket.arnForObjects(`${inventory.destination.prefix ?? ''}*`),
                    ],
                    principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': this.bucketArn,
                        },
                    },
                }));
            }
            return {
                id,
                destination: {
                    bucketArn: inventory.destination.bucket.bucketArn,
                    bucketAccountId: inventory.destination.bucketOwner,
                    prefix: inventory.destination.prefix,
                    format,
                },
                enabled: inventory.enabled ?? true,
                includedObjectVersions: inventory.includeObjectVersions ?? InventoryObjectVersion.ALL,
                scheduleFrequency: frequency,
                optionalFields: inventory.optionalFields,
                prefix: inventory.objectsPrefix,
            };
        });
    }
    enableAutoDeleteObjects() {
        const provider = core_1.CustomResourceProvider.getOrCreateProvider(this, AUTO_DELETE_OBJECTS_RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'auto-delete-objects-handler'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_14_X,
            description: `Lambda function for auto-deleting objects in ${this.bucketName} S3 bucket.`,
        });
        // Use a bucket policy to allow the custom resource to delete
        // objects in the bucket
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: [
                // list objects
                ...perms.BUCKET_READ_METADATA_ACTIONS,
                ...perms.BUCKET_DELETE_ACTIONS, // and then delete them
            ],
            resources: [
                this.bucketArn,
                this.arnForObjects('*'),
            ],
            principals: [new iam.ArnPrincipal(provider.roleArn)],
        }));
        const customResource = new core_1.CustomResource(this, 'AutoDeleteObjectsCustomResource', {
            resourceType: AUTO_DELETE_OBJECTS_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                BucketName: this.bucketName,
            },
        });
        // Ensure bucket policy is deleted AFTER the custom resource otherwise
        // we don't have permissions to list and delete in the bucket.
        // (add a `if` to make TS happy)
        if (this.policy) {
            customResource.node.addDependency(this.policy);
        }
        // We also tag the bucket to record the fact that we want it autodeleted.
        // The custom resource will check this tag before actually doing the delete.
        // Because tagging and untagging will ALWAYS happen before the CR is deleted,
        // we can set `autoDeleteObjects: false` without the removal of the CR emptying
        // the bucket as a side effect.
        core_1.Tags.of(this._resource).add(AUTO_DELETE_OBJECTS_TAG, 'true');
    }
}
_c = JSII_RTTI_SYMBOL_1;
Bucket[_c] = { fqn: "aws-cdk-lib.aws_s3.Bucket", version: "2.74.0" };
exports.Bucket = Bucket;
/**
 * What kind of server-side encryption to apply to this bucket
 */
var BucketEncryption;
(function (BucketEncryption) {
    /**
     * Previous option. Buckets can not be unencrypted now.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/userguide/serv-side-encryption.html
     * @deprecated S3 applies server-side encryption with SSE-S3 for every bucket
     * that default encryption is not configured.
     */
    BucketEncryption["UNENCRYPTED"] = "UNENCRYPTED";
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     */
    BucketEncryption["KMS_MANAGED"] = "KMS_MANAGED";
    /**
     * Server-side encryption with a master key managed by S3.
     */
    BucketEncryption["S3_MANAGED"] = "S3_MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    BucketEncryption["KMS"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 * @link https://docs.aws.amazon.com/AmazonS3/latest/userguide/notification-how-to-event-types-and-destinations.html#supported-notification-event-types
 */
var EventType;
(function (EventType) {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED"] = "s3:ObjectCreated:*";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_PUT"] = "s3:ObjectCreated:Put";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_POST"] = "s3:ObjectCreated:Post";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COPY"] = "s3:ObjectCreated:Copy";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD"] = "s3:ObjectCreated:CompleteMultipartUpload";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED"] = "s3:ObjectRemoved:*";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE"] = "s3:ObjectRemoved:Delete";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE_MARKER_CREATED"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    /**
     * Using restore object event types you can receive notifications for
     * initiation and completion when restoring objects from the S3 Glacier
     * storage class.
     *
     * You use s3:ObjectRestore:Post to request notification of object restoration
     * initiation.
     */
    EventType["OBJECT_RESTORE_POST"] = "s3:ObjectRestore:Post";
    /**
     * Using restore object event types you can receive notifications for
     * initiation and completion when restoring objects from the S3 Glacier
     * storage class.
     *
     * You use s3:ObjectRestore:Completed to request notification of
     * restoration completion.
     */
    EventType["OBJECT_RESTORE_COMPLETED"] = "s3:ObjectRestore:Completed";
    /**
     * Using restore object event types you can receive notifications for
     * initiation and completion when restoring objects from the S3 Glacier
     * storage class.
     *
     * You use s3:ObjectRestore:Delete to request notification of
     * restoration completion.
     */
    EventType["OBJECT_RESTORE_DELETE"] = "s3:ObjectRestore:Delete";
    /**
     * You can use this event type to request Amazon S3 to send a notification
     * message when Amazon S3 detects that an object of the RRS storage class is
     * lost.
     */
    EventType["REDUCED_REDUNDANCY_LOST_OBJECT"] = "s3:ReducedRedundancyLostObject";
    /**
     * You receive this notification event when an object that was eligible for
     * replication using Amazon S3 Replication Time Control failed to replicate.
     */
    EventType["REPLICATION_OPERATION_FAILED_REPLICATION"] = "s3:Replication:OperationFailedReplication";
    /**
     * You receive this notification event when an object that was eligible for
     * replication using Amazon S3 Replication Time Control exceeded the 15-minute
     * threshold for replication.
     */
    EventType["REPLICATION_OPERATION_MISSED_THRESHOLD"] = "s3:Replication:OperationMissedThreshold";
    /**
     * You receive this notification event for an object that was eligible for
     * replication using the Amazon S3 Replication Time Control feature replicated
     * after the 15-minute threshold.
     */
    EventType["REPLICATION_OPERATION_REPLICATED_AFTER_THRESHOLD"] = "s3:Replication:OperationReplicatedAfterThreshold";
    /**
     * You receive this notification event for an object that was eligible for
     * replication using Amazon S3 Replication Time Control but is no longer tracked
     * by replication metrics.
     */
    EventType["REPLICATION_OPERATION_NOT_TRACKED"] = "s3:Replication:OperationNotTracked";
    /**
     * By using the LifecycleExpiration event types, you can receive a notification
     * when Amazon S3 deletes an object based on your S3 Lifecycle configuration.
     */
    EventType["LIFECYCLE_EXPIRATION"] = "s3:LifecycleExpiration:*";
    /**
     * The s3:LifecycleExpiration:Delete event type notifies you when an object
     * in an unversioned bucket is deleted.
     * It also notifies you when an object version is permanently deleted by an
     * S3 Lifecycle configuration.
     */
    EventType["LIFECYCLE_EXPIRATION_DELETE"] = "s3:LifecycleExpiration:Delete";
    /**
     * The s3:LifecycleExpiration:DeleteMarkerCreated event type notifies you
     * when S3 Lifecycle creates a delete marker when a current version of an
     * object in versioned bucket is deleted.
     */
    EventType["LIFECYCLE_EXPIRATION_DELETE_MARKER_CREATED"] = "s3:LifecycleExpiration:DeleteMarkerCreated";
    /**
     * You receive this notification event when an object is transitioned to
     * another Amazon S3 storage class by an S3 Lifecycle configuration.
     */
    EventType["LIFECYCLE_TRANSITION"] = "s3:LifecycleTransition";
    /**
     * You receive this notification event when an object within the
     * S3 Intelligent-Tiering storage class moved to the Archive Access tier or
     * Deep Archive Access tier.
     */
    EventType["INTELLIGENT_TIERING"] = "s3:IntelligentTiering";
    /**
     * By using the ObjectTagging event types, you can enable notification when
     * an object tag is added or deleted from an object.
     */
    EventType["OBJECT_TAGGING"] = "s3:ObjectTagging:*";
    /**
     * The s3:ObjectTagging:Put event type notifies you when a tag is PUT on an
     * object or an existing tag is updated.
  
     */
    EventType["OBJECT_TAGGING_PUT"] = "s3:ObjectTagging:Put";
    /**
     * The s3:ObjectTagging:Delete event type notifies you when a tag is removed
     * from an object.
     */
    EventType["OBJECT_TAGGING_DELETE"] = "s3:ObjectTagging:Delete";
    /**
     * You receive this notification event when an ACL is PUT on an object or when
     * an existing ACL is changed.
     * An event is not generated when a request results in no change to an
     * object’s ACL.
     */
    EventType["OBJECT_ACL_PUT"] = "s3:ObjectAcl:Put";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 */
var BucketAccessControl;
(function (BucketAccessControl) {
    /**
     * Owner gets FULL_CONTROL. No one else has access rights.
     */
    BucketAccessControl["PRIVATE"] = "Private";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ access.
     */
    BucketAccessControl["PUBLIC_READ"] = "PublicRead";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     */
    BucketAccessControl["PUBLIC_READ_WRITE"] = "PublicReadWrite";
    /**
     * Owner gets FULL_CONTROL. The AuthenticatedUsers group gets READ access.
     */
    BucketAccessControl["AUTHENTICATED_READ"] = "AuthenticatedRead";
    /**
     * The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     */
    BucketAccessControl["LOG_DELIVERY_WRITE"] = "LogDeliveryWrite";
    /**
     * Object owner gets FULL_CONTROL. Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_READ"] = "BucketOwnerRead";
    /**
     * Both the object owner and the bucket owner get FULL_CONTROL over the object.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_FULL_CONTROL"] = "BucketOwnerFullControl";
    /**
     * Owner gets FULL_CONTROL. Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     */
    BucketAccessControl["AWS_EXEC_READ"] = "AwsExecRead";
})(BucketAccessControl = exports.BucketAccessControl || (exports.BucketAccessControl = {}));
class ReplaceKey {
    /**
     * The specific object key to use in the redirect request
     */
    static with(keyReplacement) {
        return new this(keyReplacement);
    }
    /**
     * The object key prefix to use in the redirect request
     */
    static prefixWith(keyReplacement) {
        return new this(undefined, keyReplacement);
    }
    constructor(withKey, prefixWithKey) {
        this.withKey = withKey;
        this.prefixWithKey = prefixWithKey;
    }
}
_d = JSII_RTTI_SYMBOL_1;
ReplaceKey[_d] = { fqn: "aws-cdk-lib.aws_s3.ReplaceKey", version: "2.74.0" };
exports.ReplaceKey = ReplaceKey;
/**
 * Modes in which S3 Object Lock retention can be configured.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-modes
 */
var ObjectLockMode;
(function (ObjectLockMode) {
    /**
     * The Governance retention mode.
     *
     * With governance mode, you protect objects against being deleted by most users, but you can
     * still grant some users permission to alter the retention settings or delete the object if
     * necessary. You can also use governance mode to test retention-period settings before
     * creating a compliance-mode retention period.
     */
    ObjectLockMode["GOVERNANCE"] = "GOVERNANCE";
    /**
     * The Compliance retention mode.
     *
     * When an object is locked in compliance mode, its retention mode can't be changed, and
     * its retention period can't be shortened. Compliance mode helps ensure that an object
     * version can't be overwritten or deleted for the duration of the retention period.
     */
    ObjectLockMode["COMPLIANCE"] = "COMPLIANCE";
})(ObjectLockMode = exports.ObjectLockMode || (exports.ObjectLockMode = {}));
/**
 * The default retention settings for an S3 Object Lock configuration.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html
 */
class ObjectLockRetention {
    /**
     * Configure for Governance retention for a specified duration.
     *
     * With governance mode, you protect objects against being deleted by most users, but you can
     * still grant some users permission to alter the retention settings or delete the object if
     * necessary. You can also use governance mode to test retention-period settings before
     * creating a compliance-mode retention period.
     *
     * @param duration the length of time for which objects should retained
     * @returns the ObjectLockRetention configuration
     */
    static governance(duration) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Duration(duration);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.governance);
            }
            throw error;
        }
        return new ObjectLockRetention(ObjectLockMode.GOVERNANCE, duration);
    }
    /**
     * Configure for Compliance retention for a specified duration.
     *
     * When an object is locked in compliance mode, its retention mode can't be changed, and
     * its retention period can't be shortened. Compliance mode helps ensure that an object
     * version can't be overwritten or deleted for the duration of the retention period.
     *
     * @param duration the length of time for which objects should be retained
     * @returns the ObjectLockRetention configuration
     */
    static compliance(duration) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Duration(duration);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.compliance);
            }
            throw error;
        }
        return new ObjectLockRetention(ObjectLockMode.COMPLIANCE, duration);
    }
    constructor(mode, duration) {
        // https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-managing.html#object-lock-managing-retention-limits
        if (duration.toDays() > 365 * 100) {
            throw new Error('Object Lock retention duration must be less than 100 years');
        }
        if (duration.toDays() < 1) {
            throw new Error('Object Lock retention duration must be at least 1 day');
        }
        this.mode = mode;
        this.duration = duration;
    }
}
_e = JSII_RTTI_SYMBOL_1;
ObjectLockRetention[_e] = { fqn: "aws-cdk-lib.aws_s3.ObjectLockRetention", version: "2.74.0" };
exports.ObjectLockRetention = ObjectLockRetention;
function mapOrUndefined(list, callback) {
    if (!list || list.length === 0) {
        return undefined;
    }
    return list.map(callback);
}
//# sourceMappingURL=data:application/json;base64,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