"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostedRotationType = exports.HostedRotation = exports.RotationSchedule = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const secret_1 = require("./secret");
const secretsmanager_generated_1 = require("./secretsmanager.generated");
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 */
const DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * A rotation schedule.
 */
class RotationSchedule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_RotationScheduleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, RotationSchedule);
            }
            throw error;
        }
        if ((!props.rotationLambda && !props.hostedRotation) || (props.rotationLambda && props.hostedRotation)) {
            throw new Error('One of `rotationLambda` or `hostedRotation` must be specified.');
        }
        if (props.rotationLambda?.permissionsNode.defaultChild) {
            if (props.secret.encryptionKey) {
                props.secret.encryptionKey.grantEncryptDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, props.rotationLambda.grantPrincipal));
            }
            props.rotationLambda.grantInvoke(new iam.ServicePrincipal('secretsmanager.amazonaws.com'));
            props.rotationLambda.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:DescribeSecret',
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:PutSecretValue',
                    'secretsmanager:UpdateSecretVersionStage',
                ],
                resources: [props.secret.secretFullArn ? props.secret.secretFullArn : `${props.secret.secretArn}-??????`],
            }));
            props.rotationLambda.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:GetRandomPassword',
                ],
                resources: ['*'],
            }));
        }
        let automaticallyAfterDays = undefined;
        if (props.automaticallyAfter?.toMilliseconds() !== 0) {
            automaticallyAfterDays = props.automaticallyAfter?.toDays() || 30;
        }
        let rotationRules = undefined;
        if (automaticallyAfterDays !== undefined) {
            rotationRules = {
                automaticallyAfterDays,
            };
        }
        new secretsmanager_generated_1.CfnRotationSchedule(this, 'Resource', {
            secretId: props.secret.secretArn,
            rotationLambdaArn: props.rotationLambda?.functionArn,
            hostedRotationLambda: props.hostedRotation?.bind(props.secret, this),
            rotationRules,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
    }
}
_a = JSII_RTTI_SYMBOL_1;
RotationSchedule[_a] = { fqn: "aws-cdk-lib.aws_secretsmanager.RotationSchedule", version: "2.74.0" };
exports.RotationSchedule = RotationSchedule;
/**
 * A hosted rotation
 */
class HostedRotation {
    /** MySQL Single User */
    static mysqlSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mysqlSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MYSQL_SINGLE_USER, options);
    }
    /** MySQL Multi User */
    static mysqlMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mysqlMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MYSQL_MULTI_USER, options, options.masterSecret);
    }
    /** PostgreSQL Single User */
    static postgreSqlSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.postgreSqlSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.POSTGRESQL_SINGLE_USER, options);
    }
    /** PostgreSQL Multi User */
    static postgreSqlMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.postgreSqlMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.POSTGRESQL_MULTI_USER, options, options.masterSecret);
    }
    /** Oracle Single User */
    static oracleSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.oracleSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.ORACLE_SINGLE_USER, options);
    }
    /** Oracle Multi User */
    static oracleMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.oracleMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.ORACLE_MULTI_USER, options, options.masterSecret);
    }
    /** MariaDB Single User */
    static mariaDbSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mariaDbSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MARIADB_SINGLE_USER, options);
    }
    /** MariaDB Multi User */
    static mariaDbMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mariaDbMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MARIADB_MULTI_USER, options, options.masterSecret);
    }
    /** SQL Server Single User */
    static sqlServerSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sqlServerSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.SQLSERVER_SINGLE_USER, options);
    }
    /** SQL Server Multi User */
    static sqlServerMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sqlServerMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.SQLSERVER_MULTI_USER, options, options.masterSecret);
    }
    /** Redshift Single User */
    static redshiftSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.redshiftSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.REDSHIFT_SINGLE_USER, options);
    }
    /** Redshift Multi User */
    static redshiftMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.redshiftMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.REDSHIFT_MULTI_USER, options, options.masterSecret);
    }
    /** MongoDB Single User */
    static mongoDbSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mongoDbSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MONGODB_SINGLE_USER, options);
    }
    /** MongoDB Multi User */
    static mongoDbMultiUser(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mongoDbMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MONGODB_MULTI_USER, options, options.masterSecret);
    }
    constructor(type, props, masterSecret) {
        this.type = type;
        this.props = props;
        this.masterSecret = masterSecret;
        if (type.isMultiUser && !masterSecret) {
            throw new Error('The `masterSecret` must be specified when using the multi user scheme.');
        }
    }
    /**
     * Binds this hosted rotation to a secret
     */
    bind(secret, scope) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_ISecret(secret);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-secretsmanager-rotationschedule-hostedrotationlambda.html
        core_1.Stack.of(scope).addTransform('AWS::SecretsManager-2020-07-23');
        if (!this.props.vpc && this.props.securityGroups) {
            throw new Error('`vpc` must be specified when specifying `securityGroups`.');
        }
        if (this.props.vpc) {
            this._connections = new ec2.Connections({
                securityGroups: this.props.securityGroups || [new ec2.SecurityGroup(scope, 'SecurityGroup', {
                        vpc: this.props.vpc,
                    })],
            });
        }
        // Prevent master secret deletion when rotation is in place
        if (this.masterSecret) {
            this.masterSecret.denyAccountRootDelete();
        }
        const defaultExcludeCharacters = secret_1.Secret.isSecret(secret)
            ? secret.excludeCharacters ?? DEFAULT_PASSWORD_EXCLUDE_CHARS
            : DEFAULT_PASSWORD_EXCLUDE_CHARS;
        return {
            rotationType: this.type.name,
            kmsKeyArn: secret.encryptionKey?.keyArn,
            masterSecretArn: this.masterSecret?.secretArn,
            masterSecretKmsKeyArn: this.masterSecret?.encryptionKey?.keyArn,
            rotationLambdaName: this.props.functionName,
            vpcSecurityGroupIds: this._connections?.securityGroups?.map(s => s.securityGroupId).join(','),
            vpcSubnetIds: this.props.vpc?.selectSubnets(this.props.vpcSubnets).subnetIds.join(','),
            excludeCharacters: this.props.excludeCharacters ?? defaultExcludeCharacters,
        };
    }
    /**
     * Security group connections for this hosted rotation
     */
    get connections() {
        if (!this.props.vpc) {
            throw new Error('Cannot use connections for a hosted rotation that is not deployed in a VPC');
        }
        // If we are in a vpc and bind() has been called _connections should be defined
        if (!this._connections) {
            throw new Error('Cannot use connections for a hosted rotation that has not been bound to a secret');
        }
        return this._connections;
    }
}
_b = JSII_RTTI_SYMBOL_1;
HostedRotation[_b] = { fqn: "aws-cdk-lib.aws_secretsmanager.HostedRotation", version: "2.74.0" };
exports.HostedRotation = HostedRotation;
/**
 * Hosted rotation type
 */
class HostedRotationType {
    /**
     * @param name The type of rotation
     * @param isMultiUser Whether the rotation uses the mutli user scheme
     */
    constructor(name, isMultiUser) {
        this.name = name;
        this.isMultiUser = isMultiUser;
    }
}
_c = JSII_RTTI_SYMBOL_1;
HostedRotationType[_c] = { fqn: "aws-cdk-lib.aws_secretsmanager.HostedRotationType", version: "2.74.0" };
/** MySQL Single User */
HostedRotationType.MYSQL_SINGLE_USER = new HostedRotationType('MySQLSingleUser');
/** MySQL Multi User */
HostedRotationType.MYSQL_MULTI_USER = new HostedRotationType('MySQLMultiUser', true);
/** PostgreSQL Single User */
HostedRotationType.POSTGRESQL_SINGLE_USER = new HostedRotationType('PostgreSQLSingleUser');
/** PostgreSQL Multi User */
HostedRotationType.POSTGRESQL_MULTI_USER = new HostedRotationType('PostgreSQLMultiUser', true);
/** Oracle Single User */
HostedRotationType.ORACLE_SINGLE_USER = new HostedRotationType('OracleSingleUser');
/** Oracle Multi User */
HostedRotationType.ORACLE_MULTI_USER = new HostedRotationType('OracleMultiUser', true);
/** MariaDB Single User */
HostedRotationType.MARIADB_SINGLE_USER = new HostedRotationType('MariaDBSingleUser');
/** MariaDB Multi User */
HostedRotationType.MARIADB_MULTI_USER = new HostedRotationType('MariaDBMultiUser', true);
/** SQL Server Single User */
HostedRotationType.SQLSERVER_SINGLE_USER = new HostedRotationType('SQLServerSingleUser');
/** SQL Server Multi User */
HostedRotationType.SQLSERVER_MULTI_USER = new HostedRotationType('SQLServerMultiUser', true);
/** Redshift Single User */
HostedRotationType.REDSHIFT_SINGLE_USER = new HostedRotationType('RedshiftSingleUser');
/** Redshift Multi User */
HostedRotationType.REDSHIFT_MULTI_USER = new HostedRotationType('RedshiftMultiUser', true);
/** MongoDB Single User */
HostedRotationType.MONGODB_SINGLE_USER = new HostedRotationType('MongoDBSingleUser');
/** MongoDB Multi User */
HostedRotationType.MONGODB_MULTI_USER = new HostedRotationType('MongoDBMultiUser', true);
exports.HostedRotationType = HostedRotationType;
//# sourceMappingURL=data:application/json;base64,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