"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = exports.SecretStringValueBeta1 = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
const SECRET_SYMBOL = Symbol.for('@aws-cdk/secretsmanager.Secret');
/**
 * An experimental class used to specify an initial secret value for a Secret.
 *
 * The class wraps a simple string (or JSON representation) in order to provide some safety checks and warnings
 * about the dangers of using plaintext strings as initial secret seed values via CDK/CloudFormation.
 *
 * @deprecated Use `cdk.SecretValue` instead.
 */
class SecretStringValueBeta1 {
    /**
     * Creates a `SecretStringValueBeta1` from a plaintext value.
     *
     * This approach is inherently unsafe, as the secret value may be visible in your source control repository
     * and will also appear in plaintext in the resulting CloudFormation template, including in the AWS Console or APIs.
     * Usage of this method is discouraged, especially for production workloads.
     */
    static fromUnsafePlaintext(secretValue) { try {
        jsiiDeprecationWarnings.print("aws-cdk-lib.aws_secretsmanager.SecretStringValueBeta1#fromUnsafePlaintext", "Use `cdk.SecretValue` instead.");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.fromUnsafePlaintext);
        }
        throw error;
    } return new SecretStringValueBeta1(secretValue); }
    /**
     * Creates a `SecretValueValueBeta1` from a string value coming from a Token.
     *
     * The intent is to enable creating secrets from references (e.g., `Ref`, `Fn::GetAtt`) from other resources.
     * This might be the direct output of another Construct, or the output of a Custom Resource.
     * This method throws if it determines the input is an unsafe plaintext string.
     *
     * For example:
     *
     * ```ts
     * // Creates a new IAM user, access and secret keys, and stores the secret access key in a Secret.
     * const user = new iam.User(this, 'User');
     * const accessKey = new iam.AccessKey(this, 'AccessKey', { user });
     * const secret = new secretsmanager.Secret(this, 'Secret', {
     * 	secretStringValue: accessKey.secretAccessKey,
     * });
     * ```
     *
     * The secret may also be embedded in a string representation of a JSON structure:
     *
     * ```ts
     * const user = new iam.User(this, 'User');
     * const accessKey = new iam.AccessKey(this, 'AccessKey', { user });
     * const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(JSON.stringify({
     *   username: user.userName,
     *   database: 'foo',
     *   password: accessKey.secretAccessKey.unsafeUnwrap(),
     * }));
     * ```
     *
     * Note that the value being a Token does *not* guarantee safety. For example, a Lazy-evaluated string
     * (e.g., `Lazy.string({ produce: () => 'myInsecurePassword' }))`) is a Token, but as the output is
     * ultimately a plaintext string, and so insecure.
     *
     * @param secretValueFromToken a secret value coming from a Construct attribute or Custom Resource output
     */
    static fromToken(secretValueFromToken) {
        try {
            jsiiDeprecationWarnings.print("aws-cdk-lib.aws_secretsmanager.SecretStringValueBeta1#fromToken", "Use `cdk.SecretValue` instead.");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromToken);
            }
            throw error;
        }
        if (!core_1.Token.isUnresolved(secretValueFromToken)) {
            throw new Error('SecretStringValueBeta1 appears to be plaintext (unsafe) string (or resolved Token); use fromUnsafePlaintext if this is intentional');
        }
        return new SecretStringValueBeta1(secretValueFromToken);
    }
    constructor(_secretValue) {
        this._secretValue = _secretValue;
    }
    /** Returns the secret value */
    secretValue() { try {
        jsiiDeprecationWarnings.print("aws-cdk-lib.aws_secretsmanager.SecretStringValueBeta1#secretValue", "Use `cdk.SecretValue` instead.");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.secretValue);
        }
        throw error;
    } return this._secretValue; }
}
_a = JSII_RTTI_SYMBOL_1;
SecretStringValueBeta1[_a] = { fqn: "aws-cdk-lib.aws_secretsmanager.SecretStringValueBeta1", version: "2.74.0" };
exports.SecretStringValueBeta1 = SecretStringValueBeta1;
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.node.addValidation({ validate: () => this.policy?.document.validateForResourcePolicy() ?? [] });
    }
    get secretFullArn() { return this.secretArn; }
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        const statement = result.principalStatement || result.resourceStatement;
        if (versionStages != null && statement) {
            statement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        const crossAccount = core_1.Token.compareStrings(core_1.Stack.of(this).account, grantee.grantPrincipal.principalAccount || '');
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && (!this.encryptionKey && crossAccount === core_1.TokenComparison.DIFFERENT)) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && !this.encryptionKey) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies.
     * If there is a full ARN, this is just the ARN;
     * if we have a partial ARN -- due to either importing by secret name or partial ARN --
     * then we need to add a suffix to capture the full ARN's format.
     */
    get arnForPolicies() {
        return this.secretFullArn ? this.secretFullArn : `${this.secretArn}-??????`;
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    /**
     * Return whether the given object is a Secret.
     */
    static isSecret(x) {
        return x !== null && typeof (x) === 'object' && SECRET_SYMBOL in x;
    }
    /** @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn` */
    static fromSecretArn(scope, id, secretArn) {
        const attrs = arnIsComplete(secretArn) ? { secretCompleteArn: secretArn } : { secretPartialArn: secretArn };
        return Secret.fromSecretAttributes(scope, id, attrs);
    }
    /** Imports a secret by complete ARN. The complete ARN is the ARN with the Secrets Manager-supplied suffix. */
    static fromSecretCompleteArn(scope, id, secretCompleteArn) {
        return Secret.fromSecretAttributes(scope, id, { secretCompleteArn });
    }
    /** Imports a secret by partial ARN. The partial ARN is the ARN without the Secrets Manager-supplied suffix. */
    static fromSecretPartialArn(scope, id, secretPartialArn) {
        return Secret.fromSecretAttributes(scope, id, { secretPartialArn });
    }
    /**
     * Imports a secret by secret name; the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Imports a secret by secret name.
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     * Please note this method returns ISecret that only contains partial ARN and could lead to AccessDeniedException
     * when you pass the partial ARN to CLI or SDK to get the secret value. If your secret name ends with a hyphen and
     * 6 characters, you should always use fromSecretCompleteArn() to avoid potential AccessDeniedException.
     * @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/troubleshoot.html#ARN_secretnamehyphen
     */
    static fromSecretNameV2(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretName = secretName;
                this.secretArn = this.partialArn;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Creates a "partial" ARN from the secret name. The "full" ARN would include the SecretsManager-provided suffix.
            get partialArn() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: secretName,
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SecretAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSecretAttributes);
            }
            throw error;
        }
        let secretArn;
        let secretArnIsPartial;
        if (attrs.secretArn) {
            if (attrs.secretCompleteArn || attrs.secretPartialArn) {
                throw new Error('cannot use `secretArn` with `secretCompleteArn` or `secretPartialArn`');
            }
            secretArn = attrs.secretArn;
            secretArnIsPartial = false;
        }
        else {
            if ((attrs.secretCompleteArn && attrs.secretPartialArn) ||
                (!attrs.secretCompleteArn && !attrs.secretPartialArn)) {
                throw new Error('must use only one of `secretCompleteArn` or `secretPartialArn`');
            }
            if (attrs.secretCompleteArn && !arnIsComplete(attrs.secretCompleteArn)) {
                throw new Error('`secretCompleteArn` does not appear to be complete; missing 6-character suffix');
            }
            [secretArn, secretArnIsPartial] = attrs.secretCompleteArn ? [attrs.secretCompleteArn, false] : [attrs.secretPartialArn, true];
        }
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = secretArn;
                this.secretName = parseSecretName(scope, secretArn);
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return secretArnIsPartial ? undefined : secretArn; }
        }(scope, id, { environmentFromArn: secretArn });
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.replicaRegions = [];
        this.autoCreatePolicy = true;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SecretProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Secret);
            }
            throw error;
        }
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        if ((props.generateSecretString ? 1 : 0)
            + (props.secretStringBeta1 ? 1 : 0)
            + (props.secretStringValue ? 1 : 0)
            + (props.secretObjectValue ? 1 : 0)
            > 1) {
            throw new Error('Cannot specify more than one of `generateSecretString`, `secretStringValue`, `secretObjectValue`, and `secretStringBeta1`.');
        }
        const secretString = props.secretObjectValue
            ? this.resolveSecretObjectValue(props.secretObjectValue)
            : props.secretStringValue?.unsafeUnwrap() ?? props.secretStringBeta1?.secretValue();
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString ?? (secretString ? undefined : {}),
            secretString,
            name: this.physicalName,
            replicaRegions: core_1.Lazy.any({ produce: () => this.replicaRegions }, { omitEmptyArray: true }),
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core_1.RemovalPolicy.DESTROY,
        });
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.encryptionKey = props.encryptionKey;
        const parseOwnedSecretName = core_1.FeatureFlags.of(this).isEnabled(cxapi.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME);
        this.secretName = parseOwnedSecretName
            ? parseSecretNameForOwnedSecret(this, this.secretArn, props.secretName)
            : parseSecretName(this, this.secretArn);
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        this.encryptionKey?.grantEncryptDecrypt(principal);
        this.encryptionKey?.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
        for (const replica of props.replicaRegions ?? []) {
            this.addReplicaRegion(replica.region, replica.encryptionKey);
        }
        this.excludeCharacters = props.generateSecretString?.excludeCharacters;
    }
    resolveSecretObjectValue(secretObject) {
        const resolvedObject = {};
        for (const [key, value] of Object.entries(secretObject)) {
            resolvedObject[key] = value.unsafeUnwrap();
        }
        return JSON.stringify(resolvedObject);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a replica region for the secret
     *
     * @param region The name of the region
     * @param encryptionKey The customer-managed encryption key to use for encrypting the secret value.
     */
    addReplicaRegion(region, encryptionKey) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_kms_IKey(encryptionKey);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addReplicaRegion);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && !core_1.Token.isUnresolved(region) && region === stack.region) {
            throw new Error('Cannot add the region where this stack is deployed as a replica region.');
        }
        this.replicaRegions.push({
            region,
            kmsKeyId: encryptionKey?.keyArn,
        });
    }
}
_b = JSII_RTTI_SYMBOL_1;
Secret[_b] = { fqn: "aws-cdk-lib.aws_secretsmanager.Secret", version: "2.74.0" };
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "deprecated_AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "deprecated_AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBProxy
     */
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_secretsmanager_SecretTargetAttachmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SecretTargetAttachment);
            }
            throw error;
        }
        this.attachedSecret = props.secret;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: this.attachedSecret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: attachmentTargetTypeToString(props.target.asSecretAttachmentTarget().targetType),
        });
        this.encryptionKey = this.attachedSecret.encryptionKey;
        this.secretName = this.attachedSecret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    /**
     * Forward any additions to the resource policy to the original secret.
     * This is required because a secret can only have a single resource policy.
     * If we do not forward policy additions, a new policy resource is created using the secret attachment ARN.
     * This ends up being rejected by CloudFormation.
     */
    addToResourcePolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToResourcePolicy);
            }
            throw error;
        }
        if (core_1.FeatureFlags.of(this).isEnabled(cxapi.SECRETS_MANAGER_TARGET_ATTACHMENT_RESOURCE_POLICY)) {
            return this.attachedSecret.addToResourcePolicy(statement);
        }
        return super.addToResourcePolicy(statement);
    }
}
_c = JSII_RTTI_SYMBOL_1;
SecretTargetAttachment[_c] = { fqn: "aws-cdk-lib.aws_secretsmanager.SecretTargetAttachment", version: "2.74.0" };
exports.SecretTargetAttachment = SecretTargetAttachment;
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${6-character SecretsManager suffix}`
        // If there is no hyphen (or 6-character suffix) assume no suffix was provided, and return the whole name.
        const lastHyphenIndex = resourceName.lastIndexOf('-');
        const hasSecretsSuffix = lastHyphenIndex !== -1 && resourceName.slice(lastHyphenIndex + 1).length === 6;
        return hasSecretsSuffix ? resourceName.slice(0, lastHyphenIndex) : resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
/**
 * Parses the secret name from the ARN of an owned secret. With owned secrets we know a few things we don't with imported secrets:
 * - The ARN is guaranteed to be a full ARN, with suffix.
 * - The name -- if provided -- will tell us how many hyphens to expect in the final secret name.
 * - If the name is not provided, we know the format used by CloudFormation for auto-generated names.
 *
 * Note: This is done rather than just returning the secret name passed in by the user to keep the relationship
 * explicit between the Secret and wherever the secretName might be used (i.e., using Tokens).
 */
function parseSecretNameForOwnedSecret(construct, secretArn, secretName) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (!resourceName) {
        throw new Error('invalid ARN format; no secret name provided');
    }
    // Secret name was explicitly provided, but is unresolved; best option is to use it directly.
    // If it came from another Secret, it should (hopefully) already be properly formatted.
    if (secretName && core_1.Token.isUnresolved(secretName)) {
        return secretName;
    }
    // If no secretName was provided, the name will be automatically generated by CloudFormation.
    // The autogenerated names have the form of `${logicalID}-${random}`.
    // Otherwise, we can use the existing secretName to determine how to parse the resulting resourceName.
    const secretNameHyphenatedSegments = secretName ? secretName.split('-').length : 2;
    // 2 => [0, 1]
    const segmentIndexes = [...new Array(secretNameHyphenatedSegments)].map((_, i) => i);
    // Create the secret name from the resource name by joining all the known segments together.
    // This should have the effect of stripping the final hyphen and SecretManager suffix.
    return core_1.Fn.join('-', segmentIndexes.map(i => core_1.Fn.select(i, core_1.Fn.split('-', resourceName))));
}
/** Performs a best guess if an ARN is complete, based on if it ends with a 6-character suffix. */
function arnIsComplete(secretArn) {
    return core_1.Token.isUnresolved(secretArn) || /-[a-z0-9]{6}$/i.test(secretArn);
}
/**
 * Mark all instances of 'Secret'.
 */
Object.defineProperty(Secret.prototype, SECRET_SYMBOL, {
    value: true,
    enumerable: false,
    writable: false,
});
function attachmentTargetTypeToString(x) {
    switch (x) {
        case AttachmentTargetType.RDS_DB_INSTANCE:
        case AttachmentTargetType.INSTANCE:
            return 'AWS::RDS::DBInstance';
        case AttachmentTargetType.RDS_DB_CLUSTER:
        case AttachmentTargetType.CLUSTER:
            return 'AWS::RDS::DBCluster';
        case AttachmentTargetType.RDS_DB_PROXY:
            return 'AWS::RDS::DBProxy';
        case AttachmentTargetType.REDSHIFT_CLUSTER:
            return 'AWS::Redshift::Cluster';
        case AttachmentTargetType.DOCDB_DB_INSTANCE:
            return 'AWS::DocDB::DBInstance';
        case AttachmentTargetType.DOCDB_DB_CLUSTER:
            return 'AWS::DocDB::DBCluster';
    }
}
//# sourceMappingURL=data:application/json;base64,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