"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationTemplate = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("../../aws-s3-assets");
const util_1 = require("./private/util");
/**
 * Represents the Product Provisioning Artifact Template.
 */
class CloudFormationTemplate {
    /**
     * Template from URL
     * @param url The url that points to the provisioning artifacts template
     */
    static fromUrl(url) {
        return new CloudFormationUrlTemplate(url);
    }
    /**
     * Loads the provisioning artifacts template from a local disk path.
     *
     * @param path A file containing the provisioning artifacts
     */
    static fromAsset(path, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_assets_AssetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAsset);
            }
            throw error;
        }
        return new CloudFormationAssetTemplate(path, options);
    }
    /**
     * Creates a product with the resources defined in the given product stack.
     */
    static fromProductStack(productStack) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_servicecatalog_ProductStack(productStack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromProductStack);
            }
            throw error;
        }
        return new CloudFormationProductStackTemplate(productStack);
    }
}
_a = JSII_RTTI_SYMBOL_1;
CloudFormationTemplate[_a] = { fqn: "aws-cdk-lib.aws_servicecatalog.CloudFormationTemplate", version: "2.74.0" };
exports.CloudFormationTemplate = CloudFormationTemplate;
/**
 * Template code from a Url.
 */
class CloudFormationUrlTemplate extends CloudFormationTemplate {
    constructor(url) {
        super();
        this.url = url;
    }
    bind(_scope) {
        return {
            httpUrl: this.url,
        };
    }
}
/**
 * Template from a local file.
 */
class CloudFormationAssetTemplate extends CloudFormationTemplate {
    /**
     * @param path The path to the asset file.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, `Template${(0, util_1.hashValues)(this.path)}`, {
                path: this.path,
                ...this.options,
            });
        }
        return {
            httpUrl: this.asset.httpUrl,
        };
    }
}
/**
 * Template from a CDK defined product stack.
 */
class CloudFormationProductStackTemplate extends CloudFormationTemplate {
    /**
     * @param productStack A service catalog product stack.
     */
    constructor(productStack) {
        super();
        this.productStack = productStack;
    }
    bind(_scope) {
        return {
            httpUrl: this.productStack._getTemplateUrl(),
            assetBucket: this.productStack._getAssetBucket(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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