"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const cdk = require("../../core");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
        this.assetBuckets = new Set();
        this.sharedAccounts = [];
    }
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    addProduct(product) {
        if (product.assetBuckets) {
            for (const bucket of product.assetBuckets) {
                this.assetBuckets.add(bucket);
            }
        }
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        this.sharedAccounts.push(accountId);
        new servicecatalog_generated_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.portfolioId, tagOptions);
    }
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    setLocalLaunchRoleName(product, launchRoleName, options = {}) {
        const launchRole = new iam.Role(this, `LaunchRole${launchRoleName}`, {
            roleName: launchRoleName,
            assumedBy: new iam.ServicePrincipal('servicecatalog.amazonaws.com'),
        });
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
        return launchRole;
    }
    setLocalLaunchRole(product, launchRole, options = {}) {
        validation_1.InputValidator.validateRoleNameSetForLocalLaunchRole(launchRole);
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
    }
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new servicecatalog_generated_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
    /**
     * Gives access to Asset Buckets to Shared Accounts.
     *
     */
    addBucketPermissionsToSharedAccounts() {
        if (this.sharedAccounts.length > 0) {
            for (const bucket of this.assetBuckets) {
                bucket.grantRead(new iam.CompositePrincipal(...this.sharedAccounts.map(account => new iam.AccountPrincipal(account))));
            }
        }
    }
}
/**
 * A Service Catalog portfolio.
 */
class Portfolio extends PortfolioBase {
    /**
     * Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return (0, util_1.hashValues)(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_servicecatalog_PortfolioProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Portfolio);
            }
            throw error;
        }
        this.validatePortfolioProps(props);
        this.portfolio = new servicecatalog_generated_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'catalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
        const portfolioNodeId = this.node.id;
        cdk.Aspects.of(this).add({
            visit(c) {
                if (c.node.id === portfolioNodeId) {
                    c.addBucketPermissionsToSharedAccounts();
                }
                ;
            },
        });
    }
    generateUniqueHash(value) {
        return (0, util_1.hashValues)(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "aws-cdk-lib.aws_servicecatalog.Portfolio", version: "2.74.0" };
exports.Portfolio = Portfolio;
//# sourceMappingURL=data:application/json;base64,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