"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const util_1 = require("./util");
const validation_1 = require("./validation");
const servicecatalog_generated_1 = require("../servicecatalog.generated");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options?.description);
        const associationKey = (0, util_1.hashValues)(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new servicecatalog_generated_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependency(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${(0, util_1.hashValues)(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependency(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${(0, util_1.hashValues)(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependency(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            roleArn: launchRole.roleArn,
        });
    }
    static setLocalLaunchRoleName(portfolio, product, launchRoleName, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            localRoleName: launchRoleName,
        });
    }
    static deployWithStackSets(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description ?? '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependency(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(resource, resourceId, tagOptions) {
        for (const cfnTagOption of tagOptions._cfnTagOptions) {
            const tagAssocationConstructId = `TagOptionAssociation${(0, util_1.hashValues)(cfnTagOption.key, cfnTagOption.value, resource.node.addr)}`;
            if (!resource.node.tryFindChild(tagAssocationConstructId)) {
                new servicecatalog_generated_1.CfnTagOptionAssociation(resource, tagAssocationConstructId, {
                    resourceId: resourceId,
                    tagOptionId: cfnTagOption.ref,
                });
            }
        }
    }
    static setLaunchRoleConstraint(portfolio, product, options, roleOptions) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: roleOptions.roleArn,
                localRoleName: roleOptions.localRoleName,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependency(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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