"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductStackSynthesizer = void 0;
const path = require("path");
const aws_s3_deployment_1 = require("../../../aws-s3-deployment");
const cdk = require("../../../core");
/**
 * Deployment environment for an AWS Service Catalog product stack.
 *
 * Interoperates with the StackSynthesizer of the parent stack.
 */
class ProductStackSynthesizer extends cdk.StackSynthesizer {
    constructor(assetBucket) {
        super();
        this.assetBucket = assetBucket;
    }
    addFileAsset(asset) {
        if (!this.assetBucket) {
            throw new Error('An Asset Bucket must be provided to use Assets');
        }
        const outdir = cdk.App.of(this.boundStack)?.outdir ?? 'cdk.out';
        const assetPath = `${outdir}/${asset.fileName}`;
        if (!this.bucketDeployment) {
            const parentStack = this.boundStack._getParentStack();
            if (!cdk.Resource.isOwnedResource(this.assetBucket)) {
                cdk.Annotations.of(parentStack).addWarning('[WARNING] Bucket Policy Permissions cannot be added to' +
                    ' referenced Bucket. Please make sure your bucket has the correct permissions');
            }
            this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(parentStack, 'AssetsBucketDeployment', {
                sources: [aws_s3_deployment_1.Source.asset(assetPath)],
                destinationBucket: this.assetBucket,
                extract: false,
                prune: false,
            });
        }
        else {
            this.bucketDeployment.addSource(aws_s3_deployment_1.Source.asset(assetPath));
        }
        const physicalName = this.physicalNameOfBucket(this.assetBucket);
        const bucketName = physicalName;
        if (!asset.fileName) {
            throw new Error('Asset file name is undefined');
        }
        const assetFileBaseName = path.basename(asset.fileName);
        const s3Filename = assetFileBaseName.split('.')[1] + '.zip';
        const objectKey = `${s3Filename}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        const httpUrl = `https://s3.${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    physicalNameOfBucket(bucket) {
        let resolvedName;
        if (cdk.Resource.isOwnedResource(bucket)) {
            resolvedName = cdk.Stack.of(bucket).resolve(bucket.node.defaultChild.bucketName);
        }
        else {
            resolvedName = bucket.bucketName;
        }
        if (resolvedName === undefined) {
            throw new Error('A bucketName must be provided to use Assets');
        }
        return resolvedName;
    }
    addDockerImageAsset(_asset) {
        throw new Error('Service Catalog Product Stacks cannot use Assets');
    }
    synthesize(session) {
        // Synthesize the template, but don't emit as a cloud assembly artifact.
        // It will be registered as an S3 asset of its parent instead.
        this.synthesizeTemplate(session);
    }
}
exports.ProductStackSynthesizer = ProductStackSynthesizer;
//# sourceMappingURL=data:application/json;base64,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