"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductStack = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const cdk = require("../../core");
const product_stack_synthesizer_1 = require("./private/product-stack-synthesizer");
/**
 * A Service Catalog product stack, which is similar in form to a Cloudformation nested stack.
 * You can add the resources to this stack that you want to define for your service catalog product.
 *
 * This stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 */
class ProductStack extends cdk.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            synthesizer: new product_stack_synthesizer_1.ProductStackSynthesizer(props.assetBucket),
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_servicecatalog_ProductStackProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ProductStack);
            }
            throw error;
        }
        this._parentStack = findParentStack(scope);
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${cdk.Names.uniqueId(this)}.product.template.json`;
        this.assetBucket = props.assetBucket;
    }
    /**
     * Set the parent product stack history
     *
     * @internal
     */
    _setParentProductStackHistory(parentProductStackHistory) {
        return this._parentProductStackHistory = parentProductStackHistory;
    }
    /**
     * Fetch the template URL.
     *
     * @internal
     */
    _getTemplateUrl() {
        return cdk.Lazy.uncachedString({ produce: () => this._templateUrl });
    }
    /**
     * Fetch the asset bucket.
     *
     * @internal
     */
    _getAssetBucket() {
        return this.assetBucket;
    }
    /**
     * Fetch the parent Stack.
     *
     * @internal
     */
    _getParentStack() {
        return this._parentStack;
    }
    /**
     * Synthesize the product stack template, overrides the `super` class method.
     *
     * Defines an asset at the parent stack which represents the template of this
     * product stack.
     *
     * @internal
     */
    _synthesizeTemplate(session) {
        const cfn = JSON.stringify(this._toCloudFormation(), undefined, 2);
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        this._templateUrl = this._parentStack.synthesizer.addFileAsset({
            packaging: cdk.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        }).httpUrl;
        if (this._parentProductStackHistory) {
            this._parentProductStackHistory._writeTemplateToSnapshot(cfn);
        }
        fs.writeFileSync(path.join(session.assembly.outdir, this.templateFile), cfn);
    }
}
_a = JSII_RTTI_SYMBOL_1;
ProductStack[_a] = { fqn: "aws-cdk-lib.aws_servicecatalog.ProductStack", version: "2.74.0" };
exports.ProductStack = ProductStack;
/**
 * Validates the scope for a product stack, which must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    try {
        const parentStack = cdk.Stack.of(scope);
        return parentStack;
    }
    catch {
        throw new Error('Product stacks must be defined within scope of another non-product stack');
    }
}
//# sourceMappingURL=data:application/json;base64,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