"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationProduct = exports.Product = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const association_manager_1 = require("./private/association-manager");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class ProductBase extends core_1.Resource {
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.productId, tagOptions);
    }
}
/**
 * Abstract class for Service Catalog Product.
 */
class Product extends ProductBase {
    /**
     * Creates a Product construct that represents an external product.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param productArn Product Arn
     */
    static fromProductArn(scope, id, productArn) {
        const arn = core_1.Stack.of(scope).splitArn(productArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const productId = arn.resourceName;
        if (!productId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + productArn);
        }
        return new class extends ProductBase {
            constructor() {
                super(...arguments);
                this.productId = productId;
                this.productArn = productArn;
                this.assetBuckets = [];
            }
        }(scope, id);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Product[_a] = { fqn: "aws-cdk-lib.aws_servicecatalog.Product", version: "2.74.0" };
exports.Product = Product;
/**
 * A Service Catalog Cloudformation Product.
 */
class CloudFormationProduct extends Product {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The asset bucket of a product created via product stack.
         * @default - Empty - no assets are used in this product
         */
        this.assetBuckets = new Array();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_servicecatalog_CloudFormationProductProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationProduct);
            }
            throw error;
        }
        this.validateProductProps(props);
        const product = new servicecatalog_generated_1.CfnCloudFormationProduct(this, 'Resource', {
            acceptLanguage: props.messageLanguage,
            description: props.description,
            distributor: props.distributor,
            name: props.productName,
            owner: props.owner,
            provisioningArtifactParameters: this.renderProvisioningArtifacts(props),
            replaceProvisioningArtifacts: props.replaceProductVersionIds,
            supportDescription: props.supportDescription,
            supportEmail: props.supportEmail,
            supportUrl: props.supportUrl,
        });
        this.productId = product.ref;
        this.productArn = core_1.Stack.of(this).formatArn({
            service: 'catalog',
            resource: 'product',
            resourceName: product.ref,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    renderProvisioningArtifacts(props) {
        return props.productVersions.map(productVersion => {
            const template = productVersion.cloudFormationTemplate.bind(this);
            if (template.assetBucket) {
                this.assetBuckets.push(template.assetBucket);
            }
            validation_1.InputValidator.validateUrl(this.node.path, 'provisioning template url', template.httpUrl);
            return {
                name: productVersion.productVersionName,
                description: productVersion.description,
                disableTemplateValidation: productVersion.validateTemplate === false ? true : false,
                info: {
                    LoadTemplateFromURL: template.httpUrl,
                },
            };
        });
    }
    ;
    validateProductProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'product product name', 1, 100, props.productName);
        validation_1.InputValidator.validateLength(this.node.path, 'product owner', 1, 8191, props.owner);
        validation_1.InputValidator.validateLength(this.node.path, 'product description', 0, 8191, props.description);
        validation_1.InputValidator.validateLength(this.node.path, 'product distributor', 0, 8191, props.distributor);
        validation_1.InputValidator.validateEmail(this.node.path, 'support email', props.supportEmail);
        validation_1.InputValidator.validateUrl(this.node.path, 'support url', props.supportUrl);
        validation_1.InputValidator.validateLength(this.node.path, 'support description', 0, 8191, props.supportDescription);
        if (props.productVersions.length == 0) {
            throw new Error(`Invalid product versions for resource ${this.node.path}, must contain at least 1 product version`);
        }
        props.productVersions.forEach(productVersion => {
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact name', 0, 100, productVersion.productVersionName);
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact description', 0, 8191, productVersion.description);
        });
    }
}
_b = JSII_RTTI_SYMBOL_1;
CloudFormationProduct[_b] = { fqn: "aws-cdk-lib.aws_servicecatalog.CloudFormationProduct", version: "2.74.0" };
exports.CloudFormationProduct = CloudFormationProduct;
//# sourceMappingURL=data:application/json;base64,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