"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationSetEventDestination = exports.CloudWatchDimensionSource = exports.EmailSendingEvent = exports.EventDestination = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const ses_generated_1 = require("./ses.generated");
/**
 * An event destination
 */
class EventDestination {
    /**
     * Use a SNS topic as event destination
     */
    static snsTopic(topic) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sns_ITopic(topic);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.snsTopic);
            }
            throw error;
        }
        return { topic };
    }
    /**
     * Use CloudWatch dimensions as event destination
     */
    static cloudWatchDimensions(dimensions) {
        return { dimensions };
    }
}
_a = JSII_RTTI_SYMBOL_1;
EventDestination[_a] = { fqn: "aws-cdk-lib.aws_ses.EventDestination", version: "2.74.0" };
exports.EventDestination = EventDestination;
/**
 * Email sending event
 */
var EmailSendingEvent;
(function (EmailSendingEvent) {
    /**
     * The send request was successful and SES will attempt to deliver the message
     * to the recipient's mail server. (If account-level or global suppression is
     * being used, SES will still count it as a send, but delivery is suppressed.)
     */
    EmailSendingEvent["SEND"] = "send";
    /**
     * SES accepted the email, but determined that it contained a virus and didn’t
     * attempt to deliver it to the recipient’s mail server.
     */
    EmailSendingEvent["REJECT"] = "reject";
    /**
     * (Hard bounce) The recipient's mail server permanently rejected the email.
     * (Soft bounces are only included when SES fails to deliver the email after
     * retrying for a period of time.)
     */
    EmailSendingEvent["BOUNCE"] = "bounce";
    /**
     * The email was successfully delivered to the recipient’s mail server, but the
     * recipient marked it as spam.
     */
    EmailSendingEvent["COMPLAINT"] = "complaint";
    /**
     * SES successfully delivered the email to the recipient's mail server.
     */
    EmailSendingEvent["DELIVERY"] = "delivery";
    /**
     * The recipient received the message and opened it in their email client.
     */
    EmailSendingEvent["OPEN"] = "open";
    /**
     * The recipient clicked one or more links in the email.
     */
    EmailSendingEvent["CLICK"] = "click";
    /**
     * The email wasn't sent because of a template rendering issue. This event type
     * can occur when template data is missing, or when there is a mismatch between
     * template parameters and data. (This event type only occurs when you send email
     * using the `SendTemplatedEmail` or `SendBulkTemplatedEmail` API operations.)
     */
    EmailSendingEvent["RENDERING_FAILURE"] = "renderingFailure";
    /**
     * The email couldn't be delivered to the recipient’s mail server because a temporary
     * issue occurred. Delivery delays can occur, for example, when the recipient's inbox
     * is full, or when the receiving email server experiences a transient issue.
     */
    EmailSendingEvent["DELIVERY_DELAY"] = "deliveryDelay";
    /**
     * The email was successfully delivered, but the recipient updated their subscription
     * preferences by clicking on an unsubscribe link as part of your subscription management.
     */
    EmailSendingEvent["SUBSCRIPTION"] = "subscription";
})(EmailSendingEvent = exports.EmailSendingEvent || (exports.EmailSendingEvent = {}));
/**
 * Source for CloudWatch dimension
 */
var CloudWatchDimensionSource;
(function (CloudWatchDimensionSource) {
    /**
     * Amazon SES retrieves the dimension name and value from a header in the email.
     *
     * Note: You can't use any of the following email headers as the Dimension Name:
     * `Received`, `To`, `From`, `DKIM-Signature`, `CC`, `message-id`, or `Return-Path`.
     */
    CloudWatchDimensionSource["EMAIL_HEADER"] = "emailHeader";
    /**
     * Amazon SES retrieves the dimension name and value from a tag that you specified in a link.
     *
     * @see https://docs.aws.amazon.com/ses/latest/dg/faqs-metrics.html#sending-metric-faqs-clicks-q5
     */
    CloudWatchDimensionSource["LINK_TAG"] = "linkTag";
    /**
     * Amazon SES retrieves the dimension name and value from a tag that you specify by using the
     * `X-SES-MESSAGE-TAGS` header or the Tags API parameter.
     *
     * You can also use the Message Tag value source to create dimensions based on Amazon SES auto-tags.
     * To use an auto-tag, type the complete name of the auto-tag as the Dimension Name. For example,
     * to create a dimension based on the configuration set auto-tag, use `ses:configuration-set` for the
     * Dimension Name, and the name of the configuration set for the Default Value.
     *
     * @see https://docs.aws.amazon.com/ses/latest/dg/event-publishing-send-email.html
     * @see https://docs.aws.amazon.com/ses/latest/dg/monitor-using-event-publishing.html#event-publishing-how-works
     */
    CloudWatchDimensionSource["MESSAGE_TAG"] = "messageTag";
})(CloudWatchDimensionSource = exports.CloudWatchDimensionSource || (exports.CloudWatchDimensionSource = {}));
/**
 * A configuration set event destination
 */
class ConfigurationSetEventDestination extends core_1.Resource {
    /**
     * Use an existing configuration set
     */
    static fromConfigurationSetEventDestinationId(scope, id, configurationSetEventDestinationId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.configurationSetEventDestinationId = configurationSetEventDestinationId;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configurationSetEventDestinationName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ses_ConfigurationSetEventDestinationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ConfigurationSetEventDestination);
            }
            throw error;
        }
        const configurationSet = new ses_generated_1.CfnConfigurationSetEventDestination(this, 'Resource', {
            configurationSetName: props.configurationSet.configurationSetName,
            eventDestination: {
                name: this.physicalName,
                enabled: props.enabled ?? true,
                matchingEventTypes: props.events ?? Object.values(EmailSendingEvent),
                snsDestination: props.destination.topic ? { topicArn: props.destination.topic.topicArn } : undefined,
                cloudWatchDestination: props.destination.dimensions
                    ? {
                        dimensionConfigurations: props.destination.dimensions.map(dimension => ({
                            dimensionValueSource: dimension.source,
                            dimensionName: dimension.name,
                            defaultDimensionValue: dimension.defaultValue,
                        })),
                    }
                    : undefined,
            },
        });
        this.configurationSetEventDestinationId = configurationSet.attrId;
        if (props.destination.topic) {
            const result = props.destination.topic.addToResourcePolicy(new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [props.destination.topic.topicArn],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                conditions: {
                    StringEquals: {
                        'AWS:SourceAccount': this.env.account,
                        'AWS:SourceArn': `arn:aws:ses:${this.env.region}:${this.env.account}:configuration-set/${props.configurationSet.configurationSetName}`,
                    },
                },
            }));
            if (result.policyDependable) {
                this.node.addDependency(result.policyDependable);
            }
        }
    }
}
_b = JSII_RTTI_SYMBOL_1;
ConfigurationSetEventDestination[_b] = { fqn: "aws-cdk-lib.aws_ses.ConfigurationSetEventDestination", version: "2.74.0" };
exports.ConfigurationSetEventDestination = ConfigurationSetEventDestination;
//# sourceMappingURL=data:application/json;base64,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