"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmailIdentity = exports.EasyDkimSigningKeyLength = exports.DkimIdentity = exports.MailFromBehaviorOnMxFailure = exports.Identity = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const route53 = require("../../aws-route53");
const core_1 = require("../../core");
const utils_1 = require("./private/utils");
const ses_generated_1 = require("./ses.generated");
/**
 * Identity
 */
class Identity {
    /**
     * Verify an email address
     *
     * To complete the verification process look for an email from
     * no-reply-aws@amazon.com, open it and click the link.
     */
    static email(email) {
        return { value: email };
    }
    /**
     * Verify a domain name
     *
     * DKIM records will have to be added manually to complete the verification
     * process
     */
    static domain(domain) {
        return { value: domain };
    }
    /**
     * Verify a public hosted zone
     *
     * DKIM and MAIL FROM records will be added automatically to the hosted
     * zone
     */
    static publicHostedZone(hostedZone) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_route53_IPublicHostedZone(hostedZone);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.publicHostedZone);
            }
            throw error;
        }
        return {
            value: hostedZone.zoneName,
            hostedZone,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
Identity[_a] = { fqn: "aws-cdk-lib.aws_ses.Identity", version: "2.74.0" };
exports.Identity = Identity;
/**
 * The action to take if the required MX record for the MAIL FROM domain isn't
 * found
 */
var MailFromBehaviorOnMxFailure;
(function (MailFromBehaviorOnMxFailure) {
    /**
     * The mail is sent using amazonses.com as the MAIL FROM domain
     */
    MailFromBehaviorOnMxFailure["USE_DEFAULT_VALUE"] = "USE_DEFAULT_VALUE";
    /**
     * The Amazon SES API v2 returns a `MailFromDomainNotVerified` error and doesn't
     * attempt to deliver the email
     */
    MailFromBehaviorOnMxFailure["REJECT_MESSAGE"] = "REJECT_MESSAGE";
})(MailFromBehaviorOnMxFailure = exports.MailFromBehaviorOnMxFailure || (exports.MailFromBehaviorOnMxFailure = {}));
/**
 * The identity to use for DKIM
 */
class DkimIdentity {
    /**
     * Easy DKIM
     *
     * @param signingKeyLength The length of the signing key. This can be changed at
     *   most once per day.
     *
     * @see https://docs.aws.amazon.com/ses/latest/dg/send-email-authentication-dkim-easy.html
     */
    static easyDkim(signingKeyLength) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ses_EasyDkimSigningKeyLength(signingKeyLength);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.easyDkim);
            }
            throw error;
        }
        return new EasyDkim(signingKeyLength);
    }
    /**
     * Bring Your Own DKIM
     *
     * @param options Options for BYO DKIM
     *
     * @see https://docs.aws.amazon.com/ses/latest/dg/send-email-authentication-dkim-bring-your-own.html
     */
    static byoDkim(options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ses_ByoDkimOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.byoDkim);
            }
            throw error;
        }
        return new ByoDkim(options);
    }
}
_b = JSII_RTTI_SYMBOL_1;
DkimIdentity[_b] = { fqn: "aws-cdk-lib.aws_ses.DkimIdentity", version: "2.74.0" };
exports.DkimIdentity = DkimIdentity;
class EasyDkim extends DkimIdentity {
    constructor(signingKeyLength) {
        super();
        this.signingKeyLength = signingKeyLength;
    }
    bind(emailIdentity, hostedZone) {
        if (hostedZone) {
            // Use CfnRecordSet instead of CnameRecord to avoid current bad handling of
            // tokens in route53.determineFullyQualifiedDomainName() at https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/aws-route53/lib/util.ts
            new route53.CfnRecordSet(emailIdentity, 'DkimDnsToken1', {
                hostedZoneId: hostedZone.hostedZoneId,
                name: core_1.Lazy.string({ produce: () => emailIdentity.dkimDnsTokenName1 }),
                type: 'CNAME',
                resourceRecords: [core_1.Lazy.string({ produce: () => emailIdentity.dkimDnsTokenValue1 })],
                ttl: '1800',
            });
            new route53.CfnRecordSet(emailIdentity, 'DkimDnsToken2', {
                hostedZoneId: hostedZone.hostedZoneId,
                name: core_1.Lazy.string({ produce: () => emailIdentity.dkimDnsTokenName2 }),
                type: 'CNAME',
                resourceRecords: [core_1.Lazy.string({ produce: () => emailIdentity.dkimDnsTokenValue2 })],
                ttl: '1800',
            });
            new route53.CfnRecordSet(emailIdentity, 'DkimDnsToken3', {
                hostedZoneId: hostedZone.hostedZoneId,
                name: core_1.Lazy.string({ produce: () => emailIdentity.dkimDnsTokenName3 }),
                type: 'CNAME',
                resourceRecords: [core_1.Lazy.string({ produce: () => emailIdentity.dkimDnsTokenValue3 })],
                ttl: '1800',
            });
        }
        return this.signingKeyLength
            ? { nextSigningKeyLength: this.signingKeyLength }
            : undefined;
    }
}
class ByoDkim extends DkimIdentity {
    constructor(options) {
        super();
        this.options = options;
    }
    bind(emailIdentity, hostedZone) {
        if (hostedZone && this.options.publicKey) {
            new route53.TxtRecord(emailIdentity, 'DkimTxt', {
                zone: hostedZone,
                recordName: `${this.options.selector}._domainkey`,
                values: [`p=${this.options.publicKey}`],
            });
        }
        return {
            domainSigningPrivateKey: this.options.privateKey.unsafeUnwrap(),
            domainSigningSelector: this.options.selector,
        };
    }
}
/**
 * The signing key length for Easy DKIM
 */
var EasyDkimSigningKeyLength;
(function (EasyDkimSigningKeyLength) {
    /**
     * RSA 1024-bit
     */
    EasyDkimSigningKeyLength["RSA_1024_BIT"] = "RSA_1024_BIT";
    /**
     * RSA 2048-bit
     */
    EasyDkimSigningKeyLength["RSA_2048_BIT"] = "RSA_2048_BIT";
})(EasyDkimSigningKeyLength = exports.EasyDkimSigningKeyLength || (exports.EasyDkimSigningKeyLength = {}));
/**
 * An email identity
 */
class EmailIdentity extends core_1.Resource {
    /**
     * Use an existing email identity
     */
    static fromEmailIdentityName(scope, id, emailIdentityName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.emailIdentityName = emailIdentityName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ses_EmailIdentityProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EmailIdentity);
            }
            throw error;
        }
        const dkimIdentity = props.dkimIdentity ?? DkimIdentity.easyDkim();
        const identity = new ses_generated_1.CfnEmailIdentity(this, 'Resource', {
            emailIdentity: props.identity.value,
            configurationSetAttributes: (0, utils_1.undefinedIfNoKeys)({
                configurationSetName: props.configurationSet?.configurationSetName,
            }),
            dkimAttributes: (0, utils_1.undefinedIfNoKeys)({
                signingEnabled: props.dkimSigning,
            }),
            dkimSigningAttributes: dkimIdentity.bind(this, props.identity.hostedZone),
            feedbackAttributes: (0, utils_1.undefinedIfNoKeys)({
                emailForwardingEnabled: props.feedbackForwarding,
            }),
            mailFromAttributes: (0, utils_1.undefinedIfNoKeys)({
                mailFromDomain: props.mailFromDomain,
                behaviorOnMxFailure: props.mailFromBehaviorOnMxFailure,
            }),
        });
        if (props.mailFromDomain && props.identity.hostedZone) {
            new route53.MxRecord(this, 'MailFromMxRecord', {
                zone: props.identity.hostedZone,
                recordName: props.mailFromDomain,
                values: [{
                        priority: 10,
                        hostName: `feedback-smtp.${core_1.Stack.of(this).region}.amazonses.com`,
                    }],
            });
            new route53.TxtRecord(this, 'MailFromTxtRecord', {
                zone: props.identity.hostedZone,
                recordName: props.mailFromDomain,
                values: ['v=spf1 include:amazonses.com ~all'],
            });
        }
        this.emailIdentityName = identity.ref;
        this.dkimDnsTokenName1 = identity.attrDkimDnsTokenName1;
        this.dkimDnsTokenName2 = identity.attrDkimDnsTokenName2;
        this.dkimDnsTokenName3 = identity.attrDkimDnsTokenName3;
        this.dkimDnsTokenValue1 = identity.attrDkimDnsTokenValue1;
        this.dkimDnsTokenValue2 = identity.attrDkimDnsTokenValue2;
        this.dkimDnsTokenValue3 = identity.attrDkimDnsTokenValue3;
        this.dkimRecords = [
            { name: this.dkimDnsTokenName1, value: this.dkimDnsTokenValue1 },
            { name: this.dkimDnsTokenName2, value: this.dkimDnsTokenValue2 },
            { name: this.dkimDnsTokenName3, value: this.dkimDnsTokenValue3 },
        ];
    }
}
_c = JSII_RTTI_SYMBOL_1;
EmailIdentity[_c] = { fqn: "aws-cdk-lib.aws_ses.EmailIdentity", version: "2.74.0" };
exports.EmailIdentity = EmailIdentity;
//# sourceMappingURL=data:application/json;base64,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