"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsSubscription = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const sns = require("../../aws-sns");
const core_1 = require("../../core");
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sqs_IQueue(queue);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sns_subscriptions_SqsSubscriptionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SqsSubscription);
            }
            throw error;
        }
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic
     */
    bind(topic) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sns_ITopic(topic);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!(this.queue instanceof constructs_1.Construct)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        const snsServicePrincipal = new iam.ServicePrincipal('sns.amazonaws.com');
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        const queuePolicyDependable = this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [snsServicePrincipal],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        })).policyDependable;
        // if the queue is encrypted, add a statement to the key resource policy
        // which allows this topic to decrypt KMS keys
        if (this.queue.encryptionMasterKey) {
            this.queue.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['kms:Decrypt', 'kms:GenerateDataKey'],
                principals: [snsServicePrincipal],
                conditions: core_1.FeatureFlags.of(topic).isEnabled(cxapi.SNS_SUBSCRIPTIONS_SQS_DECRYPTION_POLICY)
                    ? { ArnEquals: { 'aws:SourceArn': topic.topicArn } }
                    : undefined,
            }));
        }
        // if the topic and queue are created in different stacks
        // then we need to make sure the topic is created first
        if (topic instanceof sns.Topic && topic.stack !== this.queue.stack) {
            this.queue.stack.addDependency(topic.stack);
        }
        return {
            subscriberScope: this.queue,
            subscriberId: core_1.Names.nodeUniqueId(topic.node),
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            filterPolicyWithMessageBody: this.props.filterPolicyWithMessageBody,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
            subscriptionDependency: queuePolicyDependable,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            if (topic.stack !== this.queue.stack) {
                // only if we know the region, will not work for
                // env agnostic stacks
                if (!core_1.Token.isUnresolved(topic.env.region) &&
                    (topic.env.region !== this.queue.env.region)) {
                    return topic.env.region;
                }
            }
            return undefined;
        }
        return core_1.Stack.of(topic).splitArn(topic.topicArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).region;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SqsSubscription[_a] = { fqn: "aws-cdk-lib.aws_sns_subscriptions.SqsSubscription", version: "2.74.0" };
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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