"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TopicBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_ResourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, TopicBase);
            }
            throw error;
        }
        this.node.addValidation({ validate: () => this.policy?.document.validateForResourcePolicy() ?? [] });
    }
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(topicSubscription) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sns_ITopicSubscription(topicSubscription);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSubscription);
            }
            throw error;
        }
        const subscriptionConfig = topicSubscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        const subscription = new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
        // Add dependency for the subscription, for example for SQS subscription
        // the queue policy has to deploy before the subscription is created
        if (subscriptionConfig.subscriptionDependency) {
            subscription.node.addDependency(subscriptionConfig.subscriptionDependency);
        }
        return subscription;
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is imported (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToResourcePolicy);
            }
            throw error;
        }
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantPublish);
            }
            throw error;
        }
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    /**
     * Represents a notification target
     * That allows SNS topic to associate with this rule target.
     */
    bindAsNotificationRuleTarget(_scope) {
        // SNS topic need to grant codestar-notifications service to publish
        // @see https://docs.aws.amazon.com/dtconsole/latest/userguide/set-up-sns.html
        this.grantPublish(new iam.ServicePrincipal('codestar-notifications.amazonaws.com'));
        return {
            targetType: 'SNS',
            targetAddress: this.topicArn,
        };
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
_a = JSII_RTTI_SYMBOL_1;
TopicBase[_a] = { fqn: "aws-cdk-lib.aws_sns.TopicBase", version: "2.74.0" };
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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