"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.FifoThroughputLimit = exports.DeduplicationScope = exports.QueueEncryption = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a KMS key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "KMS_MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
    /**
     * Server-side encryption key managed by SQS (SSE-SQS).
     *
     * To learn more about SSE-SQS on Amazon SQS, please visit the
     * [Amazon SQS documentation](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-server-side-encryption.html).
     */
    QueueEncryption["SQS_MANAGED"] = "SQS_MANAGED";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * What kind of deduplication scope to apply
 */
var DeduplicationScope;
(function (DeduplicationScope) {
    /**
     * Deduplication occurs at the message group level
     */
    DeduplicationScope["MESSAGE_GROUP"] = "messageGroup";
    /**
     * Deduplication occurs at the message queue level
     */
    DeduplicationScope["QUEUE"] = "queue";
})(DeduplicationScope = exports.DeduplicationScope || (exports.DeduplicationScope = {}));
/**
 * Whether the FIFO queue throughput quota applies to the entire queue or per message group
 */
var FifoThroughputLimit;
(function (FifoThroughputLimit) {
    /**
     * Throughput quota applies per queue
     */
    FifoThroughputLimit["PER_QUEUE"] = "perQueue";
    /**
     * Throughput quota applies per message group id
     */
    FifoThroughputLimit["PER_MESSAGE_GROUP_ID"] = "perMessageGroupId";
})(FifoThroughputLimit = exports.FifoThroughputLimit || (exports.FifoThroughputLimit = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    /**
     * Import an existing SQS queue provided an ARN
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1)
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sqs_QueueAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromQueueAttributes);
            }
            throw error;
        }
        const stack = core_1.Stack.of(scope);
        const parsedArn = stack.splitArn(attrs.queueArn, core_1.ArnFormat.NO_RESOURCE_NAME);
        const queueName = attrs.queueName || parsedArn.resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${parsedArn.region}.${stack.urlSuffix}/${parsedArn.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = this.determineFifo();
                this.autoCreatePolicy = false;
            }
            /**
             * Determine fifo flag based on queueName and fifo attribute
             */
            determineFifo() {
                if (core_1.Token.isUnresolved(this.queueArn)) {
                    return attrs.fifo || false;
                }
                else {
                    if (typeof attrs.fifo !== 'undefined') {
                        if (attrs.fifo && !queueName.endsWith('.fifo')) {
                            throw new Error("FIFO queue names must end in '.fifo'");
                        }
                        if (!attrs.fifo && queueName.endsWith('.fifo')) {
                            throw new Error("Non-FIFO queue name may not end in '.fifo'");
                        }
                    }
                    return queueName.endsWith('.fifo') ? true : false;
                }
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_sqs_QueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Queue);
            }
            throw error;
        }
        (0, validate_props_1.validateProps)(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        queue.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.DESTROY);
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        this.deadLetterQueue = props.deadLetterQueue;
        function _determineEncryptionProps() {
            let encryption = props.encryption;
            if (encryption === QueueEncryption.SQS_MANAGED && props.encryptionMasterKey) {
                throw new Error("'encryptionMasterKey' is not supported if encryption type 'SQS_MANAGED' is used");
            }
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (!encryption) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return {
                    encryptionProps: {
                        sqsManagedSseEnabled: false,
                    },
                };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.SQS_MANAGED) {
                return {
                    encryptionProps: {
                        sqsManagedSseEnabled: true,
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
        // Enforce encryption of data in transit
        if (props.enforceSSL) {
            this.enforceSSLStatement();
        }
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.deduplicationScope) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.fifoThroughputLimit) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        if (props.deduplicationScope && !fifoQueue) {
            throw new Error('Deduplication scope can only be defined for FIFO queues');
        }
        if (props.fifoThroughputLimit && !fifoQueue) {
            throw new Error('FIFO throughput limit can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            deduplicationScope: props.deduplicationScope,
            fifoThroughputLimit: props.fifoThroughputLimit,
            fifoQueue,
        };
    }
    /**
     * Adds an iam statement to enforce encryption of data in transit.
     */
    enforceSSLStatement() {
        const statement = new iam.PolicyStatement({
            actions: ['sqs:*'],
            conditions: {
                Bool: { 'aws:SecureTransport': 'false' },
            },
            effect: iam.Effect.DENY,
            resources: [this.queueArn],
            principals: [new iam.AnyPrincipal()],
        });
        this.addToResourcePolicy(statement);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Queue[_a] = { fqn: "aws-cdk-lib.aws_sqs.Queue", version: "2.74.0" };
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicXVldWUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJxdWV1ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxxQ0FBcUM7QUFDckMscUNBQXFDO0FBQ3JDLHFDQUE4RTtBQUU5RSw2Q0FBa0U7QUFDbEUsbURBQTJDO0FBQzNDLHFEQUFpRDtBQStMakQ7O0dBRUc7QUFDSCxJQUFZLGVBeUJYO0FBekJELFdBQVksZUFBZTtJQUN6Qjs7T0FFRztJQUNILHVDQUFvQixDQUFBO0lBRXBCOztPQUVHO0lBQ0gsOENBQTJCLENBQUE7SUFFM0I7Ozs7T0FJRztJQUNILDhCQUFXLENBQUE7SUFFWDs7Ozs7T0FLRztJQUNILDhDQUEyQixDQUFBO0FBQzdCLENBQUMsRUF6QlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUF5QjFCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLGtCQVNYO0FBVEQsV0FBWSxrQkFBa0I7SUFDNUI7O09BRUc7SUFDSCxvREFBOEIsQ0FBQTtJQUM5Qjs7T0FFRztJQUNILHFDQUFlLENBQUE7QUFDakIsQ0FBQyxFQVRXLGtCQUFrQixHQUFsQiwwQkFBa0IsS0FBbEIsMEJBQWtCLFFBUzdCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLG1CQVNYO0FBVEQsV0FBWSxtQkFBbUI7SUFDN0I7O09BRUc7SUFDSCw2Q0FBc0IsQ0FBQTtJQUN0Qjs7T0FFRztJQUNILGlFQUEwQyxDQUFBO0FBQzVDLENBQUMsRUFUVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQVM5QjtBQUVEOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsc0JBQVM7SUFFbEM7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUN2RSxPQUFPLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztLQUMzRDtJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQ3BGLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUIsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLGdCQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUM3RSxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUM7UUFDeEQsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxlQUFlLFNBQVMsQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxTQUFTLENBQUMsT0FBTyxJQUFJLFNBQVMsRUFBRSxDQUFDO1FBRTFILE1BQU0sTUFBTyxTQUFRLHNCQUFTO1lBQTlCOztnQkFDa0IsYUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyw0Q0FBNEM7Z0JBQ3ZFLGFBQVEsR0FBRyxRQUFRLENBQUM7Z0JBQ3BCLGNBQVMsR0FBRyxTQUFTLENBQUM7Z0JBQ3RCLHdCQUFtQixHQUFHLEtBQUssQ0FBQyxNQUFNO29CQUNoRCxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDO29CQUMvQyxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNFLFNBQUksR0FBWSxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7Z0JBRWxDLHFCQUFnQixHQUFHLEtBQUssQ0FBQztZQW9COUMsQ0FBQztZQWxCQzs7ZUFFRztZQUNLLGFBQWE7Z0JBQ25CLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7b0JBQ3JDLE9BQU8sS0FBSyxDQUFDLElBQUksSUFBSSxLQUFLLENBQUM7aUJBQzVCO3FCQUFNO29CQUNMLElBQUksT0FBTyxLQUFLLENBQUMsSUFBSSxLQUFLLFdBQVcsRUFBRTt3QkFDckMsSUFBSSxLQUFLLENBQUMsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRTs0QkFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO3lCQUN6RDt3QkFDRCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksSUFBSSxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFOzRCQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7eUJBQy9EO3FCQUNGO29CQUNELE9BQU8sU0FBUyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7aUJBQ25EO1lBQ0gsQ0FBQztTQUNGO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUFrQ0QsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFvQixFQUFFO1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzlCLENBQUMsQ0FBQztRQUxjLHFCQUFnQixHQUFHLElBQUksQ0FBQzs7Ozs7OytDQXRGaEMsS0FBSzs7OztRQTZGZCxJQUFBLDhCQUFhLEVBQUMsS0FBSyxDQUFDLENBQUM7UUFFckIsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLGVBQWU7WUFDekMsQ0FBQyxDQUFDO2dCQUNBLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLFFBQVE7Z0JBQ3pELGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZSxDQUFDLGVBQWU7YUFDdkQ7WUFDRCxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRWQsTUFBTSxFQUFFLG1CQUFtQixFQUFFLGVBQWUsRUFBRSxHQUFHLHlCQUF5QixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV0RixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLElBQUksR0FBRyxTQUFTLENBQUMsU0FBUyxJQUFJLEtBQUssQ0FBQztRQUV6QyxNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMzQyxTQUFTLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDNUIsR0FBRyxTQUFTO1lBQ1osR0FBRyxlQUFlO1lBQ2xCLGFBQWE7WUFDYixZQUFZLEVBQUUsS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRTtZQUNwRSxrQkFBa0IsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzdDLHNCQUFzQixFQUFFLEtBQUssQ0FBQyxlQUFlLElBQUksS0FBSyxDQUFDLGVBQWUsQ0FBQyxTQUFTLEVBQUU7WUFDbEYsNkJBQTZCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQixJQUFJLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxTQUFTLEVBQUU7WUFDdkcsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUU7U0FDbEYsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxhQUFhLElBQUksb0JBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV2RSxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQzFELE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQzVCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsbUJBQW1CLEdBQUcsbUJBQW1CLENBQUM7UUFDL0MsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDO1FBQzFCLElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztRQUU3QyxTQUFTLHlCQUF5QjtZQUNoQyxJQUFJLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1lBRWxDLElBQUksVUFBVSxLQUFLLGVBQWUsQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLG1CQUFtQixFQUFFO2dCQUMzRSxNQUFNLElBQUksS0FBSyxDQUFDLGlGQUFpRixDQUFDLENBQUM7YUFDcEc7WUFFRCxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsR0FBRyxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtnQkFDbkUsVUFBVSxHQUFHLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxpREFBaUQ7YUFDcEY7WUFFRCxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNmLE9BQU8sRUFBRSxlQUFlLEVBQUUsRUFBRSxFQUFFLENBQUM7YUFDaEM7WUFFRCxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsV0FBVyxFQUFFO2dCQUM5QyxPQUFPO29CQUNMLGVBQWUsRUFBRTt3QkFDZixvQkFBb0IsRUFBRSxLQUFLO3FCQUM1QjtpQkFDRixDQUFDO2FBQ0g7WUFFRCxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsV0FBVyxFQUFFO2dCQUM5QyxPQUFPO29CQUNMLGVBQWUsRUFBRTt3QkFDZixjQUFjLEVBQUUsZUFBZTt3QkFDL0IsNEJBQTRCLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRTtxQkFDbkY7aUJBQ0YsQ0FBQzthQUNIO1lBRUQsSUFBSSxVQUFVLEtBQUssZUFBZSxDQUFDLEdBQUcsRUFBRTtnQkFDdEMsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO29CQUN0RSxXQUFXLEVBQUUsY0FBYyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtpQkFDNUMsQ0FBQyxDQUFDO2dCQUVILE9BQU87b0JBQ0wsbUJBQW1CLEVBQUUsU0FBUztvQkFDOUIsZUFBZSxFQUFFO3dCQUNmLGNBQWMsRUFBRSxTQUFTLENBQUMsTUFBTTt3QkFDaEMsNEJBQTRCLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRTtxQkFDbkY7aUJBQ0YsQ0FBQzthQUNIO1lBRUQsSUFBSSxVQUFVLEtBQUssZUFBZSxDQUFDLFdBQVcsRUFBRTtnQkFDOUMsT0FBTztvQkFDTCxlQUFlLEVBQUU7d0JBQ2Ysb0JBQW9CLEVBQUUsSUFBSTtxQkFDM0I7aUJBQ0YsQ0FBQzthQUNIO1lBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUNoRSxDQUFDO1FBRUQsd0NBQXdDO1FBQ3hDLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNwQixJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztTQUM1QjtLQUNGO0lBRUQ7O09BRUc7SUFDSyxrQkFBa0IsQ0FBQyxLQUFpQjtRQUMxQywwRUFBMEU7UUFDMUUsSUFBSSxTQUFTLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztRQUMzQixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2xDLElBQUksT0FBTyxTQUFTLEtBQUssV0FBVyxJQUFJLFNBQVMsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksU0FBUyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUFFLFNBQVMsR0FBRyxJQUFJLENBQUM7U0FBRTtRQUN6SSxJQUFJLE9BQU8sU0FBUyxLQUFLLFdBQVcsSUFBSSxLQUFLLENBQUMseUJBQXlCLEVBQUU7WUFBRSxTQUFTLEdBQUcsSUFBSSxDQUFDO1NBQUU7UUFDOUYsSUFBSSxPQUFPLFNBQVMsS0FBSyxXQUFXLElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQUUsU0FBUyxHQUFHLElBQUksQ0FBQztTQUFFO1FBQ3ZGLElBQUksT0FBTyxTQUFTLEtBQUssV0FBVyxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtZQUFFLFNBQVMsR0FBRyxJQUFJLENBQUM7U0FBRTtRQUV4Riw4REFBOEQ7UUFDOUQsSUFBSSxPQUFPLFNBQVMsS0FBSyxRQUFRLEVBQUU7WUFDakMsSUFBSSxTQUFTLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUM3QyxNQUFNLElBQUksS0FBSyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7YUFDekQ7WUFDRCxJQUFJLENBQUMsU0FBUyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7Z0JBQzdDLE1BQU0sSUFBSSxLQUFLLENBQUMsNENBQTRDLENBQUMsQ0FBQzthQUMvRDtTQUNGO1FBRUQsSUFBSSxLQUFLLENBQUMseUJBQXlCLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDakQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1NBQ3BGO1FBRUQsSUFBSSxLQUFLLENBQUMsa0JBQWtCLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5REFBeUQsQ0FBQyxDQUFDO1NBQzVFO1FBRUQsSUFBSSxLQUFLLENBQUMsbUJBQW1CLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsT0FBTztZQUNMLHlCQUF5QixFQUFFLEtBQUssQ0FBQyx5QkFBeUI7WUFDMUQsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUM1QyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQzlDLFNBQVM7U0FDVixDQUFDO0tBQ0g7SUFFRDs7T0FFRztJQUNLLG1CQUFtQjtRQUN6QixNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDeEMsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDO1lBQ2xCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUUsRUFBRSxxQkFBcUIsRUFBRSxPQUFPLEVBQUU7YUFDekM7WUFDRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJO1lBQ3ZCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDMUIsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDckMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQ3JDOzs7O0FBeFBVLHNCQUFLIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJy4uLy4uL2F3cy1rbXMnO1xuaW1wb3J0IHsgRHVyYXRpb24sIFJlbW92YWxQb2xpY3ksIFN0YWNrLCBUb2tlbiwgQXJuRm9ybWF0IH0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IElRdWV1ZSwgUXVldWVBdHRyaWJ1dGVzLCBRdWV1ZUJhc2UgfSBmcm9tICcuL3F1ZXVlLWJhc2UnO1xuaW1wb3J0IHsgQ2ZuUXVldWUgfSBmcm9tICcuL3Nxcy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgdmFsaWRhdGVQcm9wcyB9IGZyb20gJy4vdmFsaWRhdGUtcHJvcHMnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGEgbmV3IFF1ZXVlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUXVldWVQcm9wcyB7XG4gIC8qKlxuICAgKiBBIG5hbWUgZm9yIHRoZSBxdWV1ZS5cbiAgICpcbiAgICogSWYgc3BlY2lmaWVkIGFuZCB0aGlzIGlzIGEgRklGTyBxdWV1ZSwgbXVzdCBlbmQgaW4gdGhlIHN0cmluZyAnLmZpZm8nLlxuICAgKlxuICAgKiBAZGVmYXVsdCBDbG91ZEZvcm1hdGlvbi1nZW5lcmF0ZWQgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgcXVldWVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHNlY29uZHMgdGhhdCBBbWF6b24gU1FTIHJldGFpbnMgYSBtZXNzYWdlLlxuICAgKlxuICAgKiBZb3UgY2FuIHNwZWNpZnkgYW4gaW50ZWdlciB2YWx1ZSBmcm9tIDYwIHNlY29uZHMgKDEgbWludXRlKSB0byAxMjA5NjAwXG4gICAqIHNlY29uZHMgKDE0IGRheXMpLiBUaGUgZGVmYXVsdCB2YWx1ZSBpcyAzNDU2MDAgc2Vjb25kcyAoNCBkYXlzKS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uZGF5cyg0KVxuICAgKi9cbiAgcmVhZG9ubHkgcmV0ZW50aW9uUGVyaW9kPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIGluIHNlY29uZHMgdGhhdCB0aGUgZGVsaXZlcnkgb2YgYWxsIG1lc3NhZ2VzIGluIHRoZSBxdWV1ZSBpcyBkZWxheWVkLlxuICAgKlxuICAgKiBZb3UgY2FuIHNwZWNpZnkgYW4gaW50ZWdlciB2YWx1ZSBvZiAwIHRvIDkwMCAoMTUgbWludXRlcykuIFRoZSBkZWZhdWx0XG4gICAqIHZhbHVlIGlzIDAuXG4gICAqXG4gICAqIEBkZWZhdWx0IDBcbiAgICovXG4gIHJlYWRvbmx5IGRlbGl2ZXJ5RGVsYXk/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIGxpbWl0IG9mIGhvdyBtYW55IGJ5dGVzIHRoYXQgYSBtZXNzYWdlIGNhbiBjb250YWluIGJlZm9yZSBBbWF6b24gU1FTIHJlamVjdHMgaXQuXG4gICAqXG4gICAqIFlvdSBjYW4gc3BlY2lmeSBhbiBpbnRlZ2VyIHZhbHVlIGZyb20gMTAyNCBieXRlcyAoMSBLaUIpIHRvIDI2MjE0NCBieXRlc1xuICAgKiAoMjU2IEtpQikuIFRoZSBkZWZhdWx0IHZhbHVlIGlzIDI2MjE0NCAoMjU2IEtpQikuXG4gICAqXG4gICAqIEBkZWZhdWx0IDI1NktpQlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4TWVzc2FnZVNpemVCeXRlcz86IG51bWJlcjtcblxuICAvKipcbiAgICogRGVmYXVsdCB3YWl0IHRpbWUgZm9yIFJlY2VpdmVNZXNzYWdlIGNhbGxzLlxuICAgKlxuICAgKiBEb2VzIG5vdCB3YWl0IGlmIHNldCB0byAwLCBvdGhlcndpc2Ugd2FpdHMgdGhpcyBhbW91bnQgb2Ygc2Vjb25kc1xuICAgKiBieSBkZWZhdWx0IGZvciBtZXNzYWdlcyB0byBhcnJpdmUuXG4gICAqXG4gICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgQW1hem9uIFNRUyBMb25nIFBvbGwuXG4gICAqXG4gICAqICBAZGVmYXVsdCAwXG4gICAqL1xuICByZWFkb25seSByZWNlaXZlTWVzc2FnZVdhaXRUaW1lPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRpbWVvdXQgb2YgcHJvY2Vzc2luZyBhIHNpbmdsZSBtZXNzYWdlLlxuICAgKlxuICAgKiBBZnRlciBkZXF1ZXVpbmcsIHRoZSBwcm9jZXNzb3IgaGFzIHRoaXMgbXVjaCB0aW1lIHRvIGhhbmRsZSB0aGUgbWVzc2FnZVxuICAgKiBhbmQgZGVsZXRlIGl0IGZyb20gdGhlIHF1ZXVlIGJlZm9yZSBpdCBiZWNvbWVzIHZpc2libGUgYWdhaW4gZm9yIGRlcXVldWVpbmdcbiAgICogYnkgYW5vdGhlciBwcm9jZXNzb3IuXG4gICAqXG4gICAqIFZhbHVlcyBtdXN0IGJlIGZyb20gMCB0byA0MzIwMCBzZWNvbmRzICgxMiBob3VycykuIElmIHlvdSBkb24ndCBzcGVjaWZ5XG4gICAqIGEgdmFsdWUsIEFXUyBDbG91ZEZvcm1hdGlvbiB1c2VzIHRoZSBkZWZhdWx0IHZhbHVlIG9mIDMwIHNlY29uZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLnNlY29uZHMoMzApXG4gICAqL1xuICByZWFkb25seSB2aXNpYmlsaXR5VGltZW91dD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBTZW5kIG1lc3NhZ2VzIHRvIHRoaXMgcXVldWUgaWYgdGhleSB3ZXJlIHVuc3VjY2Vzc2Z1bGx5IGRlcXVldWVkIGEgbnVtYmVyIG9mIHRpbWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBubyBkZWFkLWxldHRlciBxdWV1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVhZExldHRlclF1ZXVlPzogRGVhZExldHRlclF1ZXVlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBjb250ZW50cyBvZiB0aGUgcXVldWUgYXJlIGVuY3J5cHRlZCwgYW5kIGJ5IHdoYXQgdHlwZSBvZiBrZXkuXG4gICAqXG4gICAqIEJlIGF3YXJlIHRoYXQgZW5jcnlwdGlvbiBpcyBub3QgYXZhaWxhYmxlIGluIGFsbCByZWdpb25zLCBwbGVhc2Ugc2VlIHRoZSBkb2NzXG4gICAqIGZvciBjdXJyZW50IGF2YWlsYWJpbGl0eSBkZXRhaWxzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTUVNfTUFOQUdFRCAoU1NFLVNRUykgZm9yIG5ld2x5IGNyZWF0ZWQgcXVldWVzXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uPzogUXVldWVFbmNyeXB0aW9uO1xuXG4gIC8qKlxuICAgKiBFeHRlcm5hbCBLTVMga2V5IHRvIHVzZSBmb3IgcXVldWUgZW5jcnlwdGlvbi5cbiAgICpcbiAgICogSW5kaXZpZHVhbCBtZXNzYWdlcyB3aWxsIGJlIGVuY3J5cHRlZCB1c2luZyBkYXRhIGtleXMuIFRoZSBkYXRhIGtleXMgaW5cbiAgICogdHVybiB3aWxsIGJlIGVuY3J5cHRlZCB1c2luZyB0aGlzIGtleSwgYW5kIHJldXNlZCBmb3IgYSBtYXhpbXVtIG9mXG4gICAqIGBkYXRhS2V5UmV1c2VTZWNzYCBzZWNvbmRzLlxuICAgKlxuICAgKiBJZiB0aGUgJ2VuY3J5cHRpb25NYXN0ZXJLZXknIHByb3BlcnR5IGlzIHNldCwgJ2VuY3J5cHRpb24nIHR5cGUgd2lsbCBiZVxuICAgKiBpbXBsaWNpdGx5IHNldCB0byBcIktNU1wiLlxuICAgKlxuICAgKiBAZGVmYXVsdCBJZiBlbmNyeXB0aW9uIGlzIHNldCB0byBLTVMgYW5kIG5vdCBzcGVjaWZpZWQsIGEga2V5IHdpbGwgYmUgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25NYXN0ZXJLZXk/OiBrbXMuSUtleTtcblxuICAvKipcbiAgICogVGhlIGxlbmd0aCBvZiB0aW1lIHRoYXQgQW1hem9uIFNRUyByZXVzZXMgYSBkYXRhIGtleSBiZWZvcmUgY2FsbGluZyBLTVMgYWdhaW4uXG4gICAqXG4gICAqIFRoZSB2YWx1ZSBtdXN0IGJlIGFuIGludGVnZXIgYmV0d2VlbiA2MCAoMSBtaW51dGUpIGFuZCA4Niw0MDAgKDI0XG4gICAqIGhvdXJzKS4gVGhlIGRlZmF1bHQgaXMgMzAwICg1IG1pbnV0ZXMpLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5taW51dGVzKDUpXG4gICAqL1xuICByZWFkb25seSBkYXRhS2V5UmV1c2U/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogV2hldGhlciB0aGlzIGEgZmlyc3QtaW4tZmlyc3Qtb3V0IChGSUZPKSBxdWV1ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2UsIHVubGVzcyBxdWV1ZU5hbWUgZW5kcyBpbiAnLmZpZm8nIG9yICdjb250ZW50QmFzZWREZWR1cGxpY2F0aW9uJyBpcyB0cnVlLlxuICAgKi9cbiAgcmVhZG9ubHkgZmlmbz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRvIGVuYWJsZSBjb250ZW50LWJhc2VkIGRlZHVwbGljYXRpb24uXG4gICAqXG4gICAqIER1cmluZyB0aGUgZGVkdXBsaWNhdGlvbiBpbnRlcnZhbCAoNSBtaW51dGVzKSwgQW1hem9uIFNRUyB0cmVhdHNcbiAgICogbWVzc2FnZXMgdGhhdCBhcmUgc2VudCB3aXRoIGlkZW50aWNhbCBjb250ZW50IChleGNsdWRpbmcgYXR0cmlidXRlcykgYXNcbiAgICogZHVwbGljYXRlcyBhbmQgZGVsaXZlcnMgb25seSBvbmUgY29weSBvZiB0aGUgbWVzc2FnZS5cbiAgICpcbiAgICogSWYgeW91IGRvbid0IGVuYWJsZSBjb250ZW50LWJhc2VkIGRlZHVwbGljYXRpb24gYW5kIHlvdSB3YW50IHRvIGRlZHVwbGljYXRlXG4gICAqIG1lc3NhZ2VzLCBwcm92aWRlIGFuIGV4cGxpY2l0IGRlZHVwbGljYXRpb24gSUQgaW4geW91ciBTZW5kTWVzc2FnZSgpIGNhbGwuXG4gICAqXG4gICAqIChPbmx5IGFwcGxpZXMgdG8gRklGTyBxdWV1ZXMuKVxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY29udGVudEJhc2VkRGVkdXBsaWNhdGlvbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEZvciBoaWdoIHRocm91Z2hwdXQgZm9yIEZJRk8gcXVldWVzLCBzcGVjaWZpZXMgd2hldGhlciBtZXNzYWdlIGRlZHVwbGljYXRpb25cbiAgICogb2NjdXJzIGF0IHRoZSBtZXNzYWdlIGdyb3VwIG9yIHF1ZXVlIGxldmVsLlxuICAgKlxuICAgKiAoT25seSBhcHBsaWVzIHRvIEZJRk8gcXVldWVzLilcbiAgICpcbiAgICogQGRlZmF1bHQgRGVkdXBsaWNhdGlvblNjb3BlLlFVRVVFXG4gICAqL1xuICByZWFkb25seSBkZWR1cGxpY2F0aW9uU2NvcGU/OiBEZWR1cGxpY2F0aW9uU2NvcGU7XG5cbiAgLyoqXG4gICAqIEZvciBoaWdoIHRocm91Z2hwdXQgZm9yIEZJRk8gcXVldWVzLCBzcGVjaWZpZXMgd2hldGhlciB0aGUgRklGTyBxdWV1ZVxuICAgKiB0aHJvdWdocHV0IHF1b3RhIGFwcGxpZXMgdG8gdGhlIGVudGlyZSBxdWV1ZSBvciBwZXIgbWVzc2FnZSBncm91cC5cbiAgICpcbiAgICogKE9ubHkgYXBwbGllcyB0byBGSUZPIHF1ZXVlcy4pXG4gICAqXG4gICAqIEBkZWZhdWx0IEZpZm9UaHJvdWdocHV0TGltaXQuUEVSX1FVRVVFXG4gICAqL1xuICByZWFkb25seSBmaWZvVGhyb3VnaHB1dExpbWl0PzogRmlmb1Rocm91Z2hwdXRMaW1pdDtcblxuICAvKipcbiAgICogUG9saWN5IHRvIGFwcGx5IHdoZW4gdGhlIHF1ZXVlIGlzIHJlbW92ZWQgZnJvbSB0aGUgc3RhY2tcbiAgICpcbiAgICogRXZlbiB0aG91Z2ggcXVldWVzIGFyZSB0ZWNobmljYWxseSBzdGF0ZWZ1bCwgdGhlaXIgY29udGVudHMgYXJlIHRyYW5zaWVudCBhbmQgaXRcbiAgICogaXMgY29tbW9uIHRvIGFkZCBhbmQgcmVtb3ZlIFF1ZXVlcyB3aGlsZSByZWFyY2hpdGVjdGluZyB5b3VyIGFwcGxpY2F0aW9uLiBUaGVcbiAgICogZGVmYXVsdCBpcyB0aGVyZWZvcmUgYERFU1RST1lgLiBDaGFuZ2UgaXQgdG8gYFJFVEFJTmAgaWYgdGhlIG1lc3NhZ2VzIGFyZSBzb1xuICAgKiB2YWx1YWJsZSB0aGF0IGFjY2lkZW50YWxseSBsb3NpbmcgdGhlbSB3b3VsZCBiZSB1bmFjY2VwdGFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJlbW92YWxQb2xpY3kuREVTVFJPWVxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG5cbiAgLyoqXG4gICAqIEVuZm9yY2UgZW5jcnlwdGlvbiBvZiBkYXRhIGluIHRyYW5zaXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU1NpbXBsZVF1ZXVlU2VydmljZS9sYXRlc3QvU1FTRGV2ZWxvcGVyR3VpZGUvc3FzLXNlY3VyaXR5LWJlc3QtcHJhY3RpY2VzLmh0bWwjZW5mb3JjZS1lbmNyeXB0aW9uLWRhdGEtaW4tdHJhbnNpdFxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5mb3JjZVNTTD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogRGVhZCBsZXR0ZXIgcXVldWUgc2V0dGluZ3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZWFkTGV0dGVyUXVldWUge1xuICAvKipcbiAgICogVGhlIGRlYWQtbGV0dGVyIHF1ZXVlIHRvIHdoaWNoIEFtYXpvbiBTUVMgbW92ZXMgbWVzc2FnZXMgYWZ0ZXIgdGhlIHZhbHVlIG9mIG1heFJlY2VpdmVDb3VudCBpcyBleGNlZWRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHF1ZXVlOiBJUXVldWU7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgdGltZXMgYSBtZXNzYWdlIGNhbiBiZSB1bnN1Y2Nlc2Z1bGx5IGRlcXVldWVkIGJlZm9yZSBiZWluZyBtb3ZlZCB0byB0aGUgZGVhZC1sZXR0ZXIgcXVldWUuXG4gICAqL1xuICByZWFkb25seSBtYXhSZWNlaXZlQ291bnQ6IG51bWJlcjtcbn1cblxuLyoqXG4gKiBXaGF0IGtpbmQgb2YgZW5jcnlwdGlvbiB0byBhcHBseSB0byB0aGlzIHF1ZXVlXG4gKi9cbmV4cG9ydCBlbnVtIFF1ZXVlRW5jcnlwdGlvbiB7XG4gIC8qKlxuICAgKiBNZXNzYWdlcyBpbiB0aGUgcXVldWUgYXJlIG5vdCBlbmNyeXB0ZWRcbiAgICovXG4gIFVORU5DUllQVEVEID0gJ05PTkUnLFxuXG4gIC8qKlxuICAgKiBTZXJ2ZXItc2lkZSBLTVMgZW5jcnlwdGlvbiB3aXRoIGEgS01TIGtleSBtYW5hZ2VkIGJ5IFNRUy5cbiAgICovXG4gIEtNU19NQU5BR0VEID0gJ0tNU19NQU5BR0VEJyxcblxuICAvKipcbiAgICogU2VydmVyLXNpZGUgZW5jcnlwdGlvbiB3aXRoIGEgS01TIGtleSBtYW5hZ2VkIGJ5IHRoZSB1c2VyLlxuICAgKlxuICAgKiBJZiBgZW5jcnlwdGlvbktleWAgaXMgc3BlY2lmaWVkLCB0aGlzIGtleSB3aWxsIGJlIHVzZWQsIG90aGVyd2lzZSwgb25lIHdpbGwgYmUgZGVmaW5lZC5cbiAgICovXG4gIEtNUyA9ICdLTVMnLFxuXG4gIC8qKlxuICAgKiBTZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGtleSBtYW5hZ2VkIGJ5IFNRUyAoU1NFLVNRUykuXG4gICAqXG4gICAqIFRvIGxlYXJuIG1vcmUgYWJvdXQgU1NFLVNRUyBvbiBBbWF6b24gU1FTLCBwbGVhc2UgdmlzaXQgdGhlXG4gICAqIFtBbWF6b24gU1FTIGRvY3VtZW50YXRpb25dKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NTaW1wbGVRdWV1ZVNlcnZpY2UvbGF0ZXN0L1NRU0RldmVsb3Blckd1aWRlL3Nxcy1zZXJ2ZXItc2lkZS1lbmNyeXB0aW9uLmh0bWwpLlxuICAgKi9cbiAgU1FTX01BTkFHRUQgPSAnU1FTX01BTkFHRUQnXG59XG5cbi8qKlxuICogV2hhdCBraW5kIG9mIGRlZHVwbGljYXRpb24gc2NvcGUgdG8gYXBwbHlcbiAqL1xuZXhwb3J0IGVudW0gRGVkdXBsaWNhdGlvblNjb3BlIHtcbiAgLyoqXG4gICAqIERlZHVwbGljYXRpb24gb2NjdXJzIGF0IHRoZSBtZXNzYWdlIGdyb3VwIGxldmVsXG4gICAqL1xuICBNRVNTQUdFX0dST1VQID0gJ21lc3NhZ2VHcm91cCcsXG4gIC8qKlxuICAgKiBEZWR1cGxpY2F0aW9uIG9jY3VycyBhdCB0aGUgbWVzc2FnZSBxdWV1ZSBsZXZlbFxuICAgKi9cbiAgUVVFVUUgPSAncXVldWUnLFxufVxuXG4vKipcbiAqIFdoZXRoZXIgdGhlIEZJRk8gcXVldWUgdGhyb3VnaHB1dCBxdW90YSBhcHBsaWVzIHRvIHRoZSBlbnRpcmUgcXVldWUgb3IgcGVyIG1lc3NhZ2UgZ3JvdXBcbiAqL1xuZXhwb3J0IGVudW0gRmlmb1Rocm91Z2hwdXRMaW1pdCB7XG4gIC8qKlxuICAgKiBUaHJvdWdocHV0IHF1b3RhIGFwcGxpZXMgcGVyIHF1ZXVlXG4gICAqL1xuICBQRVJfUVVFVUUgPSAncGVyUXVldWUnLFxuICAvKipcbiAgICogVGhyb3VnaHB1dCBxdW90YSBhcHBsaWVzIHBlciBtZXNzYWdlIGdyb3VwIGlkXG4gICAqL1xuICBQRVJfTUVTU0FHRV9HUk9VUF9JRCA9ICdwZXJNZXNzYWdlR3JvdXBJZCcsXG59XG5cbi8qKlxuICogQSBuZXcgQW1hem9uIFNRUyBxdWV1ZVxuICovXG5leHBvcnQgY2xhc3MgUXVldWUgZXh0ZW5kcyBRdWV1ZUJhc2Uge1xuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgU1FTIHF1ZXVlIHByb3ZpZGVkIGFuIEFSTlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgVGhlIHBhcmVudCBjcmVhdGluZyBjb25zdHJ1Y3RcbiAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QncyBuYW1lXG4gICAqIEBwYXJhbSBxdWV1ZUFybiBxdWV1ZSBBUk4gKGkuZS4gYXJuOmF3czpzcXM6dXMtZWFzdC0yOjQ0NDQ1NTU1NjY2NjpxdWV1ZTEpXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21RdWV1ZUFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBxdWV1ZUFybjogc3RyaW5nKTogSVF1ZXVlIHtcbiAgICByZXR1cm4gUXVldWUuZnJvbVF1ZXVlQXR0cmlidXRlcyhzY29wZSwgaWQsIHsgcXVldWVBcm4gfSk7XG4gIH1cblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIHF1ZXVlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21RdWV1ZUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFF1ZXVlQXR0cmlidXRlcyk6IElRdWV1ZSB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihzY29wZSk7XG4gICAgY29uc3QgcGFyc2VkQXJuID0gc3RhY2suc3BsaXRBcm4oYXR0cnMucXVldWVBcm4sIEFybkZvcm1hdC5OT19SRVNPVVJDRV9OQU1FKTtcbiAgICBjb25zdCBxdWV1ZU5hbWUgPSBhdHRycy5xdWV1ZU5hbWUgfHwgcGFyc2VkQXJuLnJlc291cmNlO1xuICAgIGNvbnN0IHF1ZXVlVXJsID0gYXR0cnMucXVldWVVcmwgfHwgYGh0dHBzOi8vc3FzLiR7cGFyc2VkQXJuLnJlZ2lvbn0uJHtzdGFjay51cmxTdWZmaXh9LyR7cGFyc2VkQXJuLmFjY291bnR9LyR7cXVldWVOYW1lfWA7XG5cbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBRdWV1ZUJhc2Uge1xuICAgICAgcHVibGljIHJlYWRvbmx5IHF1ZXVlQXJuID0gYXR0cnMucXVldWVBcm47IC8vIGFybjphd3M6c3FzOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6cXVldWUxXG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcXVldWVVcmwgPSBxdWV1ZVVybDtcbiAgICAgIHB1YmxpYyByZWFkb25seSBxdWV1ZU5hbWUgPSBxdWV1ZU5hbWU7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZW5jcnlwdGlvbk1hc3RlcktleSA9IGF0dHJzLmtleUFyblxuICAgICAgICA/IGttcy5LZXkuZnJvbUtleUFybih0aGlzLCAnS2V5JywgYXR0cnMua2V5QXJuKVxuICAgICAgICA6IHVuZGVmaW5lZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSBmaWZvOiBib29sZWFuID0gdGhpcy5kZXRlcm1pbmVGaWZvKCk7XG5cbiAgICAgIHByb3RlY3RlZCByZWFkb25seSBhdXRvQ3JlYXRlUG9saWN5ID0gZmFsc2U7XG5cbiAgICAgIC8qKlxuICAgICAgICogRGV0ZXJtaW5lIGZpZm8gZmxhZyBiYXNlZCBvbiBxdWV1ZU5hbWUgYW5kIGZpZm8gYXR0cmlidXRlXG4gICAgICAgKi9cbiAgICAgIHByaXZhdGUgZGV0ZXJtaW5lRmlmbygpOiBib29sZWFuIHtcbiAgICAgICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZCh0aGlzLnF1ZXVlQXJuKSkge1xuICAgICAgICAgIHJldHVybiBhdHRycy5maWZvIHx8IGZhbHNlO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIGlmICh0eXBlb2YgYXR0cnMuZmlmbyAhPT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgICAgICAgIGlmIChhdHRycy5maWZvICYmICFxdWV1ZU5hbWUuZW5kc1dpdGgoJy5maWZvJykpIHtcbiAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiRklGTyBxdWV1ZSBuYW1lcyBtdXN0IGVuZCBpbiAnLmZpZm8nXCIpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgaWYgKCFhdHRycy5maWZvICYmIHF1ZXVlTmFtZS5lbmRzV2l0aCgnLmZpZm8nKSkge1xuICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCJOb24tRklGTyBxdWV1ZSBuYW1lIG1heSBub3QgZW5kIGluICcuZmlmbydcIik7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICAgIHJldHVybiBxdWV1ZU5hbWUuZW5kc1dpdGgoJy5maWZvJykgPyB0cnVlIDogZmFsc2U7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhpcyBxdWV1ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHF1ZXVlQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoaXMgcXVldWVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBxdWV1ZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFVSTCBvZiB0aGlzIHF1ZXVlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcXVldWVVcmw6IHN0cmluZztcblxuICAvKipcbiAgICogSWYgdGhpcyBxdWV1ZSBpcyBlbmNyeXB0ZWQsIHRoaXMgaXMgdGhlIEtNUyBrZXkuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW5jcnlwdGlvbk1hc3RlcktleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgcXVldWUgaXMgYW4gQW1hem9uIFNRUyBGSUZPIHF1ZXVlLiBJZiBmYWxzZSwgdGhpcyBpcyBhIHN0YW5kYXJkIHF1ZXVlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZpZm86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIElmIHRoaXMgcXVldWUgaXMgY29uZmlndXJlZCB3aXRoIGEgZGVhZC1sZXR0ZXIgcXVldWUsIHRoaXMgaXMgdGhlIGRlYWQtbGV0dGVyIHF1ZXVlIHNldHRpbmdzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlYWRMZXR0ZXJRdWV1ZT86IERlYWRMZXR0ZXJRdWV1ZTtcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgYXV0b0NyZWF0ZVBvbGljeSA9IHRydWU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFF1ZXVlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5xdWV1ZU5hbWUsXG4gICAgfSk7XG5cbiAgICB2YWxpZGF0ZVByb3BzKHByb3BzKTtcblxuICAgIGNvbnN0IHJlZHJpdmVQb2xpY3kgPSBwcm9wcy5kZWFkTGV0dGVyUXVldWVcbiAgICAgID8ge1xuICAgICAgICBkZWFkTGV0dGVyVGFyZ2V0QXJuOiBwcm9wcy5kZWFkTGV0dGVyUXVldWUucXVldWUucXVldWVBcm4sXG4gICAgICAgIG1heFJlY2VpdmVDb3VudDogcHJvcHMuZGVhZExldHRlclF1ZXVlLm1heFJlY2VpdmVDb3VudCxcbiAgICAgIH1cbiAgICAgIDogdW5kZWZpbmVkO1xuXG4gICAgY29uc3QgeyBlbmNyeXB0aW9uTWFzdGVyS2V5LCBlbmNyeXB0aW9uUHJvcHMgfSA9IF9kZXRlcm1pbmVFbmNyeXB0aW9uUHJvcHMuY2FsbCh0aGlzKTtcblxuICAgIGNvbnN0IGZpZm9Qcm9wcyA9IHRoaXMuZGV0ZXJtaW5lRmlmb1Byb3BzKHByb3BzKTtcbiAgICB0aGlzLmZpZm8gPSBmaWZvUHJvcHMuZmlmb1F1ZXVlIHx8IGZhbHNlO1xuXG4gICAgY29uc3QgcXVldWUgPSBuZXcgQ2ZuUXVldWUodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgcXVldWVOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIC4uLmZpZm9Qcm9wcyxcbiAgICAgIC4uLmVuY3J5cHRpb25Qcm9wcyxcbiAgICAgIHJlZHJpdmVQb2xpY3ksXG4gICAgICBkZWxheVNlY29uZHM6IHByb3BzLmRlbGl2ZXJ5RGVsYXkgJiYgcHJvcHMuZGVsaXZlcnlEZWxheS50b1NlY29uZHMoKSxcbiAgICAgIG1heGltdW1NZXNzYWdlU2l6ZTogcHJvcHMubWF4TWVzc2FnZVNpemVCeXRlcyxcbiAgICAgIG1lc3NhZ2VSZXRlbnRpb25QZXJpb2Q6IHByb3BzLnJldGVudGlvblBlcmlvZCAmJiBwcm9wcy5yZXRlbnRpb25QZXJpb2QudG9TZWNvbmRzKCksXG4gICAgICByZWNlaXZlTWVzc2FnZVdhaXRUaW1lU2Vjb25kczogcHJvcHMucmVjZWl2ZU1lc3NhZ2VXYWl0VGltZSAmJiBwcm9wcy5yZWNlaXZlTWVzc2FnZVdhaXRUaW1lLnRvU2Vjb25kcygpLFxuICAgICAgdmlzaWJpbGl0eVRpbWVvdXQ6IHByb3BzLnZpc2liaWxpdHlUaW1lb3V0ICYmIHByb3BzLnZpc2liaWxpdHlUaW1lb3V0LnRvU2Vjb25kcygpLFxuICAgIH0pO1xuICAgIHF1ZXVlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5ID8/IFJlbW92YWxQb2xpY3kuREVTVFJPWSk7XG5cbiAgICB0aGlzLnF1ZXVlQXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShxdWV1ZS5hdHRyQXJuLCB7XG4gICAgICBzZXJ2aWNlOiAnc3FzJyxcbiAgICAgIHJlc291cmNlOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcbiAgICB0aGlzLnF1ZXVlTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHF1ZXVlLmF0dHJRdWV1ZU5hbWUpO1xuICAgIHRoaXMuZW5jcnlwdGlvbk1hc3RlcktleSA9IGVuY3J5cHRpb25NYXN0ZXJLZXk7XG4gICAgdGhpcy5xdWV1ZVVybCA9IHF1ZXVlLnJlZjtcbiAgICB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSA9IHByb3BzLmRlYWRMZXR0ZXJRdWV1ZTtcblxuICAgIGZ1bmN0aW9uIF9kZXRlcm1pbmVFbmNyeXB0aW9uUHJvcHModGhpczogUXVldWUpOiB7IGVuY3J5cHRpb25Qcm9wczogRW5jcnlwdGlvblByb3BzLCBlbmNyeXB0aW9uTWFzdGVyS2V5Pzoga21zLklLZXkgfSB7XG4gICAgICBsZXQgZW5jcnlwdGlvbiA9IHByb3BzLmVuY3J5cHRpb247XG5cbiAgICAgIGlmIChlbmNyeXB0aW9uID09PSBRdWV1ZUVuY3J5cHRpb24uU1FTX01BTkFHRUQgJiYgcHJvcHMuZW5jcnlwdGlvbk1hc3RlcktleSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCInZW5jcnlwdGlvbk1hc3RlcktleScgaXMgbm90IHN1cHBvcnRlZCBpZiBlbmNyeXB0aW9uIHR5cGUgJ1NRU19NQU5BR0VEJyBpcyB1c2VkXCIpO1xuICAgICAgfVxuXG4gICAgICBpZiAoZW5jcnlwdGlvbiAhPT0gUXVldWVFbmNyeXB0aW9uLktNUyAmJiBwcm9wcy5lbmNyeXB0aW9uTWFzdGVyS2V5KSB7XG4gICAgICAgIGVuY3J5cHRpb24gPSBRdWV1ZUVuY3J5cHRpb24uS01TOyAvLyBLTVMgaXMgaW1wbGllZCBieSBzcGVjaWZ5aW5nIGFuIGVuY3J5cHRpb24ga2V5XG4gICAgICB9XG5cbiAgICAgIGlmICghZW5jcnlwdGlvbikge1xuICAgICAgICByZXR1cm4geyBlbmNyeXB0aW9uUHJvcHM6IHt9IH07XG4gICAgICB9XG5cbiAgICAgIGlmIChlbmNyeXB0aW9uID09PSBRdWV1ZUVuY3J5cHRpb24uVU5FTkNSWVBURUQpIHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBlbmNyeXB0aW9uUHJvcHM6IHtcbiAgICAgICAgICAgIHNxc01hbmFnZWRTc2VFbmFibGVkOiBmYWxzZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgfVxuXG4gICAgICBpZiAoZW5jcnlwdGlvbiA9PT0gUXVldWVFbmNyeXB0aW9uLktNU19NQU5BR0VEKSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgZW5jcnlwdGlvblByb3BzOiB7XG4gICAgICAgICAgICBrbXNNYXN0ZXJLZXlJZDogJ2FsaWFzL2F3cy9zcXMnLFxuICAgICAgICAgICAga21zRGF0YUtleVJldXNlUGVyaW9kU2Vjb25kczogcHJvcHMuZGF0YUtleVJldXNlICYmIHByb3BzLmRhdGFLZXlSZXVzZS50b1NlY29uZHMoKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgfVxuXG4gICAgICBpZiAoZW5jcnlwdGlvbiA9PT0gUXVldWVFbmNyeXB0aW9uLktNUykge1xuICAgICAgICBjb25zdCBtYXN0ZXJLZXkgPSBwcm9wcy5lbmNyeXB0aW9uTWFzdGVyS2V5IHx8IG5ldyBrbXMuS2V5KHRoaXMsICdLZXknLCB7XG4gICAgICAgICAgZGVzY3JpcHRpb246IGBDcmVhdGVkIGJ5ICR7dGhpcy5ub2RlLnBhdGh9YCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBlbmNyeXB0aW9uTWFzdGVyS2V5OiBtYXN0ZXJLZXksXG4gICAgICAgICAgZW5jcnlwdGlvblByb3BzOiB7XG4gICAgICAgICAgICBrbXNNYXN0ZXJLZXlJZDogbWFzdGVyS2V5LmtleUFybixcbiAgICAgICAgICAgIGttc0RhdGFLZXlSZXVzZVBlcmlvZFNlY29uZHM6IHByb3BzLmRhdGFLZXlSZXVzZSAmJiBwcm9wcy5kYXRhS2V5UmV1c2UudG9TZWNvbmRzKCksXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgaWYgKGVuY3J5cHRpb24gPT09IFF1ZXVlRW5jcnlwdGlvbi5TUVNfTUFOQUdFRCkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGVuY3J5cHRpb25Qcm9wczoge1xuICAgICAgICAgICAgc3FzTWFuYWdlZFNzZUVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmV4cGVjdGVkICdlbmNyeXB0aW9uVHlwZSc6ICR7ZW5jcnlwdGlvbn1gKTtcbiAgICB9XG5cbiAgICAvLyBFbmZvcmNlIGVuY3J5cHRpb24gb2YgZGF0YSBpbiB0cmFuc2l0XG4gICAgaWYgKHByb3BzLmVuZm9yY2VTU0wpIHtcbiAgICAgIHRoaXMuZW5mb3JjZVNTTFN0YXRlbWVudCgpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBMb29rIGF0IHRoZSBwcm9wcywgc2VlIGlmIHRoZSBGSUZPIHByb3BzIGFncmVlLCBhbmQgcmV0dXJuIHRoZSBjb3JyZWN0IHN1YnNldCBvZiBwcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSBkZXRlcm1pbmVGaWZvUHJvcHMocHJvcHM6IFF1ZXVlUHJvcHMpOiBGaWZvUHJvcHMge1xuICAgIC8vIENoZWNrIGlmIGFueSBvZiB0aGUgc2lnbmFscyB0aGF0IHdlIGhhdmUgc2F5IHRoYXQgdGhpcyBpcyBhIEZJRk8gcXVldWUuXG4gICAgbGV0IGZpZm9RdWV1ZSA9IHByb3BzLmZpZm87XG4gICAgY29uc3QgcXVldWVOYW1lID0gcHJvcHMucXVldWVOYW1lO1xuICAgIGlmICh0eXBlb2YgZmlmb1F1ZXVlID09PSAndW5kZWZpbmVkJyAmJiBxdWV1ZU5hbWUgJiYgIVRva2VuLmlzVW5yZXNvbHZlZChxdWV1ZU5hbWUpICYmIHF1ZXVlTmFtZS5lbmRzV2l0aCgnLmZpZm8nKSkgeyBmaWZvUXVldWUgPSB0cnVlOyB9XG4gICAgaWYgKHR5cGVvZiBmaWZvUXVldWUgPT09ICd1bmRlZmluZWQnICYmIHByb3BzLmNvbnRlbnRCYXNlZERlZHVwbGljYXRpb24pIHsgZmlmb1F1ZXVlID0gdHJ1ZTsgfVxuICAgIGlmICh0eXBlb2YgZmlmb1F1ZXVlID09PSAndW5kZWZpbmVkJyAmJiBwcm9wcy5kZWR1cGxpY2F0aW9uU2NvcGUpIHsgZmlmb1F1ZXVlID0gdHJ1ZTsgfVxuICAgIGlmICh0eXBlb2YgZmlmb1F1ZXVlID09PSAndW5kZWZpbmVkJyAmJiBwcm9wcy5maWZvVGhyb3VnaHB1dExpbWl0KSB7IGZpZm9RdWV1ZSA9IHRydWU7IH1cblxuICAgIC8vIElmIHdlIGhhdmUgYSBuYW1lLCBzZWUgdGhhdCBpdCBhZ3JlZXMgd2l0aCB0aGUgRklGTyBzZXR0aW5nXG4gICAgaWYgKHR5cGVvZiBxdWV1ZU5hbWUgPT09ICdzdHJpbmcnKSB7XG4gICAgICBpZiAoZmlmb1F1ZXVlICYmICFxdWV1ZU5hbWUuZW5kc1dpdGgoJy5maWZvJykpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiRklGTyBxdWV1ZSBuYW1lcyBtdXN0IGVuZCBpbiAnLmZpZm8nXCIpO1xuICAgICAgfVxuICAgICAgaWYgKCFmaWZvUXVldWUgJiYgcXVldWVOYW1lLmVuZHNXaXRoKCcuZmlmbycpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcIk5vbi1GSUZPIHF1ZXVlIG5hbWUgbWF5IG5vdCBlbmQgaW4gJy5maWZvJ1wiKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAocHJvcHMuY29udGVudEJhc2VkRGVkdXBsaWNhdGlvbiAmJiAhZmlmb1F1ZXVlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NvbnRlbnQtYmFzZWQgZGVkdXBsaWNhdGlvbiBjYW4gb25seSBiZSBkZWZpbmVkIGZvciBGSUZPIHF1ZXVlcycpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5kZWR1cGxpY2F0aW9uU2NvcGUgJiYgIWZpZm9RdWV1ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdEZWR1cGxpY2F0aW9uIHNjb3BlIGNhbiBvbmx5IGJlIGRlZmluZWQgZm9yIEZJRk8gcXVldWVzJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmZpZm9UaHJvdWdocHV0TGltaXQgJiYgIWZpZm9RdWV1ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGSUZPIHRocm91Z2hwdXQgbGltaXQgY2FuIG9ubHkgYmUgZGVmaW5lZCBmb3IgRklGTyBxdWV1ZXMnKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgY29udGVudEJhc2VkRGVkdXBsaWNhdGlvbjogcHJvcHMuY29udGVudEJhc2VkRGVkdXBsaWNhdGlvbixcbiAgICAgIGRlZHVwbGljYXRpb25TY29wZTogcHJvcHMuZGVkdXBsaWNhdGlvblNjb3BlLFxuICAgICAgZmlmb1Rocm91Z2hwdXRMaW1pdDogcHJvcHMuZmlmb1Rocm91Z2hwdXRMaW1pdCxcbiAgICAgIGZpZm9RdWV1ZSxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gaWFtIHN0YXRlbWVudCB0byBlbmZvcmNlIGVuY3J5cHRpb24gb2YgZGF0YSBpbiB0cmFuc2l0LlxuICAgKi9cbiAgcHJpdmF0ZSBlbmZvcmNlU1NMU3RhdGVtZW50KCkge1xuICAgIGNvbnN0IHN0YXRlbWVudCA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnc3FzOionXSxcbiAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgQm9vbDogeyAnYXdzOlNlY3VyZVRyYW5zcG9ydCc6ICdmYWxzZScgfSxcbiAgICAgIH0sXG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuREVOWSxcbiAgICAgIHJlc291cmNlczogW3RoaXMucXVldWVBcm5dLFxuICAgICAgcHJpbmNpcGFsczogW25ldyBpYW0uQW55UHJpbmNpcGFsKCldLFxuICAgIH0pO1xuICAgIHRoaXMuYWRkVG9SZXNvdXJjZVBvbGljeShzdGF0ZW1lbnQpO1xuICB9XG59XG5cbmludGVyZmFjZSBGaWZvUHJvcHMge1xuICByZWFkb25seSBmaWZvUXVldWU/OiBib29sZWFuO1xuICByZWFkb25seSBjb250ZW50QmFzZWREZWR1cGxpY2F0aW9uPzogYm9vbGVhbjtcbiAgcmVhZG9ubHkgZGVkdXBsaWNhdGlvblNjb3BlPzogRGVkdXBsaWNhdGlvblNjb3BlO1xuICByZWFkb25seSBmaWZvVGhyb3VnaHB1dExpbWl0PzogRmlmb1Rocm91Z2hwdXRMaW1pdDtcbn1cblxuaW50ZXJmYWNlIEVuY3J5cHRpb25Qcm9wcyB7XG4gIHJlYWRvbmx5IGttc01hc3RlcktleUlkPzogc3RyaW5nO1xuICByZWFkb25seSBrbXNEYXRhS2V5UmV1c2VQZXJpb2RTZWNvbmRzPzogbnVtYmVyO1xuICByZWFkb25seSBzcXNNYW5hZ2VkU3NlRW5hYmxlZD86IGJvb2xlYW47XG59XG4iXX0=