"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CallApiGatewayEndpointBase = void 0;
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const base_types_1 = require("./base-types");
const task_utils_1 = require("../private/task-utils");
/**
 * Base CallApiGatewayEndpoint Task
 * @internal
 */
class CallApiGatewayEndpointBase extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.baseProps = props;
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        (0, task_utils_1.validatePatternSupported)(this.integrationPattern, CallApiGatewayEndpointBase.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(this.baseProps.headers)) {
                throw new Error('Task Token is required in `headers` for WAIT_FOR_TASK_TOKEN pattern. Use JsonPath.taskToken to set the token.');
            }
        }
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: (0, task_utils_1.integrationResourceArn)('apigateway', 'invoke', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ApiEndpoint: this.apiEndpoint,
                Method: this.baseProps.method,
                Headers: this.baseProps.headers?.value,
                Stage: this.stageName,
                Path: this.baseProps.apiPath,
                QueryParameters: this.baseProps.queryParameters?.value,
                RequestBody: this.baseProps.requestBody?.value,
                AuthType: this.baseProps.authType ? this.baseProps.authType : 'NO_AUTH',
            }),
        };
    }
    createPolicyStatements() {
        if (this.baseProps.authType === base_types_1.AuthType.NO_AUTH) {
            return [];
        }
        return [
            new iam.PolicyStatement({
                resources: [this.arnForExecuteApi],
                actions: ['execute-api:Invoke'],
            }),
        ];
    }
}
CallApiGatewayEndpointBase.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
exports.CallApiGatewayEndpointBase = CallApiGatewayEndpointBase;
//# sourceMappingURL=data:application/json;base64,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