"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EncryptionOption = exports.AthenaStartQueryExecution = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const cdk = require("../../../core");
const task_utils_1 = require("../private/task-utils");
/**
 * Start an Athena Query as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
 */
class AthenaStartQueryExecution extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_tasks_AthenaStartQueryExecutionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AthenaStartQueryExecution);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        (0, task_utils_1.validatePatternSupported)(this.integrationPattern, AthenaStartQueryExecution.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    createPolicyStatements() {
        const policyStatements = [
            new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        service: 'athena',
                        resource: 'datacatalog',
                        resourceName: this.props.queryExecutionContext?.catalogName ?? 'AwsDataCatalog',
                    }),
                    cdk.Stack.of(this).formatArn({
                        service: 'athena',
                        resource: 'workgroup',
                        resourceName: this.props.workGroup ?? 'primary',
                    }),
                ],
                actions: ['athena:getDataCatalog', 'athena:startQueryExecution', 'athena:getQueryExecution'],
            }),
        ];
        policyStatements.push(new iam.PolicyStatement({
            actions: ['s3:CreateBucket',
                's3:ListBucket',
                's3:GetBucketLocation',
                's3:GetObject'],
            resources: ['*'], // Need * permissions to create new output location https://docs.aws.amazon.com/athena/latest/ug/security-iam-athena.html
        }));
        policyStatements.push(new iam.PolicyStatement({
            actions: ['s3:AbortMultipartUpload',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:PutObject'],
            resources: [
                this.props.resultConfiguration?.outputLocation?.bucketName
                    ? cdk.Stack.of(this).formatArn({
                        // S3 Bucket names are globally unique in a partition,
                        // and so their ARNs have empty region and account components
                        region: '',
                        account: '',
                        service: 's3',
                        resource: this.props.resultConfiguration?.outputLocation?.bucketName,
                        resourceName: this.props.resultConfiguration?.outputLocation?.objectKey,
                    })
                    : '*',
            ],
        }));
        policyStatements.push(new iam.PolicyStatement({
            actions: ['lakeformation:GetDataAccess'],
            resources: ['*'], // State machines scoped to output location fail and * permissions are required as per documentation https://docs.aws.amazon.com/lake-formation/latest/dg/permissions-reference.html
        }));
        policyStatements.push(new iam.PolicyStatement({
            actions: ['glue:BatchCreatePartition',
                'glue:BatchDeletePartition',
                'glue:BatchDeleteTable',
                'glue:BatchGetPartition',
                'glue:CreateDatabase',
                'glue:CreatePartition',
                'glue:CreateTable',
                'glue:DeleteDatabase',
                'glue:DeletePartition',
                'glue:DeleteTable',
                'glue:GetDatabase',
                'glue:GetDatabases',
                'glue:GetPartition',
                'glue:GetPartitions',
                'glue:GetTable',
                'glue:GetTables',
                'glue:UpdateDatabase',
                'glue:UpdatePartition',
                'glue:UpdateTable'],
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'catalog',
                }),
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'database',
                    resourceName: this.props.queryExecutionContext?.databaseName ?? 'default',
                }),
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'table',
                    resourceName: (this.props.queryExecutionContext?.databaseName ?? 'default') + '/*', // grant access to all tables in the specified or default database to prevent cross database access https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html
                }),
                cdk.Stack.of(this).formatArn({
                    service: 'glue',
                    resource: 'userDefinedFunction',
                    resourceName: (this.props.queryExecutionContext?.databaseName ?? 'default') + '/*', // grant access to get all user defined functions for the particular database in the request or the default database https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html
                }),
            ],
        }));
        return policyStatements;
    }
    renderEncryption() {
        const encryptionConfiguration = this.props.resultConfiguration?.encryptionConfiguration !== undefined
            ? {
                EncryptionOption: this.props.resultConfiguration.encryptionConfiguration.encryptionOption,
                KmsKey: this.props.resultConfiguration.encryptionConfiguration.encryptionKey,
            }
            : undefined;
        return encryptionConfiguration;
    }
    /**
     * Provides the Athena start query execution service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: (0, task_utils_1.integrationResourceArn)('athena', 'startQueryExecution', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                QueryString: this.props.queryString,
                ClientRequestToken: this.props.clientRequestToken,
                QueryExecutionContext: (this.props.queryExecutionContext?.catalogName || this.props.queryExecutionContext?.databaseName) ? {
                    Catalog: this.props.queryExecutionContext?.catalogName,
                    Database: this.props.queryExecutionContext?.databaseName,
                } : undefined,
                ResultConfiguration: {
                    EncryptionConfiguration: this.renderEncryption(),
                    OutputLocation: this.props.resultConfiguration?.outputLocation ? `s3://${this.props.resultConfiguration.outputLocation.bucketName}/${this.props.resultConfiguration.outputLocation.objectKey}/` : undefined,
                },
                WorkGroup: this.props?.workGroup,
            }),
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
AthenaStartQueryExecution[_a] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecution", version: "2.74.0" };
AthenaStartQueryExecution.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
exports.AthenaStartQueryExecution = AthenaStartQueryExecution;
/**
 * Encryption Options of the S3 bucket
 *
 * @see https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html#athena-Type-EncryptionConfiguration-EncryptionOption
 */
var EncryptionOption;
(function (EncryptionOption) {
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    EncryptionOption["S3_MANAGED"] = "SSE_S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    EncryptionOption["KMS"] = "SSE_KMS";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    EncryptionOption["CLIENT_SIDE_KMS"] = "CSE_KMS";
})(EncryptionOption = exports.EncryptionOption || (exports.EncryptionOption = {}));
//# sourceMappingURL=data:application/json;base64,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