"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrAddStep = exports.ActionOnFailure = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const core_1 = require("../../../core");
const task_utils_1 = require("../private/task-utils");
/**
 * The action to take when the cluster step fails.
 * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
 *
 * Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
 *
 * @default CONTINUE
 *
 */
var ActionOnFailure;
(function (ActionOnFailure) {
    /**
     * Terminate the Cluster on Step Failure
     */
    ActionOnFailure["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    /**
     * Cancel Step execution and enter WAITING state
     */
    ActionOnFailure["CANCEL_AND_WAIT"] = "CANCEL_AND_WAIT";
    /**
     * Continue to the next Step
     */
    ActionOnFailure["CONTINUE"] = "CONTINUE";
})(ActionOnFailure = exports.ActionOnFailure || (exports.ActionOnFailure = {}));
/**
 * A Step Functions Task to add a Step to an EMR Cluster
 *
 * The StepConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the StepId
 *
 */
class EmrAddStep extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_tasks_EmrAddStepProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EmrAddStep);
            }
            throw error;
        }
        this.actionOnFailure = props.actionOnFailure ?? ActionOnFailure.CONTINUE;
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.RUN_JOB;
        (0, task_utils_1.validatePatternSupported)(this.integrationPattern, EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: (0, task_utils_1.integrationResourceArn)('elasticmapreduce', 'addStep', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
                Step: {
                    Name: this.props.name,
                    ActionOnFailure: this.actionOnFailure.valueOf(),
                    HadoopJarStep: {
                        Jar: this.props.jar,
                        MainClass: this.props.mainClass,
                        Args: this.props.args,
                        Properties: (this.props.properties === undefined) ?
                            undefined :
                            Object.entries(this.props.properties).map(kv => ({
                                Key: kv[0],
                                Value: kv[1],
                            })),
                    },
                },
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call AddStep.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:AddJobFlowSteps',
                    'elasticmapreduce:DescribeStep',
                    'elasticmapreduce:CancelSteps',
                ],
                resources: [
                    stack.formatArn({
                        service: 'elasticmapreduce',
                        resource: 'cluster',
                        resourceName: '*',
                    }),
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRAddJobFlowStepsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
_a = JSII_RTTI_SYMBOL_1;
EmrAddStep[_a] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.EmrAddStep", version: "2.74.0" };
EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
exports.EmrAddStep = EmrAddStep;
//# sourceMappingURL=data:application/json;base64,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